/**
 * @module quick-edit/form/configurable/create-issue
 */
define('quick-edit/form/configurable/create-issue', [
    'quick-edit/form/abstract/configurable',
    'quick-edit/form/helper/helper',
    'quick-edit/templates/issue',
    'quick-edit/util/jira',
    'jira/util/users/logged-in-user',
    'wrm/context-path',
    'jquery'
], function(
    AbstractConfigurableForm,
    CreateIssueHelper,
    IssueTemplates,
    JiraUtil,
    User,
    contextPath,
    jQuery
){
    'use strict';

    /**
     * A View that renders a configurable version of the create issue form. This version allows you to reorder fields and
     * add and remove fields.
     *
     * @class ConfigurableCreateIssueForm
     * @extends AbstractConfigurableForm
     */
    return AbstractConfigurableForm.extend({
        /**
         * @constructor
         * @param options
         * ... {FormModel} model that gets fields and sets user preferences
         * ... {string|number} issue id
         * ... {Array<String>} globalEventNamespaces - Events will be triggered on these namespaces
         * e.g CreateForm.switchedToConfigurableForm. CreateForm being the specified global namespace.
         */
        init: function (options) {
            this.model = options.model;
            this.helper = new CreateIssueHelper(this);
            this.globalEventNamespaces = options.globalEventNamespaces || [];
            this.issueId = options.issueId;
            this.title = options.title;
            this._serialization = {};

            // Form values will be serialized and POSTed to this url
            this.action = contextPath() + "/secure/QuickCreateIssue.jspa?decorator=none";
            // The container element. All html will be injected into this guy.
            this.$element = jQuery("<div />").addClass("qf-form qf-configurable-form");
        },

        /**
         * Delegates submit success to Create Issue helper
         * @param {issueKey: <String>} data
         */
        handleSubmitSuccess: function (data) {
            return this.helper.handleSubmitSuccess(data);
        },

        /**
         * Renders create issue specific chrome and furniture before delegating to super class for the rendering of fields
         *
         */
        _render: function () {
            var deferred = jQuery.Deferred(),
                instance = this;

            instance.model.getIssueSetupFields().done(function (issueSetupFields) {
                var html = IssueTemplates.createIssueForm({
                    issueSetupFields: issueSetupFields,
                    atlToken: instance.model.getAtlToken(),
                    formToken: instance.model.getFormToken(),
                    isConfigurable: true,
                    title: instance.title,
                    parentIssueId: instance.model.getParentIssueId(),
                    multipleMode: instance.model.isInMultipleMode(),
                    showFieldConfigurationToolBar: !User.isAnonymous(),
                    modifierKey: JiraUtil.getModifierKey()
                });

                // add form chrome to container element
                instance.$element.html(html);

                instance.helper.decorate();

                // render fields
                instance.renderFormContents().done(function (el, scripts) {
                    instance.helper.decorateFields();

                    deferred.resolveWith(instance, [instance.$element, scripts]);
                });

                instance.bind("QuickForm.fieldAdded", function() {
                    instance.helper.decorateFields();
                });
            });

            return deferred.promise();
        }
    });
});

/**
 * @deprecated JIRA.Forms.ConfigurableCreateIssueForm
 */
AJS.namespace('JIRA.Forms.ConfigurableCreateIssueForm', null, require('quick-edit/form/configurable/create-issue'));