AJS.test.require("com.atlassian.jira.jira-projects-plugin:browseproject", function() {

    module("JIRA.Projects.TabModel", {
        setup: function() {
            this.sandbox = sinon.sandbox.create();
            this.server = this.sandbox.useFakeServer();
        },
        teardown: function() {
            this.sandbox.restore();
        }
    });

    test("fetches when new tab requested", function () {
        var model = new JIRA.Projects.TabModel();
        sinon.stub(model, 'fetch');
        model.newTabRequested('/our/url');

        equal(model.fetch.callCount, 1);
    });

    test("points to url of undecorated target tab", function () {
        var model = new JIRA.Projects.TabModel();
        model.newTabRequested('/bob?selectedTab=foo');
        equal(model.url(), '/bob?selectedTab=foo&decorator=none&contentOnly=true');
    });

    test("always requests data in plain-text", function () {
        var model = new JIRA.Projects.TabModel();
        sinon.stub(model, 'fetch');
        model.newTabRequested('/bob?selectedTab=foo');
        equal(typeof model.fetch.getCall(0).args[0], 'object');
        equal(model.fetch.getCall(0).args[0].dataType, 'text', "should always request text");
    });

    test("sets html attribute on fetch", function () {
        var model = new JIRA.Projects.TabModel();
        model.newTabRequested();
        ok(!model.has('html'), "won't be set yet because we haven't attempted to fetch");
        equal(this.server.requests.length, 1);

        this.server.requests[0].respond();

        ok(model.has('html'), "should have gotten set after fetch");
    });

    test("html attribute empty when no data given", function () {
        var model = new JIRA.Projects.TabModel();
        model.newTabRequested();

        this.server.requests[0].respond();

        equal(model.get('html'), '');
    });

    test("html attribute set to response", function () {
        var html = '<div class="foo">yo</div>';
        var model = new JIRA.Projects.TabModel();
        model.newTabRequested();

        this.server.requests[0].respond(200, { "Content-Type": "text/html" }, html);

        equal(model.get('html'), html);
    });

    test("triggers switch event on successful fetch", function () {
        var html = '<div class="foo">yo</div>';
        var model = new JIRA.Projects.TabModel();
        var spy = sinon.spy();
        var url = "/bob";
        model.on('switch', spy);
        model.newTabRequested(url);

        this.server.requests[0].respond(200, { "Content-Type": "text/html" }, html);

        equal(spy.callCount, 1);
        ok(spy.calledWithExactly(url), "should be passed the url that was retrieved");
    });

    test("triggers error event if HTTP headers indicate user is not logged in", function () {
        var html = '<div class="foo">yo</div>';
        var model = new JIRA.Projects.TabModel();
        var switchSpy = sinon.spy(),
            errorSpy = sinon.spy(),
            url = '/bob';
        model.on('switch', switchSpy);
        model.on('switch-error', errorSpy);
        model.newTabRequested(url);

        this.server.requests[0].respond(200, { "Content-Type": "text/html", 'X-Atlassian-Dialog-Control': 'permissionviolation' }, html);

        equal(switchSpy.callCount, 0, "Permission violation should not trigger a tab switched event");
        ok(errorSpy.calledOnce, "Permission violation should an error event");
        ok(errorSpy.calledWithExactly({loggedOut : true}, url),
                "Permission violation event should indicate logged out status with return url");

        var reason = errorSpy.getCall(0).args[0];
        equal(typeof reason, 'object');
        ok(reason.loggedOut, "should indicate user is logged out");
    });

    test("triggers error event on non-200 response", function () {
        var html = '<div class="foo">yo</div>';
        var model = new JIRA.Projects.TabModel();
        var switchSpy = sinon.spy(),
                errorSpy = sinon.spy(),
                url = '/bob';
        model.on('switch', switchSpy);
        model.on('switch-error', errorSpy);
        model.newTabRequested(url);

        this.server.requests[0].respond(404, { "Content-Type": "text/html" }, html);

        ok(!switchSpy.called, "should not switch tabs if there is an error");
        ok(errorSpy.calledOnce, "should raise error event on non-200 response");
        equal(errorSpy.getCall(0).args[1], url, "should be passed the url that was retrieved");
    });

});
