(function(){
    "use strict";

    AJS.namespace("JIRA.Projects.Subnavigator");

    /**
     * Thin wrapper around AUI's dropdown2 that renders a list of items to choose from.
     * The component is initialized like this:
     *
     * var items = JIRA.Projects.Subnavigator({
     *     id: someId,
     *     triggerPlaceholder: cssSelector,
     *     contentPlaceholder: cssSelector,
     *     itemGroups: [...],
     *     selectedItem: itemId
     * });
     *
     * On the initialization object:
     * @param {string} id An identifier for the component, just in case there are more than one on the same page
     * @param {string} triggerPlaceholder A css selector or DOM element where the dropdown2 trigger will be placed
     * @param {string} contentPlaceholder A css selector or DOM element where the dropdown2 content (aka the options) will be placed
     * @param {string} titlePlaceholder A css selector or DOM element where the dropdown2 title will be placed (optional).
     *                                  If not provided, the component will not handle the title.
     * @param {Object[]} itemGroups An array with the list of items to choose from
     * @param {string} selectedItem The identifier of the item that will appear as selected
     * @param {string} changeViewText Text used for the trigger (optinal). If not provided, it will show the selected item.
     * @param {boolean} hideSelectedItem Whether we should hide the selected item from the dropdown2 (defaults to true)
     *
     * The items passed to itemGroups are an object as follows:
     *
     * {
     *     id: "some-id", // The identifier of the item
     *     label: "A label", // The label used to display the item
     *     description: "A description", // A description that will be shown as a tooltip on mouseover [Optional]
     *     link: "example.com", // The link to go to if the item is selected [Optional]
     * }
     *
     * The array of itemGroups passed to the component should contain an array for each section of items that
     * needs to be rendered. For example, if we pass in
     * [
     *      [
     *          { id: "1", label: "First"},
     *          { id: "2", label: "Second"}
     *      ],
     *      [
     *          { id: "3", label: "Third"}
     *      ]
     * ]
     *
     * the component will render the options in two option sections (one containing First and Second and another one containing Third).
     *
     * For convenience, if you only want to render one section, you can pass the items a simple array, like so:
     *
     * [
     *    { id: "1", label: "First"},
     *    { id: "2", label: "Second"}
     * ]
     *
     * This component triggers a "itemSelected" event every time an item is selected.
     * The event object passed on to listeners to this event will contain the item selected.
     * Preventing the default action for that event is also possible:
     *
     * items.on("itemSelected", function(e) {
     *     e.preventDefault();
     *     console.log(e.item);
     * });
     *
     */
    JIRA.Projects.Subnavigator = JIRA.Projects.Libs.Marionette.Controller.extend({
        _shouldShow: function() {
            return !this.model.isEmpty();
        },

        _shouldShowTriggerView: function() {
            if (this.options.changeViewText) {
                var numberOfItems = this.model.getAllItems().length;
                return numberOfItems > 1;
            } else {
                return true;
            }
        },

        _sanitizeItems: function() {
            var items = this.options.itemGroups;

            if (items && items[0]) {
                if (!_.isArray(items[0])) {
                    return [items];
                }
            }

            return items;
        },

        _buildModel: function() {
            this.model = new JIRA.Projects.Subnavigator.Entities.Items({
                itemGroups: this._sanitizeItems()
            });
            this.model.selectItem(this.options.selectedItem);
            this.listenTo(this.model, "change:selectedItem", this.show);
        },

        _buildTitleView: function() {
            this.titleView = new JIRA.Projects.Subnavigator.Views.Title({
                model: this.model
            });
            this.titleView.setElement(AJS.$(this.options.titlePlaceholder));
        },

        _buildTriggerView: function() {
            if (this.options.changeViewText) {
                this.triggerView = new JIRA.Projects.Subnavigator.Views.StaticTrigger({
                    model: this.model,
                    id: this.options.id,
                    label: this.options.changeViewText
                });
            } else {
                this.triggerView = new JIRA.Projects.Subnavigator.Views.Trigger({
                    model: this.model,
                    id: this.options.id
                });
            }
            this.triggerView.setElement(AJS.$(this.options.triggerPlaceholder));
        },

        _buildContentView: function() {
            this.contentView = new JIRA.Projects.Subnavigator.Views.Content({
                model: this.model,
                id: this.options.id,
                hideSelectedItem: this.options.hideSelectedItem
            });
            this.contentView.setElement(AJS.$(this.options.contentPlaceholder));

            this.listenTo(this.contentView, 'click', function(itemId) {
                this.model.selectItem(itemId);
                var item = this.model.get("selectedItem");
                var event = this.triggerPreventable("itemSelected", {
                    item: item
                });

                if (event.isPrevented) {
                    return;
                }
                if (item.link) {
                    require('jira/util/browser').reloadViaWindowLocation(item.link);
                }
            }.bind(this));
        },

        initialize: function() {
            this._buildModel();
            if (this._shouldShow()) {
                this._buildTriggerView();
                this._buildContentView();
                if (this.options.titlePlaceholder) {
                    this._buildTitleView();
                }
            }
        },

        show: function() {
            if (this._shouldShow()) {
                if (this._shouldShowTriggerView()) {
                    this.triggerView.render();
                    this.contentView.render();
                }
                if (this.titleView) {
                    this.titleView.render();
                }
            }
        },

        selectItem: function(itemId) {
            if (this._shouldShow()) {
                this.model.selectItem(itemId);
            }
        }
    });

}());