AJS.test.require("com.atlassian.jira.jira-projects-plugin:subnavigator-test", function() {
    "use strict";

    var ITEM_1 = {id: "item1", label: "Item 1", link: "http://www.atlassian.com"};
    var ITEM_2 = {id: "item2", label: "Item 2", link: "http://jira.atlassian.com"};
    var ITEMS = [[ITEM_1], [ITEM_2]];

    function listener() {
        return _.extend({}, Backbone.Events);
    }

    module("JIRA.Projects.Subnavigator.Subnavigator", {
        setup: function() {
            this.triggerPlaceholder = AJS.$("<div/>");
            this.contentPlaceholder = AJS.$("<div/>");
            this.titlePlaceholder = AJS.$("<div/>");

            this.browserUtilsStub = {
                reloadLink: undefined,
                reloadViaWindowLocation: function(link) {
                    this.reloadLink = link;
                }
            };

            this.requireStub = sinon.stub(window, "require");
            this.requireStub.withArgs("jira/util/browser").returns(this.browserUtilsStub);

            this.clock = sinon.useFakeTimers();
        },

        teardown: function() {
            this.requireStub.restore();
            this.clock.restore();
        },

        oldControllerWithItems: function(items) {
            return new JIRA.Projects.Subnavigator({
                triggerPlaceholder: this.triggerPlaceholder,
                contentPlaceholder: this.contentPlaceholder,
                itemGroups: items
            });
        },

        controllerWithItems: function(items) {
            return new JIRA.Projects.Subnavigator({
                triggerPlaceholder: this.triggerPlaceholder,
                contentPlaceholder: this.contentPlaceholder,
                itemGroups: items,
                titlePlaceholder: this.titlePlaceholder,
                hideSelectedItem: true,
                changeViewText: ""
            });
        },

        selectItem: function(item) {
            this.contentPlaceholder.find("a[data-item-id='" + item.id + "']").trigger("aui-dropdown2-item-check");
            this.contentPlaceholder.find(".subnav-opts").trigger("aui-dropdown2-hide");
        }
    });

    test("Select item selects it on the model", function() {
        var controller = this.controllerWithItems(ITEMS);

        controller.selectItem(ITEM_2.id);

        deepEqual(controller.model.get("selectedItem"), ITEM_2);
    });

    test("Select item does nothing if there are no items", function() {
        var controller = this.controllerWithItems([]);

        controller.selectItem(ITEM_2.id);

        equal(controller.model.get("selectedItem"), undefined);
    });

    test("Show old Subnav does not render the trigger view if there are no items", function() {
        var controller = this.oldControllerWithItems([]);

        controller.show();

        ok(this.triggerPlaceholder.html().length === 0);
    });

    test("Show does not render the trigger view if there are no items", function() {
        var controller = this.controllerWithItems([]);

        controller.show();

        ok(this.triggerPlaceholder.html().length === 0);
    });

    test("Show old Subnav renders the trigger view if there are items", function() {
        var controller = this.oldControllerWithItems(ITEMS);

        controller.show();

        ok(this.triggerPlaceholder.html().length > 0);
    });

    test("Show renders the trigger view if there are items", function() {
        var controller = this.controllerWithItems(ITEMS);

        controller.show();

        ok(this.triggerPlaceholder.html().length > 0);
    });

    test("Show does not render the content view if there are no items", function() {
        var controller = this.controllerWithItems([]);

        controller.show();

        ok(this.contentPlaceholder.html().length === 0);
    });

    test("Show renders the content view if there are items", function() {
        var controller = this.controllerWithItems(ITEMS);

        controller.show();

        ok(this.contentPlaceholder.html().length > 0);
    });

    test("Triggers an 'itemSelected' event when an item is selected", function() {
        var controller = this.controllerWithItems(ITEMS);
        controller.show();

        var spy = this.spy();
        listener().listenTo(controller, "itemSelected", spy);

        this.selectItem(ITEM_2);

        this.clock.tick(100);

        sinon.assert.calledOnce(spy);
        deepEqual(spy.firstCall.args[0].item, ITEM_2);
    });

    test("The 'itemSelected' event is preventable", function() {
        var controller = this.controllerWithItems(ITEMS);
        controller.show();

        var isPreventable = false;
        listener().listenTo(controller, "itemSelected", function(e) {
            isPreventable = typeof e.preventDefault === 'function';
        });

        this.selectItem(ITEM_2);

        this.clock.tick(100);

        ok(isPreventable);
    });

    test("Follows the selected item link when a new item is selected", function() {
        var controller = this.controllerWithItems(ITEMS);
        controller.show();

        this.selectItem(ITEM_2);

        this.clock.tick(100);

        equal(this.browserUtilsStub.reloadLink, ITEM_2.link);
    });

    test("Does not follow the selected item link when a new item is selected if the 'itemSelected' event is prevented", function() {
        var controller = this.controllerWithItems(ITEMS);
        controller.show();

        listener().listenTo(controller, "itemSelected", function(e) {
            e.preventDefault();
        });

        this.selectItem(ITEM_2);

        this.clock.tick(100);

        equal(this.browserUtilsStub.reloadLink, undefined);
    });

    test("Transforms a single list of items into a list of item groups with one group", function() {
        var controller = this.controllerWithItems([ITEM_1, ITEM_2]);

        deepEqual(controller.model.get("itemGroups"), [[ITEM_1, ITEM_2]]);
    });
});
