(function(){
    "use strict";

    AJS.namespace("JIRA.Projects.PinnableNavigator.Entities.Items");

    JIRA.Projects.PinnableNavigator.Entities.Items = Backbone.Model.extend({
        defaults: {
            itemGroups: [],
            selectedItem: null
        },

        initialize: function() {
            if(this.attributes.selectedItem) {
                this.selectItem(this.attributes.selectedItem);
            }
        },

        hasDisplayableGroups: function (itemGroups) {
            itemGroups = itemGroups || this.get("itemGroups");
            return _.any(itemGroups, function (group) {
                if (group.type === "buttons") {
                    return false;
                } else {
                    return group.emptyText || group.items && group.items.length;
                }
            });
        },

        isManageable: function () {
            return !!_.flatten(_.pluck(this.getManageableItemGroups(), "items")).length;
        },

        getManageableItemGroups: function () {
            return _.filter(this.get("itemGroups"), function (group) {
                return group.allowDelete || group.allowReorder;
            });
        },

        isEmpty: function() {
            return _.isEmpty(this.getAllItems());
        },

        selectItem: function(itemId) {
            var items = this.getAllItems();
            this.set("selectedItem",  this.getItemById(itemId) || _.first(items));
        },

        getItemById: function (itemId) {
            return _.findWhere(this.getAllItems(), {id: itemId});
        },

        getAllItems: function() {
            return _.flatten(_.map(this.get("itemGroups"), function (group) {
                return group.items;
            }));
        },

        getAllItemsExceptButtons: function() {
            return _.compact(_.flatten(_.map(this.get("itemGroups"), function (group) {
                return group.type === "buttons"? null: group.items;
            })));
        },

        getAllItemsWithName: function(searchedValue) {
            // Clone new object, so we can save the orginal and do filter on clone one
            var itemGroups = AJS.$.extend(true, {}, this.get("itemGroups"));

            return _.compact(_.map(itemGroups, function (group) {
                if (group.type !== "buttons") {
                    group.items = _.filter(group.items, function (item) {
                        return item.label.toUpperCase().indexOf(searchedValue.toUpperCase()) > -1;
                    });
                }
                if (!group.items.length) {
                    return null;
                }
                return group;
            }));
        },

        getGroup: function (id) {
            return _.findWhere(this.get("itemGroups"), {id: id});
        },

        applyModifications: function (modifications) {
            _.each(modifications, function (group) {
                var currentGroup = this.getGroup(group.id);
                currentGroup.items = _.map(group.order, function (id) {
                    return this.getItemById(id);
                }, this);
            }, this);
        }
    });

    JIRA.Projects.PinnableNavigator.Entities.Items.create = function (props, options) {
        var model = new JIRA.Projects.PinnableNavigator.Entities.Items(props);
        if (options.save) {
            model.save = function (modifications) {
                return options.save(modifications).done(function () {
                    model.applyModifications(modifications);
                    JIRA.trace("jira.projects.items.updated");
                });
            };
        }
        return model;
    };

}());