(function($){
    "use strict";
    /**
     * This will create a dropdown component which can be pinned somewhere
     * The component is initialized like this:
     *
     * var items = JIRA.Projects.PinnableNavigator({
     *     id: someId,
     *     triggerView: view object,
     *     unpinnedView: view object,
     *     pinnedView: view object,
     *     pinnedHolderId: cssSelector,
     *     isPinned: boolean,
     *     manageDialogControl: manageDialogControl,
     *     model: model,
     *     triggerHolderId: myOptions.triggerHolderId,
     *     unpinnedHolderId: myOptions.unpinnedHolderId
     * });
     *
     * On the initialization object:
     * @param {string} id An identifier for the component, just in case there are more than one on the same page
     * @param {object} triggerView A view object which content the template of button to open the dropdown.
     * @param {object} unpinnedView A view object which content the template of dropdown.
     * @param {object} pinnedView A view object which content the template of pinned panel.   
     * @param {string} pinnedHolderId A ID selector where the dropdown2 will be pinned
     * @param {boolean} isPinned Pin or unpin the navigator at the first time
     * @param {object} manageDialogControl control of manage view
     * @param {object} model model of the views (pinned, unpinned, trigger and manage view)
     * @param {string} triggerHolderId An identifier for the trigger view
     * @param {string} unpinnedHolderId An identifier for the unpinned view
     *
     * The items passed to itemGroups are an object as follows:
     *
     * {
     *     id: "some-id", // The identifier of the item
     *     label: "A label", // The label used to display the item
     *     description: "A description", // A description that will be shown as a tooltip on mouseover [Optional]
     *     link: "example.com", // The link to go to if the item is selected [Optional]
     *     count: number, // The number need to show [Optional]
     * }
     *
     * The array of itemGroups passed to the component should contain an array for each section of items that
     * needs to be rendered. For example, if we pass in
     * [
     *      [
     *          { id: "1", label: "First"},
     *          { id: "2", label: "Second", count: 5}
     *      ],
     *      [
     *          { id: "3", label: "Third"}
     *      ]
     * ]
     *
     * the component will render the options in two option sections (one containing First and Second and another one containing Third).
     *
     * For convenience, if you only want to render one section, you can pass the items a simple array, like so:
     *
     * [
     *    { id: "1", label: "First"},
     *    { id: "2", label: "Second"}
     * ]
     *
     * This component triggers a "itemSelected" event every time an item is selected.
     * The event object passed on to listeners to this event will contain the item selected.
     * Preventing the default action for that event is also possible:
     *
     * items.on("itemSelected", function(e) {
     *     e.preventDefault();
     *     console.log(e.item);
     * });
     *
     */
    JIRA.Projects.PinnableNavigator = JIRA.Projects.Libs.Marionette.Controller.extend({
        _renderPinned: function() {
            if (this.options.pinnedView.isAttachedToDom() && !this.options.searchView.isSearchable() && this.options.searchView.isExisted()) {
                this.options.pinnedView.$el.show();
                this.options.pinnedView.renderList();
            } else {
                this.saveState('pin');
                this.options.isPinned = true;
                this.options.triggerView.$el.empty().hide();
                this.options.unpinnedView.$el.empty();

                $('#'+this.options.pinnedHolderId).prepend(this.options.pinnedView.render().$el);
                this.options.pinnedView.renderSearch();

                this.options.pinnedView.putListintoView();
                this.trigger("navigatorPinned");
            }
        },

        _renderUnpinned: function() {
            if (AJS.$("#" + this.options.triggerHolderId).length) {
                // We want to expose the show method for developers as a way they can re-evaluate the container elements
                // for the case where the original dom elements have been removed and a re-render is needed. To do so we
                // re-evaluate the elements for each method call.
                this.options.triggerView.setElement(AJS.$("#" + this.options.triggerHolderId));
                this.options.unpinnedView.setElement(AJS.$("#" + this.options.unpinnedHolderId));
            }
            this.saveState('unpin');
            this.options.isPinned = false;
            this.options.triggerView.render().$el.show();
            this.options.unpinnedView.render();
            this.options.pinnedView.unpin();
            this.options.unpinnedView.renderSearch();
            this.options.unpinnedView.putListintoView();
            this.trigger("navigatorUnPinned");
        },

        _openManageDialog: function () {
            this.options.manageDialogControl.show();
        },
        /**
         * Update the model of Pin, Trigger, Unpin view.
         * @param {Array} itemGroups - The group of items to render
         * @param {Array} selectedItem - The id of the time to select (optional)
         */
        update: function (itemGroups, selectedItem) {
            this.options.model.set("itemGroups", itemGroups);
            if (selectedItem) {
                this.options.model.selectItem(selectedItem);
            }
            this.show();
        },
        /**
         * Set selected item of model
         * @param {int} id
         */
        setSelectedItem: function(id){
            this.options.model.selectItem(id);
        },
        saveState: function (state) {
            AJS.$.ajax({
                url: this.url + '/' + state,
                type: "PUT",
                contentType:"application/json"
            }).error(function () {
                // Do something to let user know there is an error here
            }).done(function () {
                JIRA.trace("jira.projects.pinnablenavigator.state.updated");
            });
        },

        initialize: function() {

            this.url = AJS.contextPath() + '/rest/projects/1.0/subnav/' + this.options.id;

            this.listenTo(this.options.unpinnedView, {
                "navigatorPinned": this._renderPinned,
                "manageDialogOpened": this._openManageDialog,
                "itemSelected": function(e) {
                    this.trigger("itemSelected", e);
                },
                "aui-dropdown2-show": function() {
                    this.trigger("aui-dropdown2-show");
                },
                "aui-dropdown2-hide": function() {
                    this.trigger("aui-dropdown2-hide");
                }
            });
            this.listenTo(this.options.pinnedView, {
                "navigatorUnPinned": this._renderUnpinned,
                "itemSelected": function (e) {
                    this.trigger("itemSelected", e);
                }
            });

            if (this.options.manageDialogControl) {
                this.listenTo(this.options.manageDialogControl, {
                    "itemsUpdated": this.update
                });
            }
        },

        show: function () {
            if (this.options.isPinned) {
                this._renderPinned();
            } else {
                this._renderUnpinned();
            }
        },
        hide: function () {
            // hide all view elements
            this.options.triggerView.$el.hide();
            this.options.unpinnedView.$el.hide();
            this.options.pinnedView.$el.hide();
        }
    });
}(AJS.$));

/**
 * This creates a Pinnable Navigator by creating and wiring together all the Views & Models required.
 *
 * A pinnable Navigator has to modes:
 *
 * Pinned - this means that the navigator and all it's contents are visible all the time. The contents of this
 * view is usually more variable to that of the Unpinned view as there is more screen real estate to play with, 
 * as such we allow the developer to specify a custom view to render the content. The pinned view is 
 * prepended to the element with the id options.pinnedHolderId. It is rendered by the Backbone.View options.pinnedView.
 *
 * UnPinned - this means the navigator contents are in a dropdown (AUI dropdown2). Unlike the pinned view, the
 * renderer is fixed as dropdown contents are more predicatable. The trigger for this dropdown is rendered to
 * options.unpinnedHolderId.
 *
 * @param {Object} options
 * ... {String) id - Unique id for this navigator, used for persistence of state & identifiying dom nodes.
 * ... {Backbone.View} pinnedView - the renderer for the pinned View
 * ... {Array[Array[Object]]} itemGroups - The groups of items to render
 * ... {String} selectedItem - The id of the item to be selected (one of the items in the itemGroups)
 * ... {String} pinnedHolderId - The id of the element where the pinned view is rendered
 * ... {String} unpinnedHolderId - The id of the element where the unpinned view is rendered
 * ... {String} triggerHolderId - The id of the element where the trigger view is rendered
 * ... {String} title - The title of manage component
 * ... {String} manageText - The title of manage button and manage Diaglog
 * ... {Function} save - The function will be executed when user press save button
 * ... {String} emptyText - The message will be shown when there is no item in navigator
 *
 * @returns {JIRA.Projects.PinnableNavigator}
 */

JIRA.Projects.PinnableNavigator.create = function (options) {

    var myOptions = {};
    var defaultOptions = {
        title: "",
        manageText: "",
        emptyText: AJS.I18n.getText('project.subnavigator.manage.emptyText')
    };

    _.defaults(myOptions, options, defaultOptions);

    var model = new JIRA.Projects.PinnableNavigator.Entities.Items.create({
        itemGroups: myOptions.itemGroups,
        selectedItem: myOptions.selectedItem
    }, myOptions);

    var searchView = new JIRA.Projects.PinnableNavigator.Views.Search.Search({
        searchPlaceHolder: myOptions.searchPlaceHolder,
        el: myOptions.searchHolder || ".js-subnav-search-block",
        model: model
    });

    var triggerView = new JIRA.Projects.PinnableNavigator.Views.Trigger({
        model: model,
        el: AJS.$("#" + myOptions.triggerHolderId),
        id: myOptions.id,
        label: options.title
    });

    var unpinnedView = new JIRA.Projects.PinnableNavigator.Views.NavigatorView.UnpinnedView({
        model: model,
        el: AJS.$("#" + myOptions.unpinnedHolderId),
        id: myOptions.id,
        title: myOptions.title,
        manageText: myOptions.manageText,
        manageLocation: myOptions.manageLocation,
        emptyText: myOptions.emptyText,
        searchView: searchView
    });

    var pinnedView = new JIRA.Projects.PinnableNavigator.Views.NavigatorView.PinnedView({
        model: model,
        id: myOptions.id,
        title: myOptions.title,
        manageText: myOptions.manageText,
        emptyText: myOptions.emptyText,
        searchView: searchView
    });

    var manageDialogControl = JIRA.Projects.PinnableNavigator.ManageDialog.create({
        model: model,
        title: myOptions.title,
        manageText: myOptions.manageText,
        emptyText: AJS.I18n.getText('project.subnavigator.manage.emptyText.warning')
    });

    return new JIRA.Projects.PinnableNavigator({
        id: myOptions.id,
        isPinned: !!myOptions.isPinned,
        pinnedHolderId: myOptions.pinnedHolderId,
        triggerView: triggerView,
        pinnedView: pinnedView,
        unpinnedView: unpinnedView,
        manageDialogControl: manageDialogControl,
        model: model,
        triggerHolderId: myOptions.triggerHolderId,
        unpinnedHolderId: myOptions.unpinnedHolderId,
        searchView: searchView
    });
};
