AJS.test.require(["com.atlassian.jira.jira-projects-plugin:sidebar-project-shortcuts"], function(){
    "use strict";

    module('JIRA.Projects.Sidebar.ProjectShortcuts.Entities.Shortcut', {
        setup: function () {
            this.model = new JIRA.Projects.Sidebar.ProjectShortcuts.Entities.Shortcut({
                id: "1",
                url: "http://example.com/",
                name: "Example",
                icon: "jira"
            }, {
                projectKey: "TEST"
            });

            this.server = sinon.fakeServer.create();
        },
        expectSuccessfulRequest: function () {
            this.server.respondWith(
                    AJS.contextPath() + "/rest/projects/1.0/project/TEST/shortcut/1",
                    [
                        200,
                        { "Content-Type": "application/json" },
                        '{ "id": 1, "url": "http://example.com/response", "name": "Example response", "icon": "Example icon" }'
                    ]
            );
        },
        expectFailedRequest: function () {
            this.server.respondWith(
                    AJS.contextPath() + "/rest/projects/1.0/project/TEST/shortcut/1",
                    [
                        400,
                        { "Content-Type": "application/json" },
                        '{ "errors": { "url": "Invalid url", "name": "Invalid name", "icon": "Invalid icon" } }'
                    ]
            );
        },
        teardown: function () {
            this.server.restore();
        }
    });

    test('Should throw when initialized without projectKey', function () {
        raises(
                function () {
                    new JIRA.Projects.Sidebar.ProjectShortcuts.Entities.Shortcut(undefined);
                },
                "Project key is required"
        );
    });

    test('Should remove id and set url, name and icon to empty string on clear', function () {
        this.model.clear();

        equal(this.model.get("id"), undefined);
        equal(this.model.get("url"), "");
        equal(this.model.get("name"), "");
        equal(this.model.get("icon"), "");
    });

    test('Should create model when it has no id when saving', function () {
        var syncSpy = this.sandbox.spy(this.model, "sync");
        this.model.unset("id");
        this.server.respondWith(
                AJS.contextPath() + "/rest/projects/1.0/project/TEST/shortcut",
                [
                    200,
                    { "Content-Type": "application/json" },
                    '{ "id": 1, "url": "http://example.com/response", "name": "Example response", "icon": "Example icon"}'
                ]
        );
        this.model.save();
        this.server.respond();

        sinon.assert.calledWith(syncSpy, "create");
    });

    test('Should update model when it has id when saving', function () {
        var syncSpy = this.sandbox.spy(this.model, "sync");
        this.expectSuccessfulRequest();
        this.model.save();
        this.server.respond();

        sinon.assert.calledWith(syncSpy, "update");
    });

    test('Should trigger save:start event when save request is sent', function () {
        var event = this.sandbox.spy();
        this.model.on("save:start", event);
        this.expectSuccessfulRequest();
        this.model.save();
        this.server.respond();

        sinon.assert.calledOnce(event);
    });

    test('Should trigger save:finish and save:success event when save request is successful', function () {
        var finish = this.sandbox.stub();
        var success = this.sandbox.stub();
        this.model.on("save:finish", finish);
        this.model.on("save:success", success);
        this.expectSuccessfulRequest();
        this.model.save();
        this.server.respond();

        sinon.assert.calledOnce(finish);
        sinon.assert.calledOnce(success);
        sinon.assert.callOrder(finish, success);
    });

    test('Should trigger save:finish and save:failure event with error details when save request fails', function () {
        var finish = this.sandbox.spy();
        var fail = this.sandbox.spy();
        this.model.on("save:finish", finish);
        this.model.on("save:failure", fail);
        this.expectFailedRequest();
        this.model.save();
        this.server.respond();

        sinon.assert.calledOnce(finish);
        sinon.assert.calledOnce(fail);
        sinon.assert.calledWith(fail, { "errors": { "url": "Invalid url", "name": "Invalid name", "icon": "Invalid icon" } });
        sinon.assert.callOrder(finish, fail);
    });

    test('Should not trigger another save/remove request when one is pending', function () {
        var finish = this.sandbox.spy();
        this.model.on("save:finish", finish);
        this.expectSuccessfulRequest();
        this.model.save();
        this.model.save();
        this.model.destroy();
        this.server.respond();

        sinon.assert.calledOnce(finish);
        equal(this.server.requests.length, 1);
    });

    test('Should trigger remove:start event when remove request is sent', function () {
        var event = this.sandbox.spy();
        this.model.on("remove:start", event);
        this.expectSuccessfulRequest();
        this.model.destroy();
        this.server.respond();

        sinon.assert.calledOnce(event);
    });

    test('Should trigger remove:finish and remove:success event when remove request is successful', function () {
        var finish = this.sandbox.spy();
        var success = this.sandbox.spy();
        this.model.on("remove:finish", finish);
        this.model.on("remove:success", success);
        this.expectSuccessfulRequest();
        this.model.destroy();
        this.server.respond();

        sinon.assert.calledOnce(finish);
        sinon.assert.calledOnce(success);
        sinon.assert.callOrder(finish, success);
    });

    test('Should trigger remove:finish and remove:failure event with error details when remove request fails', function () {
        var finish = this.sandbox.spy();
        var fail = this.sandbox.spy();
        this.model.on("remove:finish", finish);
        this.model.on("remove:failure", fail);
        this.expectFailedRequest();
        this.model.destroy();
        this.server.respond();

        sinon.assert.calledOnce(finish);
        sinon.assert.calledOnce(fail);
        sinon.assert.calledWith(fail, { "errors": { "url": "Invalid url", "name": "Invalid name", "icon": "Invalid icon" } })
        sinon.assert.callOrder(finish, fail);
    });

});

