(function(){
    "use strict";

    /**
     * Triggers a preventable event.
     *
     * The event will include an EventObject, and the handler can call eventObject.preventDefault() to prevent the event.
     *
     * @param {string} eventName Name of the event being triggered.
     * @param {EventObject} [eventObject] EventObject used as template to construct the actual EventObject used in the event.
     *
     * @returns {EventObject} EventObject passed to the event.
     */
    function triggerPreventable(eventName, eventObject) {
        var instance = this;
        /**
         * EventObject passed to preventable events
         * @typedef {Object} EventObject
         * @property {Object} emitter Original emitter of the event.
         * @property {boolean} isPrevented Whether the event has been prevented by the event handler.
         * @property {Function} preventDefault Syntax sugar for set the `isPrevented` value.
         */
        var event = _.defaults({}, eventObject || {}, {
            isPrevented: false,
            emitter: this,
            preventDefault: function () {
                this.isPrevented = true;
                instance.trigger(eventName + ":prevented", this);
            }
        });

        this.trigger(eventName, event);
        return event;
    }

    function retriggerPreventable(eventName, eventObject) {
        var groupEvent = this.triggerPreventable(eventName, eventObject);
        if (groupEvent.isPrevented) {
            eventObject.preventDefault();
        }
    }

    _.extend(JIRA.Projects.Libs.Marionette.View.prototype, {
        /**
         * This method unwraps the Backbone.View.
         *
         * By default, Backbone will create a <div> and render the template inside. By calling this
         * method, you can get rid of that <div>, so the main element in your template will be the
         * root element in your template.
         */
        unwrapTemplate: function unwrapTemplate() {
            if (this.$el.parent().length) {
                // If the template is already rendered in the page
                var children = this.$el.children();
                this.$el.replaceWith(children);
                this.setElement(children);
            } else {
                // If the template is in memory
                this.setElement(this.$el.children());
            }
        },
        triggerPreventable: triggerPreventable,
        retriggerPreventable: retriggerPreventable
    });

    _.extend(JIRA.Projects.Libs.Marionette.Controller.prototype, {
        triggerPreventable: triggerPreventable,
        retriggerPreventable: retriggerPreventable
    });

    JIRA.Projects.Libs.Marionette.ViewManager = JIRA.Projects.Libs.Marionette.Controller.extend({
        constructor: function() {
            JIRA.Projects.Libs.Marionette.Controller.apply(this, arguments);
            this.views={};
        },

        hideView: function(viewName) {
            var view = this.views[viewName];
            if (view) {
                this.stopListening(view);
                if (!view.isDestroyed) {
                    view.destroy();
                }
                delete this.views[viewName];
            }
        },

        showView: function(viewName, factory) {
            var view = this.buildView(viewName, factory);
            view.render();
        },

        buildView: function(viewName, factory) {
            var view = this.views[viewName];
            if (!view) {
                view = factory.call(this);
                this.listenTo(view, "destroy", function() {
                    this.hideView(viewName);
                });
                this.views[viewName] = view;
            }
            return view;
        },

        getView: function(viewName) {
            return this.views[viewName];
        }
    });

}());
