AJS.test.require(["com.atlassian.jira.jira-projects-plugin:sidebar-component"], function(){
    "use strict";

    module("JIRA.Projects.Sidebar.Component.NavigationItem", {
        markup: '<li><a class="aui-nav-item" data-link-id="my-navigation-item"></a></li>',
        markupWithBadge: '<li><a class="aui-nav-item" data-link-id="my-navigation-item"><span class="aui-badge">badge</span></a></li>',

        buildNavigationItem: function() {
            return new JIRA.Projects.Sidebar.Component.NavigationItem({
                el: this.markup
            });
        },

        buildSelectedNavigationItem: function() {
            var navigationItem = new JIRA.Projects.Sidebar.Component.NavigationItem({
                el: this.markup
            });
            navigationItem.select();
            return navigationItem;
        },

        buildNavigationItemWithLink: function(link) {
            var navigationItem = this.buildNavigationItem();

            var stub = this.stub(navigationItem.$el, "find");
            stub.withArgs("a").returns(link);

            return navigationItem;
        },

        buildNavigationItemWithBadge: function() {
            return new JIRA.Projects.Sidebar.Component.NavigationItem({
                el: this.markupWithBadge
            });
        }
    });

    test("Extracts the link ID from the markup when constructed", function() {
        var navigationItem = this.buildNavigationItem();

        equal(navigationItem.id, 'my-navigation-item');
    });

    test("When navigating on a item, it selects the item", function () {
        var navigationItem = this.buildNavigationItem();
        navigationItem.ui.link.attr("href", "http://example.com");
        this.stub(require('jira/util/browser'), "reloadViaWindowLocation");

        navigationItem.navigate();

        ok(navigationItem.isSelected());
    });

    test("When navigating on a item, it navigates to the link url", function() {
        var navigationItem = this.buildNavigationItem();
        navigationItem.ui.link.attr("href", "http://example.com");
        var stub = this.stub(require('jira/util/browser'), "reloadViaWindowLocation");

        navigationItem.navigate();

        sinon.assert.calledOnce(stub);
        sinon.assert.calledWith(stub, 'http://example.com');
    });

    test("When navigating on a item, it triggers the event 'before:select', 'select' and 'before:navigate', in that order", function() {
        var onBeforeSelect = this.spy();
        var onSelect = this.spy();
        var onBeforeNavigate = this.spy();
        var navigationItem = this.buildNavigationItem();
        navigationItem.on("before:navigate", onBeforeNavigate);
        navigationItem.on("before:select", onBeforeSelect);
        navigationItem.on("select", onSelect);
        this.stub(require('jira/util/browser'), "reloadViaWindowLocation");

        navigationItem.navigate();

        sinon.assert.calledOnce(onBeforeSelect);
        sinon.assert.calledOnce(onSelect);
        sinon.assert.calledOnce(onBeforeNavigate);
        sinon.assert.callOrder(onBeforeSelect, onSelect, onBeforeNavigate);
    });

    test("When navigating on a item, it does not trigger the event 'select' if 'before:select' has been prevented", function () {
        var onSelect = this.spy();
        var navigationItem = this.buildNavigationItem();
        navigationItem.on("before:select", function (ev) {ev.preventDefault();});
        navigationItem.on("select", onSelect);

        navigationItem.navigate();

        sinon.assert.notCalled(onSelect);
    });

    test("When navigating on a item, it does not trigger the event 'before:navigate' if 'before:select' has been prevented", function () {
        var onBeforeNavigate = this.spy();
        var navigationItem = this.buildNavigationItem();
        navigationItem.on("before:select", function (ev) {ev.preventDefault();});
        navigationItem.on("before:navigate", onBeforeNavigate);

        navigationItem.navigate();

        sinon.assert.notCalled(onBeforeNavigate);
    });

    test("When navigating on a item, it does not navigates if the 'before:select' event is prevented", function () {
        var navigationItem = this.buildNavigationItem();
        navigationItem.on("before:select", function (ev) {ev.preventDefault();});
        navigationItem.ui.link.attr("href", "http://example.com");
        var stub = this.stub(require('jira/util/browser'), "reloadViaWindowLocation");

        navigationItem.ui.link.click();

        sinon.assert.notCalled(stub);
    });

    test("When navigating on a item, it does not navigates if the 'before:navigate' event is prevented", function() {
        var navigationItem = this.buildNavigationItem();
        navigationItem.on("before:navigate", function(ev) {ev.preventDefault();});
        navigationItem.ui.link.attr("href", "http://example.com");
        var stub = this.stub(require('jira/util/browser'), "reloadViaWindowLocation");

        navigationItem.navigate();

        sinon.assert.notCalled(stub);
    });

    test("Selects the element", function() {
        var navigationItem = this.buildNavigationItem();

        navigationItem.select();

        ok(navigationItem.isSelected());
    });

    test("When selecting the item, it triggers the events 'before:select' and 'select', in that order", function() {
        var onBeforeSelect = this.spy();
        var onSelect = this.spy();
        var navigationItem = this.buildNavigationItem();
        navigationItem.on("before:select", onBeforeSelect);
        navigationItem.on("select", onSelect);

        navigationItem.select();

        sinon.assert.calledOnce(onBeforeSelect);
        sinon.assert.calledOnce(onSelect);
        sinon.assert.callOrder(onBeforeSelect, onSelect);
    });

    test("When selecting the item, it does not trigger the event 'select' if 'before:select' has been prevented", function() {
        var onSelect = this.spy();
        var navigationItem = this.buildNavigationItem();
        navigationItem.on("before:select", function(ev) {ev.preventDefault();});
        navigationItem.on("select", onSelect);

        navigationItem.select();

        ok(!onSelect.called);
    });

    test("Deselects the element", function() {
        var navigationItem = this.buildSelectedNavigationItem();

        navigationItem.deselect();

        ok(!navigationItem.isSelected());
    });

    test("When deselecting the item, it triggers the event 'before:deselect' and 'deselect', in that order", function() {
        var onBeforeDeselect = this.spy();
        var onDeselect = this.spy();
        var navigationItem = this.buildSelectedNavigationItem();
        navigationItem.on("before:deselect", onBeforeDeselect);
        navigationItem.on("deselect", onDeselect);

        navigationItem.deselect();

        sinon.assert.calledOnce(onBeforeDeselect);
        sinon.assert.calledOnce(onDeselect);
        sinon.assert.callOrder(onBeforeDeselect, onDeselect);
    });

    test("When deselecting the item, it does not trigger the event 'deselect' if 'before:deselect' has been prevented", function() {
        var onDeselect = this.spy();
        var navigationItem = this.buildSelectedNavigationItem();
        navigationItem.on("before:deselect", function(ev) {ev.preventDefault();});
        navigationItem.on("deselect", onDeselect);

        navigationItem.deselect();

        sinon.assert.notCalled(onDeselect);
    });

    test("When deselecting the item, it does not trigger any event if the element was not initially selected", function() {
        var onDeselect = this.spy();
        var onBeforeDeselected = this.spy();
        var navigationItem = this.buildNavigationItem();
        navigationItem.on("before:deselect", onBeforeDeselected);
        navigationItem.on("deselect", onDeselect);

        navigationItem.deselect();

        sinon.assert.notCalled(onDeselect);
        sinon.assert.notCalled(onBeforeDeselected);
    });

    test("When deselecting the navigation item, it removes the focus from its link", function() {
        var link = AJS.$("<a>");
        this.spy(link, "blur");

        var navigationItem = this.buildNavigationItemWithLink(link);
        navigationItem.select();

        navigationItem.deselect();

        sinon.assert.calledOnce(link.blur);
    });

    test("When clicking the item's link, it triggers the events 'before:select', 'select' and 'before:navigate', in that order", function() {
        var onBeforeSelect = this.spy();
        var onSelect = this.spy();
        var onBeforeNavigate = this.spy();
        var navigationItem = this.buildNavigationItem();
        navigationItem.on("before:select", onBeforeSelect);
        navigationItem.on("select", onSelect);
        navigationItem.on("before:navigate", onBeforeNavigate);

        navigationItem.ui.link.click();

        sinon.assert.calledOnce(onBeforeSelect);
        sinon.assert.calledOnce(onSelect);
        sinon.assert.calledOnce(onBeforeNavigate);
        sinon.assert.callOrder(onBeforeSelect, onSelect, onBeforeNavigate);
    });

    test("When clicking the item's link, it navigates to the link url", function() {
        var navigationItem = this.buildNavigationItem();
        navigationItem.ui.link.attr("href", "http://example.com");
        var stub = this.stub(require('jira/util/browser'), "reloadViaWindowLocation");

        navigationItem.ui.link.click();

        sinon.assert.calledOnce(stub);
        sinon.assert.calledWith(stub, 'http://example.com');
    });

    test("When clicking the item's link with ctrl/cmd key, it uses standard browser's mechanism to navigate to the link url", function() {
        var navigationItem = this.buildNavigationItem();
        navigationItem.ui.link.attr("href", "http://example.com");
        var navigateStub = this.stub(require('jira/util/browser'), "reloadViaWindowLocation");
        var eventStub = sinon.stub();
        navigationItem.on("before:navigate", eventStub);
        navigationItem.on("before:select", eventStub);
        navigationItem.on("select", eventStub);

        var event = jQuery.Event("click", {
            metaKey: true
        });
        navigationItem.ui.link.trigger(event);

        sinon.assert.notCalled(navigateStub);
        sinon.assert.notCalled(eventStub);
    });

    test("When clicking the item's link with target=_blank attribute, it uses standard browser's mechanism to navigate to the link url", function() {
        var navigationItem = this.buildNavigationItem();
        navigationItem.ui.link.attr("href", "http://example.com");
        navigationItem.ui.link.attr("target", "_blank");
        var navigateStub = this.stub(require('jira/util/browser'), "reloadViaWindowLocation");
        var eventStub = sinon.stub();
        navigationItem.on("before:navigate", eventStub);
        navigationItem.on("before:select", eventStub);
        navigationItem.on("select", eventStub);

        navigationItem.ui.link.trigger("click");

        sinon.assert.notCalled(navigateStub);
        sinon.assert.notCalled(eventStub);
    });

    test("When clicking the item's link, it does not trigger the event 'select' if 'before:select' has been prevented", function() {
        var onSelect = this.spy();
        var navigationItem = this.buildNavigationItem();
        navigationItem.on("before:select", function(ev) {ev.preventDefault();});
        navigationItem.on("select", onSelect);

        navigationItem.ui.link.click();

        sinon.assert.notCalled(onSelect);
    });

    test("When clicking the item's link, it does not navigates if the 'before:select' event is prevented", function() {
        var navigationItem = this.buildNavigationItem();
        navigationItem.on("before:select", function(ev) {ev.preventDefault();});
        navigationItem.ui.link.attr("href", "http://example.com");
        var stub = this.stub(require('jira/util/browser'), "reloadViaWindowLocation");

        navigationItem.ui.link.click();

        sinon.assert.notCalled(stub);
    });

    test("When clicking the item's link, it does not navigates if the 'before:navigate' event is prevented", function() {
        var navigationItem = this.buildNavigationItem();
        navigationItem.on("before:navigate", function(ev) {ev.preventDefault();});
        navigationItem.ui.link.attr("href", "http://example.com");
        var stub = this.stub(require('jira/util/browser'), "reloadViaWindowLocation");

        navigationItem.ui.link.click();

        sinon.assert.notCalled(stub);
    });

    test("When removing the badge, the right DOM element is removed", function() {
        var navigationItem = this.buildNavigationItemWithBadge();

        navigationItem.removeBadge();

        equal(navigationItem.$el.find("a .aui-badge").length, 0);
    });

    test("Returns the expected ID when the getId method is called", function() {
        var navigationItem = this.buildNavigationItem();

        equal(navigationItem.getId(), "my-navigation-item");
    });

    test("Returns itself if it is selected and the getSelectedNavigationItem method is called", function() {
        var navigationItem = this.buildSelectedNavigationItem();

        var selectedItem = navigationItem.getSelectedNavigationItem();

        ok(selectedItem === navigationItem);
    });

    test("Does not return itself if it isn't selected and the getSelectedNavigationItem method is called", function() {
        var navigationItem = this.buildNavigationItem();

        var selectedItem = navigationItem.getSelectedNavigationItem();

        ok(!(selectedItem === navigationItem));
    });

    test("Reports that it has a selected item when it is selected", function() {
        var navigationItem = this.buildSelectedNavigationItem();

        ok(navigationItem.hasASelectedItem());
    });

    test("Reports that it does not have a selected item when not selected", function() {
        var navigationItem = this.buildNavigationItem();

        ok(!navigationItem.hasASelectedItem());
    });

    test("When it is destroyed, events are unbound", function() {
        var view = this.buildNavigationItem();
        var listener = sinon.stub();
        view.on('test-event', listener);

        view.destroy();
        view.trigger('test-event');

        sinon.assert.notCalled(listener);
    });
});
