AJS.test.require(["com.atlassian.jira.jira-projects-plugin:sidebar-component"], function(){
    "use strict";

    module("JIRA.Projects.Sidebar.Component.NavigationGroup", {
        markup: [
            '<div data-id="my-navigation-group">',
            '   <ul>',
            '       <li><a class="aui-nav-item" data-link-id="my-navigation-item-1">Link #1</a></li>',
            '       <li><a class="aui-nav-item"></a>Link without id</li>',
            '       <li><a class="aui-nav-item" data-link-id="my-navigation-item-3">Link #3</a></li>',
            '       <li>',
            '           <a class="aui-nav-item" data-link-id="my-navigation-item-4">Subgroup</a>',
            '           <ul>',
            '               <li><a class="aui-nav-item" data-link-id="my-navigation-subitem-1">Link #1</a></li>',
            '           </ul>',
            '       </li>',
            '   </ul>',
            '</div>'
        ].join(''),

        buildNavigationGroup: function () {
            return new JIRA.Projects.Sidebar.Component.NavigationGroup({
                el: this.markup
            });
        },

        buildNavigationGroupWithDuplicatedIDs: function () {
            return new JIRA.Projects.Sidebar.Component.NavigationGroup({
                el: this.markup.replace("my-navigation-item-3","my-navigation-item-1")
            });
        },

        triggerEventWithObject: function(emitter, event) {
            var eventObject = {
                isPrevented: false,
                emitter: emitter,
                preventDefault: function () {
                    this.isPrevented = true;
                }
            };
            emitter.trigger(event, eventObject);
            return eventObject;
        },

        assertEventRetriggered: function(navigationGroup, emitter, event) {
            var eventHandler = this.spy();
            navigationGroup.on(event, eventHandler);

            var eventObject = this.triggerEventWithObject(emitter, event);

            sinon.assert.calledOnce(eventHandler);
            sinon.assert.calledWith(eventHandler, eventObject);
        },

        assertEventPreventedAndRetriggered: function(navigationGroup, emitter, event) {
            var handler = function(ev){ev.preventDefault();};
            emitter.on(event, handler);

            var eventHandler = this.spy();
            navigationGroup.on(event, eventHandler);

            var eventObject = this.triggerEventWithObject(emitter, event);

            sinon.assert.calledOnce(eventHandler);
            sinon.assert.calledWith(eventHandler, eventObject);
            ok(eventObject.isPrevented);
            emitter.off(event, handler);
        }
    });

    test("When constructed, it extracts the link ID from the markup", function () {
        var navigationGroup = this.buildNavigationGroup();

        equal(navigationGroup.id, 'my-navigation-group');
    });

    test("When constructed, it extracts the NavigationItems and NavigationSubgroups from the markup", function () {
        var navigationGroup = this.buildNavigationGroup();

        ok (navigationGroup.getItemAt(0) instanceof JIRA.Projects.Sidebar.Component.NavigationItem, "Item #0 is a NavigationItem");
        ok (navigationGroup.getItemAt(1) instanceof JIRA.Projects.Sidebar.Component.NavigationItem, "Item #1 is a NavigationItem");
        ok (navigationGroup.getItemAt(2) instanceof JIRA.Projects.Sidebar.Component.NavigationItem, "Item #2 is a NavigationItem");
        ok (navigationGroup.getItemAt(3) instanceof JIRA.Projects.Sidebar.Component.NavigationSubgroup, "Item #3 is a NavigationSubgroup");
    });

    test("When constructed, it lots a warning if the markup has duplicated IDs for groups", function() {
        var warnStub = this.stub(AJS,'warn');
        this.buildNavigationGroupWithDuplicatedIDs();

        sinon.assert.calledOnce(warnStub);
        sinon.assert.calledWith(warnStub, "Duplicated IDs detected. There are more than one NavigationItem with id data-link-id='my-navigation-item-1'");
    });

    test("It deselects all the items", function () {
        var deselectStub = this.stub(JIRA.Projects.Sidebar.Component.NavigationItem.prototype, "deselect");
        var navigationGroup = this.buildNavigationGroup();

        navigationGroup.deselect();

        sinon.assert.calledOn(deselectStub, navigationGroup.getItemAt(0));
        sinon.assert.calledOn(deselectStub, navigationGroup.getItemAt(1));
        sinon.assert.calledOn(deselectStub, navigationGroup.getItemAt(2));
    });

    test("When deselecting all the items, it propagates the call to the subgroups", function () {
        var deselectStub = this.stub(JIRA.Projects.Sidebar.Component.NavigationSubgroup.prototype, "deselect");
        var navigationGroup = this.buildNavigationGroup();

        navigationGroup.deselect();

        sinon.assert.calledOn(deselectStub, navigationGroup.getItemAt(3));
    });

    test("It fetches the items by the id", function () {
        var navigationGroup = this.buildNavigationGroup();

        ok(navigationGroup.getItem("my-navigation-item-1") instanceof JIRA.Projects.Sidebar.Component.NavigationItem, "Item #1 is a NavigationItem");
        ok(navigationGroup.getItem("my-navigation-item-3") instanceof JIRA.Projects.Sidebar.Component.NavigationItem, "Item #2 is a NavigationItem");
        ok(navigationGroup.getItem("my-navigation-item-4") instanceof JIRA.Projects.Sidebar.Component.NavigationGroup, "Item #3 is a NavigationGroup");
    });

    test("It fetches the items by the index", function () {
        var navigationGroup = this.buildNavigationGroup();

        ok(navigationGroup.getItemAt(0) instanceof JIRA.Projects.Sidebar.Component.NavigationItem, "Item #0 is a NavigationItem");
        ok(navigationGroup.getItemAt(1) instanceof JIRA.Projects.Sidebar.Component.NavigationItem, "Item #1 is a NavigationItem");
        ok(navigationGroup.getItemAt(2) instanceof JIRA.Projects.Sidebar.Component.NavigationItem, "Item #2 is a NavigationItem");
        ok(navigationGroup.getItemAt(3) instanceof JIRA.Projects.Sidebar.Component.NavigationGroup, "Item #3 is a NavigationGroup");
    });

    test("When a children triggers an event, the NavigationGroup re-triggers it with the same EventObject", function () {
        var navigationGroup = this.buildNavigationGroup();

        var navigationItem = navigationGroup.getItemAt(0);
        this.assertEventRetriggered(navigationGroup, navigationItem, "before:select");
        this.assertEventRetriggered(navigationGroup, navigationItem, "select");
        this.assertEventRetriggered(navigationGroup, navigationItem, "before:deselect");
        this.assertEventRetriggered(navigationGroup, navigationItem, "deselect");
        this.assertEventRetriggered(navigationGroup, navigationItem, "before:navigate");

        var navigationSubgroup = navigationGroup.getItemAt(3);
        this.assertEventRetriggered(navigationGroup, navigationSubgroup, "before:select");
        this.assertEventRetriggered(navigationGroup, navigationSubgroup, "select");
        this.assertEventRetriggered(navigationGroup, navigationSubgroup, "before:deselect");
        this.assertEventRetriggered(navigationGroup, navigationSubgroup, "deselect");
        this.assertEventRetriggered(navigationGroup, navigationSubgroup, "before:navigate");
    });

    test("When a children triggers an event, the NavigationGroup re-triggers it with the same EventObject, even if the original event was prevented", function () {
        var navigationGroup = this.buildNavigationGroup();

        var navigationItem = navigationGroup.getItemAt(0);
        this.assertEventPreventedAndRetriggered(navigationGroup, navigationItem, "before:select");
        this.assertEventPreventedAndRetriggered(navigationGroup, navigationItem, "before:deselect");
        this.assertEventPreventedAndRetriggered(navigationGroup, navigationItem, "before:navigate");

        var navigationSubgroup = navigationGroup.getItemAt(3);
        this.assertEventPreventedAndRetriggered(navigationGroup, navigationSubgroup, "before:select");
        this.assertEventPreventedAndRetriggered(navigationGroup, navigationSubgroup, "before:deselect");
        this.assertEventPreventedAndRetriggered(navigationGroup, navigationSubgroup, "before:navigate");
    });

    test("When a children triggers the before:select event, it deselects all the other items in the group", function () {
        var deselectItem = this.stub(JIRA.Projects.Sidebar.Component.NavigationItem.prototype, "deselect");
        var deselectSubgroup = this.stub(JIRA.Projects.Sidebar.Component.NavigationSubgroup.prototype, "deselect");
        var navigationGroup = this.buildNavigationGroup();

        this.triggerEventWithObject(navigationGroup.getItemAt(0), "before:select");

        sinon.assert.calledOn(deselectItem, navigationGroup.getItemAt(0));
        sinon.assert.calledOn(deselectItem, navigationGroup.getItemAt(1));
        sinon.assert.calledOn(deselectItem, navigationGroup.getItemAt(2));
        sinon.assert.calledOn(deselectSubgroup, navigationGroup.getItemAt(3));
    });

    test("When a children is selected, the item is not selected if the previously selected child prevents the 'before:deselect' event", function () {
        var navigationGroup = this.buildNavigationGroup();
        navigationGroup.getItemAt(0).select();
        navigationGroup.getItemAt(0).on("before:deselect", function(ev){ev.preventDefault();});

        navigationGroup.getItemAt(1).select();

        ok(navigationGroup.getItemAt(0).isSelected(), "Item #0 is still selected");
        ok(!navigationGroup.getItemAt(1).isSelected(), "Item #1 is not selected");
    });

    test("When deselecting all the items, it returns true if all the views were deselected", function () {
        var navigationGroup = this.buildNavigationGroup();

        var result = navigationGroup.deselect();

        strictEqual(result, true);
    });

    test("When deselecting all the items, it returns false if any 'before:deselect' event is cancelled", function () {
        var navigationGroup = this.buildNavigationGroup();
        navigationGroup.getItemAt(0).select();
        navigationGroup.getItemAt(0).on("before:deselect", function(ev){ev.preventDefault();});

        var result = navigationGroup.deselect();

        strictEqual(result, false);
    });

    test("When receiving a before:navigate:prevented event, we retrigger it", function() {
        var navigationGroup = this.buildNavigationGroup();
        var trigger = this.spy(navigationGroup, "trigger");

        var firstNavItem = navigationGroup.getItemAt(0);
        var eventObject = { emitter: firstNavItem };
        firstNavItem.trigger("before:navigate:prevented", eventObject);

        sinon.assert.calledWith(trigger, "before:navigate:prevented", eventObject);
    });

    test("Gets the correct navigation item if there is one selected", function() {
        var navigationGroup = this.buildNavigationGroup();
        var navItemToSelect = navigationGroup.getItemAt(2);
        navItemToSelect.select();

        ok(navigationGroup.getSelectedNavigationItem() === navItemToSelect);
    });

    test("Does not get the selected navigation item if one does not exist", function() {
        var navigationGroup = this.buildNavigationGroup();

        ok(navigationGroup.getSelectedNavigationItem() === undefined);
    });

    test("Reports that it has a selected navigation item if there is one selected", function() {
        var navigationGroup = this.buildNavigationGroup();
        var navItemToSelect = navigationGroup.getItemAt(2);
        navItemToSelect.select();

        ok(navigationGroup.hasASelectedItem());
    });

    test("Reports that it does not have a selected navigation item if none is selected", function() {
        var navigationGroup = this.buildNavigationGroup();

        ok(!navigationGroup.hasASelectedItem());
    });

    test("When it is destroyed, events are unbound", function() {
        var view = this.buildNavigationGroup();
        var listener = sinon.stub();
        view.on('test-event', listener);

        view.destroy();
        view.trigger('test-event');

        sinon.assert.notCalled(listener);
    });
});
