AJS.test.require(['com.atlassian.jira.jira-projects-plugin:sidebar-component'], function(){

    module('JIRA.Projects.Sidebar.Component', {
        setup: function() {
            this.context = AJS.test.mockableModuleContext();
            this.WRM = {
                data: {
                    claim: sinon.stub()
                }
            };

            this.context.mock('jira/projects/data/WRM', this.WRM);
            this.component = require('jira/projects/sidebar/component');
        },

        sidebarWithGroups: function() {
            return this.buildSidebar(
                [
                    '<div class="aui-sidebar">',
                    '   <div class="aui-sidebar-group" data-id="sidebar-navigation-panel">',
                    '      <ul>',
                    '         <li><a class="aui-nav-item" data-link-id="my-navigation-item-1"></a></li>',
                    '         <li><a class="aui-nav-item"></a></li>',
                    '         <li><a class="aui-nav-item" data-link-id="my-navigation-item-3"></a></li>',
                    '      </ul>',
                    '   </div>',
                    '   <div class="aui-sidebar-group">',
                    '      <ul>',
                    '         <li><a class="aui-nav-item" data-link-id="my-navigation-item-1"></a></li>',
                    '         <li><a class="aui-nav-item"></a></li>',
                    '         <li><a class="aui-nav-item" data-link-id="my-navigation-item-3"></a></li>',
                    '      </ul>',
                    '   </div>',
                    '   <div class="aui-sidebar-group" data-id="my-navigation-group-3">',
                    '      <ul>',
                    '         <li><a class="aui-nav-item" data-link-id="my-navigation-item-1"></a></li>',
                    '         <li><a class="aui-nav-item"></a></li>',
                    '         <li><a class="aui-nav-item" data-link-id="my-navigation-item-3"></a></li>',
                    '      </ul>',
                    '   </div>',
                    '</div>'
                ].join('')
            );
        },

        sidebarWithSubgroups: function() {
            return this.buildSidebar(
                [
                    '<div class="aui-sidebar">',
                    '   <div class="aui-sidebar-group" data-id="sidebar-navigation-panel">',
                    '      <ul>',
                    '         <li>',
                    '            <ul>',
                    '               <li><a href="#"></a></li>',
                    '            </ul>',
                    '         </li>',
                    '         <li><a class="aui-nav-item" data-link-id="my-navigation-item-1"></a></li>',
                    '      </ul>',
                    '   </div>',
                    '</div>'
                ].join('')
            );
        },

        sidebarWithDuplicatedIds: function() {
            return this.buildSidebar(
                [
                    '<div class="aui-sidebar">',
                    '   <div class="aui-sidebar-group" data-id="duplicated">',
                    '      <ul>',
                    '         <li><a class="aui-nav-item" data-link-id="my-navigation-item-1"></a></li>',
                    '         <li><a class="aui-nav-item"></a></li>',
                    '         <li><a class="aui-nav-item" data-link-id="my-navigation-item-3"></a></li>',
                    '      </ul>',
                    '   </div>',
                    '   <div class="aui-sidebar-group" data-id="duplicated">',
                    '      <ul>',
                    '         <li><a class="aui-nav-item" data-link-id="my-navigation-item-1"></a></li>',
                    '         <li><a class="aui-nav-item"></a></li>',
                    '         <li><a class="aui-nav-item" data-link-id="my-navigation-item-3"></a></li>',
                    '      </ul>',
                    '   </div>',
                    '</div>'
                ].join('')
            );
        },

        sidebarWithScopeFilter: function(scopeFilterId) {
            return this.buildSidebar(
                [
                    '<div class="aui-sidebar">',
                    '   <div class="scope-filter">',
                    '      <a class="scope-filter-trigger" data-scope-filter-id="' + scopeFilterId + '"></a>',
                    '   </div>',
                    '</div>'
                ].join('')
            );
        },

        sidebarWithoutScopeFilter: function() {
            return this.sidebarWithGroups();
        },

        sidebarWithReportsLink: function(reportsLinkId) {
            reportsLinkId = reportsLinkId || 'com.atlassian.jira.jira-projects-plugin:report-page';
            return this.buildSidebar(
                [
                    '<div class="aui-sidebar">',
                    '   <span class="aui-avatar-project"/>',
                    '   <div class="aui-sidebar-group" data-id="sidebar-navigation-panel">',
                    '      <ul class="aui-nav">',
                    '         <li>',
                    '            <a class="aui-nav-item" href="#" data-link-id="' + reportsLinkId + '">',
                    '               <span>Reports</span>',
                    '            </a>',
                    '         </li>',
                    '      </ul>',
                    '   </div>',
                    '</div>'
                ].join('')
            );
        },

        buildSidebar: function(markup) {
            return new this.component({
                el: markup
            });
        },

        buildGroup: function(markup) {
            return new JIRA.Projects.Sidebar.Component.NavigationGroup({
                el: markup
            });
        },

        triggerEventWithObject: function(emitter, event) {
            var eventObject = {
                isPrevented: false,
                emitter: emitter,
                preventDefault: function() {
                    this.isPrevented = true;
                }
            };
            emitter.trigger(event, eventObject);
            return eventObject;
        },

        getReportsLink: function(sidebar) {
            return sidebar.$el.find([
                "a[data-link-id='com.atlassian.jira.jira-projects-plugin:report-page']",
                "a[data-link-id='com.pyxis.greenhopper.jira:global-sidebar-report']"
            ].join(",")).attr("href");
        },

        assertEventRetriggered: function(navigationGroup, emitter, event) {
            var eventHandler = this.spy();
            navigationGroup.on(event, eventHandler);

            var eventObject = this.triggerEventWithObject(emitter, event);

            sinon.assert.calledOnce(eventHandler);
            sinon.assert.calledWith(eventHandler, eventObject);
        },

        assertEventPreventedAndRetriggered: function(navigationGroup, emitter, event) {
            var handler = function(ev) {ev.preventDefault();};
            emitter.on(event, handler);

            var eventHandler = this.spy();
            navigationGroup.on(event, eventHandler);

            var eventObject = this.triggerEventWithObject(emitter, event);

            sinon.assert.calledOnce(eventHandler);
            sinon.assert.calledWith(eventHandler, eventObject);
            ok(eventObject.isPrevented);
            emitter.off(event, handler);
        }
    });

    test("When constructed, it extracts the NavigationGroups", function() {
        var navigationGroupSpy = this.spy(JIRA.Projects.Sidebar.Component, "NavigationGroup");
        this.sidebarWithGroups();

        ok(navigationGroupSpy.calledThrice);
    });

    test("When constructed, it logs a warning if the markup has duplicated IDs for groups", function() {
        var warnStub = this.stub(AJS, 'warn');
        this.sidebarWithDuplicatedIds();

        sinon.assert.calledOnce(warnStub);
        sinon.assert.calledWith(warnStub, "Duplicated IDs detected. There are more than one NavigationGroup with id data-id=\"duplicated\"");
    });

    test("It deselect all groups", function() {
        var deselectStub = this.stub(JIRA.Projects.Sidebar.Component.NavigationGroup.prototype, "deselect");
        var sidebar = this.sidebarWithGroups();

        sidebar.deselectAllGroups();

        ok(deselectStub.calledThrice);
    });

    test("It can fetch the groups by the id", function() {
        var sidebar = this.sidebarWithGroups();

        ok(sidebar.getGroup("sidebar-navigation-panel") instanceof JIRA.Projects.Sidebar.Component.NavigationGroup);
        ok(sidebar.getGroup("my-navigation-group-3") instanceof JIRA.Projects.Sidebar.Component.NavigationGroup);
    });

    test("It can fetch the groups by the index", function() {
        var sidebar = this.sidebarWithGroups();

        ok(sidebar.getGroupAt(0) instanceof JIRA.Projects.Sidebar.Component.NavigationGroup);
        ok(sidebar.getGroupAt(1) instanceof JIRA.Projects.Sidebar.Component.NavigationGroup);
        ok(sidebar.getGroupAt(2) instanceof JIRA.Projects.Sidebar.Component.NavigationGroup);
        ok(sidebar.getGroup("sidebar-navigation-panel") === sidebar.getGroupAt(0));
        ok(sidebar.getGroup("my-navigation-group-3") === sidebar.getGroupAt(2));
    });

    test("It can fetch the default group", function() {
        var sidebar = this.sidebarWithGroups();

        ok(sidebar.getDefaultGroup() === sidebar.getGroup('sidebar-navigation-panel'));
        ok(sidebar.getDefaultGroup() === sidebar.getGroupAt(0));
    });

    test("It can fetch an item from the default group", function() {
        var sidebar = this.sidebarWithGroups();

        ok(sidebar.getItem("my-navigation-item-1") === sidebar.getDefaultGroup().getItem("my-navigation-item-1"));
    });

    test("It can fetch a subgroup from the default group", function() {
        var sidebar = this.sidebarWithSubgroups();

        ok(sidebar.getDefaultGroup().getItemAt(0) instanceof JIRA.Projects.Sidebar.Component.NavigationSubgroup);
    });

    test("It can fetch items from a subgroup", function() {
        var sidebar = this.sidebarWithSubgroups();

        ok(sidebar.getDefaultGroup().getItemAt(0).getItemAt(0) instanceof JIRA.Projects.Sidebar.Component.NavigationItem);
    });

    test("It can replace an existing group", function() {
        var sidebar = this.sidebarWithGroups();
        var newGroup = this.buildGroup([
            '<div class="aui-sidebar-group" data-id="my-new-group">',
            '   <ul>',
            '      <li><a class="aui-nav-item" data-link-id="my-new-item"></a></li>',
            '   </ul>',
            '</div>'
        ].join(''));

        sidebar.replaceGroup('sidebar-navigation-panel', newGroup);

        ok(sidebar.getGroup('sidebar-navigation-panel') === undefined);
        ok(sidebar.getGroup('my-new-group') === newGroup);
        ok(sidebar.getGroup('my-new-group').getItem('my-new-item') instanceof JIRA.Projects.Sidebar.Component.NavigationItem);
    });

    test("When an item of a particular group is selected, it deselect all the items in all the groups", function() {
        var deselectStub = this.stub(JIRA.Projects.Sidebar.Component.NavigationGroup.prototype, "deselect");
        var sidebar = this.sidebarWithGroups();

        sidebar.getGroupAt(0).getItemAt(0).select();

        // 1st call: NavigationGroup detects the selection and deselect all its items.
        // 2nd, 3rd and 4th calls: Sidebar detects the selection and call deselect for all the groups.
        equal(deselectStub.callCount, 4);
    });

    test("When a children triggers an event, the Component re-triggers it with the same EventObject", function() {
        var sidebar = this.sidebarWithGroups();

        var navigationGroup = sidebar.getGroupAt(0);
        this.assertEventRetriggered(sidebar, navigationGroup, "before:select");
        this.assertEventRetriggered(sidebar, navigationGroup, "select");
        this.assertEventRetriggered(sidebar, navigationGroup, "before:deselect");
        this.assertEventRetriggered(sidebar, navigationGroup, "deselect");
        this.assertEventRetriggered(sidebar, navigationGroup, "before:navigate");
    });

    test("When a children triggers an event, the Component re-triggers it with the same EventObject, even if the original event was prevented", function() {
        var sidebar = this.sidebarWithGroups();

        var navigationGroup = sidebar.getGroupAt(0);
        this.assertEventPreventedAndRetriggered(sidebar, navigationGroup, "before:select");
        this.assertEventPreventedAndRetriggered(sidebar, navigationGroup, "before:deselect");
        this.assertEventPreventedAndRetriggered(sidebar, navigationGroup, "before:navigate");
    });

    test("Can retrieve the identifier of the selected scope in the scope filter", function() {
        var sidebar = this.sidebarWithScopeFilter("scope-filter-id");

        var scopeFilterId = sidebar.getSelectedScopeFilterId();

        equal(scopeFilterId, "scope-filter-id");
    });

    test("Returns undefined as the identifier of the selected scope if there is no scope filter", function() {
        var sidebar = this.sidebarWithoutScopeFilter();

        var scopeFilterId = sidebar.getSelectedScopeFilterId();

        equal(scopeFilterId, undefined);
    });

    test("Sets the reports link correctly", function() {
        var sidebar = this.sidebarWithReportsLink();

        sidebar.setReportsItemLink("http://atlassian.com");

        equal(this.getReportsLink(sidebar), "http://atlassian.com");
    });

    test("Sets the reports link correctly for Agile-provided reports link", function() {
        var sidebar = this.sidebarWithReportsLink('com.pyxis.greenhopper.jira:global-sidebar-report');

        sidebar.setReportsItemLink("http://some.other.link.com");

        equal(this.getReportsLink(sidebar), "http://some.other.link.com");
    });

    test("When it receives a before:select event and it does not have a selected item, it retriggers the before:select with an additional flag", function() {
        var sidebar = this.sidebarWithGroups();
        var trigger = this.spy(sidebar, "retriggerPreventable");
        var event = {
            emitter: sidebar.getDefaultGroup(),
            isPrevented: false
        };
        var eventWithInitialFlag = _.extend({ isInitial: true }, event);

        sidebar.getDefaultGroup().trigger("before:select", event);

        sinon.assert.calledWith(trigger, "before:select", eventWithInitialFlag);
    });

    test("When it receives a before:navigate:prevented event, it retriggers it", function() {
        var sidebar = this.sidebarWithGroups();
        var trigger = this.spy(sidebar, "trigger");
        var event = { emitter: sidebar.getDefaultGroup() };

        sidebar.getDefaultGroup().trigger("before:navigate:prevented", event);

        sinon.assert.calledWith(trigger, "before:navigate:prevented", event);
    });

    test("When it contains a selected item, it returns the selected item when getSelectedNavigationItem is called", function() {
        var sidebar = this.sidebarWithGroups();
        var itemToSelect = sidebar.getItem("my-navigation-item-1");
        itemToSelect.select();

        ok(sidebar.getSelectedNavigationItem() === itemToSelect);
    });

    test("When does not contain a selected item, it returns undefined when getSelectedNavigationItem is called", function() {
        var sidebar = this.sidebarWithGroups();

        ok(sidebar.getSelectedNavigationItem() === undefined);
    });

    test("When it contains a selected item, it returns true when the hasASelectedItem method is called", function() {
        var sidebar = this.sidebarWithGroups();
        sidebar.getItem("my-navigation-item-1").select();

        ok(sidebar.hasASelectedItem());
    });

    test("When it does not contain a selected item, it returns false when the hasASelectedItem method is called", function() {
        var sidebar = this.sidebarWithGroups();

        ok(!sidebar.hasASelectedItem());
    });

    test("It adds a dimmed attribute to the sidebar when the dim method is called", function() {
        var sidebar = this.sidebarWithGroups();

        sidebar.dim();

        ok(sidebar.$el.is("[dimmed]"));
    });

    test("When the dim method is called multiple times the sidebar remains with a dimmed attribute", function() {
        var sidebar = this.sidebarWithGroups();

        sidebar.dim();
        sidebar.dim();

        ok(sidebar.$el.is("[dimmed]"));
    });

    test("It removes the dimmed attribute when undim is called", function() {
        var sidebar = this.sidebarWithGroups();

        sidebar.dim();
        sidebar.undim();

        ok(!sidebar.$el.is("[dimmed]"));
    });

    test("When dim is not called, undim behaves as a no-op", function() {
        var sidebar = this.sidebarWithGroups();

        sidebar.undim();

        ok(!sidebar.$el.is("[dimmed]"));
    });

    test("It should know when it is in a global board context", function() {
        this.WRM.data.claim.withArgs("is-global-sidebar").returns(true);
        var Component = this.context.require("jira/projects/sidebar/component"),
            instance = new Component();

        var result = instance.isProjectSidebar();

        ok(this.WRM.data.claim.calledOnce);
        ok(!result);

    });

    test("It should know when it is a project board context", function() {
        this.WRM.data.claim.withArgs("is-global-sidebar").returns(false);
        var Component = this.context.require("jira/projects/sidebar/component"),
            instance = new Component();

        var result = instance.isProjectSidebar();

        ok(this.WRM.data.claim.calledOnce);
        ok(result);
    });

    test("It can be re-rendered with new markup", function() {
        var sidebar = this.sidebarWithGroups();
        var listener = sinon.stub();
        sidebar.on('render', listener);

        sidebar.render({
            el: [
                '<div class="aui-sidebar">',
                '   <div class="aui-sidebar-group" data-id="test-sidebar-group">',
                '      <ul>',
                '         <li><a class="aui-nav-item" data-link-id="test-sidebar-item"></a></li>',
                '         <li><a class="aui-nav-item"></a></li>',
                '      </ul>',
                '   </div>',
                '   <div class="aui-sidebar-group">',
                '      <ul>',
                '         <li><a class="aui-nav-item" data-link-id="test-sidebar-item-2"></a></li>',
                '      </ul>',
                '   </div>',
                '</div>'
            ].join('')
        });

        sinon.assert.calledOnce(listener);

        ok(sidebar.getGroupAt(0) instanceof JIRA.Projects.Sidebar.Component.NavigationGroup);
        ok(sidebar.getGroupAt(1) instanceof JIRA.Projects.Sidebar.Component.NavigationGroup);
        ok(sidebar.getGroup("test-sidebar-group") === sidebar.getGroupAt(0));
        ok(sidebar.getGroup("test-sidebar-group").getItemAt(0) instanceof JIRA.Projects.Sidebar.Component.NavigationItem);
        ok(sidebar.getGroup("test-sidebar-group").getItem("test-sidebar-item") === sidebar.getGroup("test-sidebar-group").getItemAt(0));
    });

    test("It isn't re-rendered if the markup hasn't been updated", function() {
        var sidebar = this.sidebarWithGroups();
        var listener = sinon.stub();
        sidebar.on('render', listener);

        sidebar.render();

        sinon.assert.notCalled(listener);
    });

    test("It can be forced to re-render if necessary", function() {
        var sidebar = this.sidebarWithGroups();
        var listener = sinon.stub();
        sidebar.on('render', listener);

        sidebar.render({force: true});

        sinon.assert.calledOnce(listener);
    });

    test("It fires a 'detach' event when child components are destroyed", function() {
        var sidebar = this.sidebarWithGroups();
        var listener = sinon.stub();
        sidebar.on('detach', listener);

        sidebar.render({force: true});

        sinon.assert.calledOnce(listener);
    });

    test("It can be destroyed", function() {
        var sidebar = this.sidebarWithGroups();
        var listener = sinon.stub();
        sidebar.on('destroy', listener);

        sidebar.destroy();

        sinon.assert.calledOnce(listener);
    });

    test("It fires lifecycle events in the correct order", function() {
        var sidebar = this.sidebarWithGroups();
        var preRenderListener = sinon.stub();
        var preDetachListener = sinon.stub();
        var detachListener = sinon.stub();
        var renderListener = sinon.stub();
        var destroyListener = sinon.stub();
        sidebar.on('before:render', preRenderListener);
        sidebar.on('before:detach', preDetachListener);
        sidebar.on('detach', detachListener);
        sidebar.on('render', renderListener);
        sidebar.on('destroy', destroyListener);

        sidebar.render({force: true});
        sidebar.destroy();

        ok(preRenderListener.calledBefore(preDetachListener));
        ok(preDetachListener.calledBefore(detachListener));
        ok(detachListener.calledBefore(renderListener));
        ok(renderListener.calledBefore(destroyListener));
        ok(destroyListener.calledOnce);
    });

    test("It only fires each lifecycle event once per cycle", function() {
        var sidebar = this.sidebarWithGroups();
        var preRenderListener = sinon.stub();
        var preDetachListener = sinon.stub();
        var detachListener = sinon.stub();
        var renderListener = sinon.stub();
        var destroyListener = sinon.stub();
        sidebar.on('before:render', preRenderListener);
        sidebar.on('before:detach', preDetachListener);
        sidebar.on('detach', detachListener);
        sidebar.on('render', renderListener);
        sidebar.on('destroy', destroyListener);

        sidebar.render({force: true});
        sidebar.destroy();

        sinon.assert.calledOnce(preRenderListener);
        sinon.assert.calledOnce(preDetachListener);
        sinon.assert.calledOnce(detachListener);
        sinon.assert.calledOnce(renderListener);
        sinon.assert.calledOnce(destroyListener);
    });

    test("When it is destroyed, events on the sidebar root are still bound", function() {
        var sidebar = this.sidebarWithGroups();
        var listener = sinon.stub();
        sidebar.on('test-event', listener);

        sidebar.destroy();
        sidebar.trigger('test-event');

        sinon.assert.calledOnce(listener);
    });

    test("When it is destroyed, all event listeners are unbound from NavigationGroups", function() {
        var sidebar = this.sidebarWithGroups();
        var navigationGroup = sidebar.getGroupAt(0);
        var listener = sinon.stub();
        navigationGroup.on('test-event', listener);

        sidebar.destroy();
        navigationGroup.trigger('test-event');

        sinon.assert.notCalled(listener);
    });

    test("When it is destroyed, all event listeners are unbound from NavigationSubgroups", function() {
        var sidebar = this.sidebarWithSubgroups();
        var navigationSubgroup = sidebar.getGroupAt(0).getItemAt(0);
        var listener = sinon.stub();
        navigationSubgroup.on('test-event', listener);

        sidebar.destroy();
        navigationSubgroup.trigger('test-event');

        sinon.assert.notCalled(listener);
    });

    test("When it is destroyed, all event listeners are unbound from NavigationItems", function() {
        var sidebar = this.sidebarWithGroups();
        var navigationItem = sidebar.getGroup('sidebar-navigation-panel').getItem('my-navigation-item-1');
        var listener = sinon.stub();
        navigationItem.on('test-event', listener);

        sidebar.destroy();
        navigationItem.trigger('test-event');

        sinon.assert.notCalled(listener);
    });

    test("When it is re-rendered, events on the sidebar root are still bound", function() {
        var sidebar = this.sidebarWithGroups();
        var listener = sinon.stub();
        sidebar.on('test-event', listener);

        sidebar.render({el: '<div class="aui-sidebar"></div>'});
        sidebar.trigger('test-event');

        sinon.assert.calledOnce(listener);
    });

    test("When it is re-rendered, event listeners are unbound from old NavigationGroups", function() {
        var sidebar = this.sidebarWithGroups();
        var navigationGroup = sidebar.getGroupAt(0);
        var listener = sinon.stub();
        navigationGroup.on('test-event', listener);

        sidebar.render({el: '<div class="aui-sidebar"></div>'});
        navigationGroup.trigger('test-event');

        sinon.assert.notCalled(listener);
    });

    test("When it is re-rendered, event listeners are unbound from old NavigationSubgroups", function() {
        var sidebar = this.sidebarWithSubgroups();
        var navigationSubgroup = sidebar.getGroupAt(0).getItemAt(0);
        var listener = sinon.stub();
        navigationSubgroup.on('test-event', listener);

        sidebar.render({el: '<div class="aui-sidebar"></div>'});
        navigationSubgroup.trigger('test-event');

        sinon.assert.notCalled(listener);
    });

    test("When it is re-rendered, event listeners are unbound from old NavigationItems", function() {
        var sidebar = this.sidebarWithGroups();
        var navigationItem = sidebar.getGroup('sidebar-navigation-panel').getItem('my-navigation-item-1');
        var listener = sinon.stub();
        navigationItem.on('test-event', listener);

        sidebar.render({el: '<div class="aui-sidebar"></div>'});
        navigationItem.trigger('test-event');

        sinon.assert.notCalled(listener);
    });

    test("When it is destroyed, it is removed from the page", function() {
        var sidebar = this.sidebarWithGroups();
        var container = jQuery('<div></div>').append(sidebar.getElement());

        sidebar.destroy();

        equal(container.find('.aui-sidebar').length, 0);
    });

    test("When it is destroyed, its children no longer exist", function() {
        var sidebar = this.sidebarWithGroups();

        sidebar.destroy();

        equal(sidebar.children.length, 0);
    });

    test("Detach can be cancelled", function() {
        var sidebar = this.sidebarWithGroups();
        var listener = sinon.stub();
        sidebar.on('before:detach', function(event) {
            event.preventDefault();
        });
        sidebar.on('detach', listener);

        sidebar.render({force: true});

        sinon.assert.notCalled(listener);
    });

    test("Render can be cancelled", function() {
        var sidebar = this.sidebarWithGroups();
        var listener = sinon.stub();
        sidebar.on('before:render', function(event) {
            event.preventDefault();
        });
        sidebar.on('render', listener);

        sidebar.render({force: true});

        sinon.assert.notCalled(listener);
    });

    test("It doesn't continue with the render and detach action if 'before:render' is prevented", function() {
        var sidebar = this.sidebarWithGroups();
        sidebar.on('before:render', function(event) {
            event.preventDefault();
        });
        var preDetachListener = sinon.stub();
        var detachListener = sinon.stub();
        var renderListener = sinon.stub();
        sidebar.on('before:detach', preDetachListener);
        sidebar.on('detach', detachListener);
        sidebar.on('render', renderListener);

        sidebar.render({force: true});

        sinon.assert.notCalled(preDetachListener);
        sinon.assert.notCalled(detachListener);
        sinon.assert.notCalled(renderListener);
    });

    test("Sidebar render updates page DOM", function() {
        var sidebar = this.sidebarWithGroups();
        var parent = jQuery('<div id="parent"></div>');
        parent.append(sidebar.$el);

        sidebar.render({el: '<div class="aui-sidebar"></div>'});

        ok(sidebar.$el.parent().length > 0);
    })
});
