require([
    "jquery",
    "underscore",
    "jira/skate",
    "jira/projects/release/analytics",
    "jira/projects/release/calendar-params-data",
    "jira/ajs/ajax/ajax-util"
], function(
    $,
    _,
    skate,
    Analytics,
    calendarParams,
    AjaxUtil
) {
    "use strict";

    skate("version-edit-form", {
        type: skate.type.CLASSNAME,

        prototype: {

            init: function() {
                Analytics.showEditVersion();

                $(this).find('input[date-picker]').each(function(idx, inputField) {
                    var params = _.extend({}, calendarParams, {
                        button: $(inputField).next('a'),
                        inputField: inputField
                    });
                    Calendar.setup(params);
                });

                this.bindEvents();
            },

            destroy: function() {
                this.unbindEvents();
            },

            bindEvents: function() {
                $(this).on("submit.versionEdit", this.submit.bind(this));
            },

            unbindEvents: function() {
                $(this).off(".versionEdit");
            },

            submit: function(event) {
                event.preventDefault();

                var data = this.getData();
                this.disable();

                this.clearMessages();

                $.ajax({
                    type: 'PUT',
                    url: this.action,
                    contentType: "application/json",
                    data: JSON.stringify(data)
                }).done(this.onSubmitSuccess.bind(this))
                  .fail(this.onSubmitFailure.bind(this))
                  .always(function() {
                      this.enable();
                      $(this).trigger("submitFinished");
                      JIRA.trace("jira.version.edit.save.done")
                  }.bind(this));
            },

            onSubmitSuccess: function() {
                Analytics.editVersionSuccess();

                $(this).trigger("submitSuccess");

                if (!this.isInDialog()) {
                    $(aui.message.success({
                        content: AJS.I18n.getText('project.page.release.edit.save.success')
                    })).prependTo(this.querySelector('.form-body'));
                }
            },

            onSubmitFailure: function(response) {
                Analytics.editVersionFail();

                $(this).trigger("submitFailure");


                try {
                    var result = JSON.parse(response.responseText);

                    _.each(result.errors, function (error, field) {
                        $(JIRA.Projects.Release.Templates.errorDiv({
                            name: field,
                            errors: result,
                            extraAttributes: 'data-field=' + field + ''
                        })).appendTo($(this.getFields()[field]).parent());
                    }, this);


                    result.errorMessages.forEach(this.showError, this);
                } catch(e) {
                    this.showError(AjaxUtil.getErrorMessageFromXHR(response));
                }
            },

            showError: function(message) {
                $(aui.message.error({
                    content: message
                })).prependTo(this.querySelector('.form-body'))
            },

            getData: function() {
                var data = _.object(_.map($(this).serializeArray(), _.values));

                return _.omit(data, "decorator");
            },

            clearMessages: function() {
                $(this).find('div.error, div.aui-message-success').remove();
            },

            enable: function() {
                this.getInputs().removeAttr("disabled");
            },

            disable: function() {
                this.getInputs().attr("disabled", "disabled");
            },

            getInputs: function() {
                return $(this).find('button, input');
            },

            getFields: function() {
                return $(this).find('input[data-field-name], input[name]').toArray().reduce(function(r, el) {
                    r[$(el).attr("data-field-name") || el.name] = el;

                    return r;
                }, {});
            },

            isInDialog: function() {
                return $(this).parents(".jira-dialog").length === 1;
            }
        },

        attached: function(element) {
            element.init();
        },

        detached: function(element) {
            element.destroy();
        }
    })
});