define("jira/projects/release/item-view", [
    "jira/projects/abstract-list/item-view",
    "jquery",
    "jira/projects/libs/marionette",
    "jira/projects/abstract-model/submit-model-states",
    "jira/ajs/ajax/ajax-util",
    "jira-project-config/utils",
    "jira/projects/release/version-adapter"
], function (
    AbstractItemView,
    $,
    Marionette,
    SubmitModelStates,
    ajaxUtil,
    Utils,
    VersionAdapter
) {
    "use strict";

    var DeleteForm = JIRA.Admin.Version.DeleteForm;
    var ReleaseForm = JIRA.Admin.Version.ReleaseForm;

    var ItemView = AbstractItemView.extend({
        template: JIRA.Projects.Release.Templates.row,

        ui: _.extend({}, AbstractItemView.prototype.ui, {
            "dragHandle": ".versions-table__handle .aui-icon",
            "progressbar":  ".progressbar",
            "deleteLink": ".project-config-operations-delete",
            "releaseversionLink": ".project-config-operations-release",
            "unreleaseversionLink": ".project-config-operations-unrelease",
            "archiveversionLink": ".project-config-operations-archive",
            "unarchiveversionLink": ".project-config-operations-unarchive",
            "actionButtons": ".version-actions a",
            "detailsButton": ".details-button"
        }),

        modelEvents: _.extend({}, AbstractItemView.prototype.modelEvents, {
            "retry-delete": "_deleteItemRetry",
            "retry-update": "_updateItemRetry"
        }),

        outterEvents: {
            "click @ui.actionButtons": "_closeDropdown",
            "click @ui.deleteLink": "deleteItem",
            "click @ui.releaseversionLink": "release",
            "click @ui.unreleaseversionLink": "unrelease",
            "click @ui.archiveversionLink": "archive",
            "click @ui.unarchiveversionLink": "unarchive"
        },

        onRender: function onRender() {
            this.listenToElements(this.outterEvents);
            this.unwrapTemplate();
            this.initializeProgressDialog();
        },

        _discardOperation: function (e) {
            e.preventDefault();
            this.model.discard();

            if (this.flag) {
                this.flag.close();
            }
        },

        deleteItem: function (e) {
            e.preventDefault();
            this._deletePerform();
        },

        _deleteItemRetry: function () {
            this.model.set("state", SubmitModelStates.IN_PROGRESS_ERROR);
            this.model.destroy(this.model.lastDestroyOptions);
        },

        _deletePerform: function () {
            this._showLoading();
            var detailsSync = this.model.detailsSync();
            detailsSync.then(function() {
                Utils.openDialogForRow(DeleteForm, this, "version-" + this.model.get("id") + "-delete-dialog", {width: 700});
            }.bind(this)).fail(function(xhr) {
                this.model.set("state", SubmitModelStates.ERROR_DELETE);
                this.model.set("errorMsg", getErrorMessageFromXHR(xhr));
            }.bind(this), this._hideLoading.bind(this));
        },

        _updateItemRetry: function () {
            this._saveChanges(this.model.lastSaveAttributes);
        },

        _saveChanges: function (attr) {
            this.model.save(attr).done(function () {
                setTimeout(function () {
                    if (this.model.get("state") === SubmitModelStates.SUCCESSFUL) {
                        this.model.set("state", SubmitModelStates.READY);
                    }
                }.bind(this), 5000);
            }.bind(this));
        },

        /**
         * Opens Release Dialog
         */
        release: function (e) {
            e.preventDefault();
            this._releasePerform();
        },

        _releasePerform: function () {
            // Release Version dialog uses incompatible model type
            // There are type difference in 'releaseDate' model property
            // the dialog also relies on some restful table methods to be present
            var model = this.model;
            var versionModel = new VersionAdapter(model);
            var releaseDialog = Utils.openDialogForRow(ReleaseForm, versionModel, "version-release-dialog");
            $(releaseDialog).on("Dialog.hide", function onDialogHide() {
                var submitDataObj = this.$form.serializeObject();
                var moveVersionId = submitDataObj.moveUnfixedIssuesTo;
                if (moveVersionId) {
                    // reload the version where issues has been transferred to
                    var version = model.collection.get(moveVersionId);
                    if (version) {
                        version.detailsSync().always(function () {
                            JIRA.trace("jira.version.release.reload.finished");
                        });
                    }
                }
            });
            return releaseDialog;
        },

        unrelease: function (e) {
            e.preventDefault();
            this._unreleasePerform();
        },

        _unreleasePerform: function () {
            this._saveChanges({
                released: false
            });
        },

        archive: function (e) {
            e.preventDefault();
            this._archivePerform();
        },

        _archivePerform: function () {
            this._saveChanges({
                archived: true
            });
        },

        unarchive: function (e) {
            e.preventDefault();
            this._unarchivePerform();
        },

        _unarchivePerform: function () {
            this._saveChanges({
                archived: false
            });
        },

        initializeProgressDialog: function initializeProgressDialog() {
            var instance = this;
            var versionInlineDialog = AJS.InlineDialog(this.ui.progressbar, "version-progress-" + this.cid,
                function(content, trigger, showPopup) {
                    var $trigger = AJS.$(trigger);
                    var $row = $trigger.closest("tr");
                    var $versionName = $row.find("td.versions-table__name a");
                    var entries = [];
                    var totalCount = 0;
                    $trigger.find(".progressbar__entry").each(function(i, el){
                        var $el = $(el);
                        var count = +$el.attr("data-count");
                        totalCount += +count;
                        entries.push({
                            name: $el.attr("data-title"),
                            type: $el.attr("data-type"),
                            url: $el.attr("data-url"),
                            count: count
                        });
                    });
                    content.html(JIRA.Projects.Release.Templates.progressBarDialog({
                        name: $versionName.text(),
                        url: $versionName.attr("href"),
                        totalCount: totalCount,
                        entries: entries
                    }));
                    showPopup();
                    versionInlineDialog.refresh();

                    return false;
                },
                {
                    gravity: 'w',
                    autoWidth: true,
                    onHover: true,
                    hideDelay: 500
                }
            );
            versionInlineDialog.on("click", "a", function () {
                instance.trigger("progressDialogClicked");
            });
        },

        _closeDropdown: function() {
            if (this.ui.detailsButton.hasClass("active")) {
                this.ui.detailsButton.trigger('aui-button-invoke');
            }
        }
    });

    return ItemView;


    function getErrorMessageFromXHR(xhr) {
        var msg;
        if (xhr.readyState === 404) {
            msg = AJS.I18n.getText("project.page.release.not.found");
        } else {
            msg = ajaxUtil.getErrorMessageFromXHR(xhr);
        }

        return AJS.escapeHtml(msg);
    }
});