define("jira/projects/page/release/view/filter", [
    "jira/projects/libs/marionette",
    "underscore",
    "jira/projects/release/filter-model"
], function(
    Marionette,
    _,
    FilterModel
) {
    "use strict";
    var Templates = JIRA.Projects.Release.Templates;
    var trace = function() {
        return JIRA.trace.apply(JIRA, arguments);
    };

    var CLASSES = {
        CONTAINS_OPENED: "version-filter-search-opened",
        STATUS_PRESSED: "version-filter-status-pressed"
    };

    return Marionette.ItemView.extend({
        template: Templates.filter,
        ui: {
            "form": "form",
            "unreleased": ".version-filter-status-unreleased",
            "released": ".version-filter-status-released",
            "archived": ".version-filter-status-archived",
            "contains": ".text",
            "quickFilter": ".version-filter-status-text",
            "searchContainer": ".version-filter-search-container"
        },
        events: {
            // Prevent the form from being submitted when user hits enter
            "submit @ui.form": function formSubmit(e) {
                e.preventDefault();
            },
            "input @ui.contains": "inputContains",
            "change @ui.contains": "inputContains",
            "click @ui.released": "clickStatusReleased",
            "click @ui.unreleased": "clickStatusUnreleased",
            "click @ui.archived": "clickStatusArchived",
            "focus @ui.contains": function() {
                this.ui.searchContainer.addClass(CLASSES.CONTAINS_OPENED);
            },
            "blur @ui.contains": function() {
                if (this._isContainsEmpty()) {
                    this.ui.searchContainer.removeClass(CLASSES.CONTAINS_OPENED);
                }
            }
        },
        modelEvents: {
            "change:status": function onStatusChange() {
                var isReleasedPressed = this._getReleased();
                var isUnreleasedPressed = this._getUnreleased();

                var hasReleasedPressedClass = this.ui.released.hasClass(CLASSES.STATUS_PRESSED);
                var hasUnreleasedPressedClass = this.ui.unreleased.hasClass(CLASSES.STATUS_PRESSED);

                if (isReleasedPressed && !hasReleasedPressedClass) {
                    this.ui.released.addClass(CLASSES.STATUS_PRESSED);
                }

                if (!isReleasedPressed && hasReleasedPressedClass) {
                    this.ui.released.removeClass(CLASSES.STATUS_PRESSED);
                }

                if (isUnreleasedPressed && !hasUnreleasedPressedClass) {
                    this.ui.unreleased.addClass(CLASSES.STATUS_PRESSED);
                }

                if (!isUnreleasedPressed && hasUnreleasedPressedClass) {
                    this.ui.unreleased.removeClass(CLASSES.STATUS_PRESSED);
                }

                this.ui.archived.toggleClass(CLASSES.STATUS_PRESSED, this._getArchived());

                trace("project.release.status.changed");
            }
        },
        onRender: function() {
            this.ui.quickFilter.tooltip({
                gravity: "w"
            });
        },
        clickStatusReleased: function (e) {
            this._clickStatus(e, !this._getReleased(), this._getUnreleased(), this._getArchived());
        },
        clickStatusUnreleased: function (e) {
            this._clickStatus(e, this._getReleased(), !this._getUnreleased(), this._getArchived());
        },
        clickStatusArchived: function (e) {
            this._clickStatus(e, this._getReleased(), this._getUnreleased(), !this._getArchived());
        },
        _clickStatus: function (e, released, unreleased, archived) {
            e.preventDefault();
            this.model.set("status", this._getCurrentStatus(released, unreleased, archived));
        },
        _isContainsEmpty: function() {
            return _.isEmpty(this.ui.contains.val());
        },
        searchContainerResize: function() {
            if (!this._isContainsEmpty()) {
                this.ui.searchContainer.addClass(CLASSES.CONTAINS_OPENED);
            } else {
                if (!this.ui.contains.is(":focus")) {
                    this.ui.searchContainer.removeClass(CLASSES.CONTAINS_OPENED);
                }
            }
        },
        _inputContains: _.debounce(function inputContains() {
            var filter = this.ui.contains.val();
            this.model.set("contains", filter);
        }, 100),
        inputContains: function inputContains() {
            this.searchContainerResize();
            this._inputContains();
        },
        _getReleased: function() {
            var status = this.model.get("status");
            return status === FilterModel.FILTER_ALL || this.model.containsStatus('released');
        },
        _getUnreleased: function() {
            var status = this.model.get("status");
            return status === FilterModel.FILTER_ALL || this.model.containsStatus('unreleased');
        },
        _getArchived: function () {
            var status = this.model.get("status");
            return status === FilterModel.FILTER_ALL || this.model.containsStatus('archived');
        },
        _getCurrentStatus: function getCurrentStatus(released, unreleased, archived) {
            if (released && unreleased && archived) {
                return FilterModel.FILTER_ALL;
            } else if (!released && !unreleased && !archived) {
                return FilterModel.FILTER_NONE;
            }
            var statuses = [];
            if (released) {
                statuses.push("released");
            }
            if (unreleased) {
                statuses.push("unreleased");
            }
            if (archived) {
                statuses.push("archived");
            }
            return statuses.join("-");
        },

        serializeData: function () {
            var that = this;
            return {
                contains: this.model.get("contains"),
                isReleasedPressed: that._getReleased(),
                isUnreleasedPressed: that._getUnreleased(),
                isArchivedPressed: that._getArchived(),
                showArchived: that.options.isProjectAdmin
            };
        }
    });
});