define("jira/projects/components/submitview", [
    "jquery",
    "jira/util/events",
    "jira/util/events/types",
    "jira/ajs/select/single-select",
    "jira/projects/components/component-form-view"
], function(
    $,
    Events,
    Types,
    SingleSelect,
    ComponentFormView
) {
    var EMPTY_ERROR = new Object();

    return ComponentFormView.extend({
        template: JIRA.Projects.Components.Templates.addComponent,

        ui: {
            "form": "form",
            // inputs
            "componentLeadInput": ".components-add__lead input",
            "componentLeadSelect": ".components-add__lead select",
            "description": ".components-add__description input",
            "componentName": ".components-add__name input",
            // errors
            "err_componentName": ".components-add__name .error-container",
            "err_assigneeTypeInput": '.components-add__assignee .error-container',
            // buttons
            "addButton": ".components-add__confirm button"
        },

        events: {
            "submit @ui.form": "_onSubmit",
            "change @ui.assigneeTypeSelect": "_validateForm",
            "change @ui.assigneeTypeInput": "_validateForm",
            "blur @ui.assigneeTypeInput": "_validateForm",
            "input @ui.componentName": "_validateForm"
        },

        validation: {
            "componentName": ["_uniqueName", "_constraint", "_empty"],
            "assigneeTypeInput": ["_empty"]
        },

        onRender: function() {
            // we have to initialize user picker - it is loading on NEW_CONTENT_ADDED,
            // so we have to trigger this event
            Events.trigger(Types.NEW_CONTENT_ADDED, [this.ui.form]);

            this.wrapAndRebindElements();
            this.assigneeTypeSingleSelect.clear();

            // placeholders are not supported by SingleSelect
            this.ui.componentLeadInput.attr("placeholder", AJS.I18n.getText('project.page.components.add.lead.placeholder'));
            this.ui.assigneeTypeInput.attr("placeholder", AJS.I18n.getText('project.page.components.add.default.assignee.placeholder'));

            this._validateForm();
        },

        serializeData: function () {
            return {
                leadName: this.options.displayableProjectLead ,
                isDefaultAssigneeProjectLead: this.options.isDefaultAssigneeProjectLead
            }
        },

        _validateForm: function (event) {
            var allErrors = _.flatten(_.map(this.validation, function(fns, fieldName) {
                var value = this.ui[fieldName].val();
                var errors = fns.map(function(fn) {
                    return this[fn](value);
                }, this).filter(_.identity);

                var messageErrors = errors.filter(function(error) {
                    return error !== EMPTY_ERROR;
                });
                this.ui["err_" + fieldName].html(errors.length > 0 ? messageErrors.join(', ') : '').toggleClass('error', messageErrors.length > 0);

                return errors;
            }, this));

            if (allErrors.length > 0) {
                this.ui.addButton.attr("disabled", "disabled");
            } else if (this.ui.addButton.is(':disabled')) {
                this.ui.addButton.removeAttr("disabled");
                // focus on add button, otherwise browser would skip to next element even though button is already enabled
                if (event && this.ui.assigneeTypeInput.is(event.target)) {
                    this.ui.addButton.focus();
                }
            }

            return allErrors;
        },

        _onSubmit: function(e) {
            e.preventDefault();

            if (this._validateForm(e).length > 0) {
                return;
            }

            this.model.unset("id");
            this.model.set(this._getFormData());
            this.trigger("save", this.model);
            this.model.unset("name");
            this.model.unset("description");

            if (!this.isDestroyed) {
                this.ui.componentName.val('');
                this.ui.description.val('');

                this.ui.componentName.focus();

                this._validateForm();
            }
        },

        _getFormData: function() {
            return this.ui.form.serializeArray().reduce(function(r, param) {
                r[param.name] = param.value;
                return r;
            }, {
                project: this.options.project,
                leadDisplayName: this.ui.componentLeadSelect.find('option:selected').text()
            });
        },

        // validation functions
        _uniqueName: function(val) {
            if (this.options.pageableItems.originalCollection.some(function(itemData) {
                    return (itemData.get("name") || '').toLowerCase() === (val || '').toLowerCase();
                })) {
                return AJS.I18n.getText("project.page.components.add.error.already.exists");
            }
        },

        _constraint: function(val) {
            if (val && val.length > 255) {
                return AJS.I18n.getText("project.page.components.add.limit.exceeded");
            }
        },

        _empty: function(val) {
            if (!val || val.trim().length === 0) {
                return EMPTY_ERROR;
            }
        }
    });
});