AJS.test.require(['com.atlassian.jira.jira-projects-plugin:components-page-assets'], function(){
    "use strict";

    var Marionette = require("jira/projects/libs/marionette");
    var Backbone = require("backbone");
    var $ = require("jquery");
    var SubmitModelStates = require("jira/projects/abstract-model/submit-model-states");

    module('SubmitController', {
        setup: function () {
            this.context = AJS.test.mockableModuleContext();
            this.sandbox = sinon.sandbox.create();
            this.sandbox.useFakeServer();
            this.clock = sinon.useFakeTimers();

            this.modelSave = $.Deferred();
            this.modelDetails = $.Deferred();
            this.sandbox.stub(Backbone.Model.prototype, "save").returns(this.modelSave);
            Backbone.Model.prototype.detailsSync = this.sandbox.stub().returns(this.modelDetails);

            this.pageableItems = {
                addElement: this.sandbox.stub()
            };

            this.featureManager = {
                isFeatureEnabled: this.sandbox.stub()
            };
            this.flags = {
                showErrorMsg: this.sandbox.stub()
            };

            this.context.mock("jira/featureflags/feature-manager", this.featureManager);
            this.context.mock("jira/projects/components/submitview", Marionette.ItemView);
            this.context.mock("jira/projects/components/submitmodel", Backbone.Model);
            this.context.mock("jira/flag", this.flags);
            this.context.mock("jira/ajs/ajax/ajax-util", this.mockAjaxUtil());

            this.submitController = this.context.require("jira/projects/components/submitcontroller");
        },
        teardown: function () {
            this.sandbox.restore();
        },

        mockAjaxUtil: function () {
            return {
                getErrorMessageFromXHR: function(){
                    return "";
                }
            }
        },

        newSubmitView: function () {
            return new this.submitController({
                pageableItems: this.pageableItems
            })
        },

        mockedModel: function() {
            return new Backbone.Model({
                name: "some name",
                leadUserName: "some leadUserName",
                description: "some description"
            });
        }
    });

    test("SubmitView should be displayed if flag is enabled and user is admin", function () {
        this.featureManager.isFeatureEnabled.withArgs("jira.sidebar.components.managecomponents").returns(true);

        var SubmitController = new this.submitController({
            isProjectAdmin: true
        });

        ok(SubmitController.displaySubmitView());
    });

    test("SubmitView should be hidden if flag is disabled", function () {
        this.featureManager.isFeatureEnabled.withArgs("jira.sidebar.components.managecomponents").returns(false);

        var SubmitController = new this.submitController({
            isProjectAdmin: true
        });

        ok(!SubmitController.displaySubmitView());
    });

    test("SubmitView should be hidden if user is not admin", function () {
        this.featureManager.isFeatureEnabled.withArgs("jira.sidebar.components.managecomponents").returns(true);

        var SubmitController = new this.submitController({
            isProjectAdmin: false
        });

        ok(!SubmitController.displaySubmitView());
    });

    test("New element is added to pageableItems on view save event", function () {
        var submitController = this.newSubmitView();
        var view = submitController.showView();

        view.trigger("save", this.mockedModel());

        ok(this.pageableItems.addElement.calledOnce);
    });

    test("componentAdd should listen to model retry-add event", function () {
        var submitController = this.newSubmitView();
        submitController.listenTo = sinon.spy();

        submitController.componentAdd(this.mockedModel());

        sinon.assert.calledOnce(submitController.listenTo);
        equal(submitController.listenTo.args[0][1], "retry-add");
    });

    test("retry-add event after componentAdd should trigger save of model as for the normal paths", function () {
        var submitController = this.newSubmitView();
        submitController._componentAdd = sinon.spy();

        var model = submitController.componentAdd(this.mockedModel());
        model.trigger("retry-add");

        sinon.assert.calledTwice(submitController._componentAdd);
        ok(submitController._componentAdd.getCall(0).calledWith(model, true));
        ok(submitController._componentAdd.getCall(1).calledWith(model, false));
    });

    test("componentAdded should trigger change model state to SubmitModelStates.READY from SubmitModelStates.SUCCESSFUL", function () {
        var submitController = this.newSubmitView();

        var model = submitController.componentAdd(this.mockedModel());
        equal(model.get('state'), SubmitModelStates.IN_PROGRESS);
        this.modelSave.resolve();
        model.set("state", SubmitModelStates.SUCCESSFUL);
        this.modelDetails.resolve();

        this.clock.tick(6000);

        equal(model.get('state'), SubmitModelStates.READY);
    });

    test("componentAdded should NOT trigger change model state to SubmitModelStates.READY from other state than SubmitModelStates.SUCCESSFUL", function () {
        var submitController = this.newSubmitView();

        var model = submitController.componentAdd(this.mockedModel());
        equal(model.get('state'), SubmitModelStates.IN_PROGRESS);
        this.modelSave.resolve();
        this.modelDetails.resolve();
        //simulate user deleting right after component adding
        this.clock.tick(500);
        model.set("state", SubmitModelStates.ERROR_DELETE);
        this.clock.tick(6000);

        equal(model.get('state'), SubmitModelStates.ERROR_DELETE);
    });
});

