AJS.test.require(['com.atlassian.jira.jira-projects-plugin:components-page-assets'], function(){
    "use strict";

    var SubmitModelStates = require("jira/projects/abstract-model/submit-model-states");

    function generateComponent(name, description) {
        return {
            id: name,
            name: "Component " + name,
            description: description || ("Component " + name + " description"),
            componentUrl: "/jira/browse/X/component/" + name,
            issuesCount: Math.floor(Math.random() * 10),
            issuesUrl: "/jira/secure/IssueNavigator.jspa?reset=true&jqlQuery=project+%3D+X+AND+component+%3D+" + name,
            leadUserDisplayName: "admin",
            leadUserProfileLinkHtml: "admin link html"
        }
    }

    function generateComponentsArray(length) {
        var components = [];
        for (var i = 1; i <= length; i++) {
            components.push(generateComponent(i));
        }
        return components;
    }

    var App = require("jira/projects/components/app");

    module('App', {
        setup: function () {
            sinon.stub(JIRA.Projects.AbstractList.NavigationUtils, "navigate");
            this.container = jQuery(
                    '<div id="components-page">' +
                    '    <div id="filter-container"></div>' +
                    '    <div id="submit-container"></div>' +
                    '    <div id="content-container"></div>' +
                    '    <div id="pagination-container"></div>' +
                    '</div>'
            );
            this.options = {
                container: this.container
            };
        },
        teardown: function () {
            JIRA.Projects.AbstractList.NavigationUtils.navigate.restore();
        },
        triggerContainsChange: function () {
            this.sandbox.useFakeTimers();
            this.container.find('#component-filter-text').trigger("input");
            this.sandbox.clock.tick(300);
        },
        stubData: function stubData(data) {
            var wrmDataClaimStub = this.stub(WRM.data, "claim");
            wrmDataClaimStub.withArgs("com.atlassian.jira.projects.page.components:components").returns(data);
            wrmDataClaimStub.withArgs("com.atlassian.jira.projects.page.components:learnMoreUrl").returns("learnMoreUrl");
            wrmDataClaimStub.withArgs("com.atlassian.jira.projects.page.components:assigneeSelectParams").returns({
                leadDisplayableName: 'Lead Displayable Name',
                isDefaultAssigneeProjectLead: false
            });
        },
        mockedModel: function() {
            return new Backbone.Model({
                name: "some name",
                leadUserName: "some leadUserName",
                description: "some description"
            });
        }
    });

    test('Should filter when changing criteria', function () {
        this.stubData(generateComponentsArray(20));
        App.start(this.options);

        this.container.find('#component-filter-text').val("Component 4");
        this.triggerContainsChange();

        equal(this.container.find('#components-table tbody.items tr:first td:first').text(), "Component 4");
        equal(this.container.find('#components-table tbody.items tr').length, 1);
    });

    test('Should display message when no components match filter criteria', function () {
        this.stubData(generateComponentsArray(10));
        App.start(this.options);

        this.container.find('#component-filter-text').val("nothing");
        this.triggerContainsChange();

        equal(this.container.find('#components-table tbody.items tr').length, 1);
        equal(this.container.find('#components-table .no-results').length, 1);
    });

    test('Should break a very long component name', function () {
        var longName = "A veryveryveryveryveryvery veryveryveryveryveryveryveryveryveryveryveryveryveryveryveryveryveryveryveryveryveryveryveryveryver yveryveryveryvery veryveryveryveryveryveryveryveryveryvery longname";
        this.stubData([generateComponent(longName, "short description")]);
        App.start(this.options);

        // fix headers to some actual translation
        this.container.find('#components-table tbody.header th, #components-table .components-table__issues-count').each(function(idx, el) {
            el.innerHTML = {
                "common.concepts.component": "Component",
                "common.concepts.issues": "Issues",
                "common.concepts.lead": "Lead",
                "common.concepts.description": "Description",
                "common.concepts.issues.with.count": "0 Issues"
            }[el.innerHTML];
        });

        // otherwise width/height is 0
        this.container.appendTo(jQuery('#qunit-fixture'));

        var bodyWidth = this.container.find('#components-table tbody.items').width();
        var description = this.container.find('#components-table tbody.items').find('.components-table__description').width();

        var epsilon = (description / bodyWidth) * 100 - 60;
        equal(Math.abs(epsilon) < 11, true, "Epsilon should be below 11, actual: " + epsilon);
    });

    test('Should display no components item view when there are no components defined', function () {
        this.stubData(generateComponentsArray(0));
        App.start(this.options);

        equal(this.container.find('#components-table').length, 0);
        equal(this.container.find('div.no-components').length, 1);
    });

    test('Should not display searcher when there are no components defined', function () {
        this.stubData(generateComponentsArray(0));
        App.start(this.options);

        equal(this.container.find('#component-filter-text').length, 0);
    });

    test('Should temporarily display success icon for newly added component', function () {
        this.modelSave = jQuery.Deferred();
        this.modelDetails = jQuery.Deferred();
        this.sandbox.stub(Backbone.Model.prototype, "save").returns(this.modelSave);
        Backbone.Model.prototype.detailsSync = this.sandbox.stub().returns(this.modelDetails);

        this.stubData(generateComponentsArray(0));
        App.start(this.options);

        this.container.appendTo(jQuery('#qunit-fixture'));

        var model = App.submitController.componentAdd(this.mockedModel());

        model.set("state", SubmitModelStates.SUCCESSFUL);
        equal(this.container.find('.operation-success').length, 1, "We have successfully added component.");

        model.set("state", SubmitModelStates.READY);
        equal(this.container.find('.operation-success').length, 0, "And it's disappeared.");
    });
});

