define("jira/projects/test-utils/projectissuenavigatorworldmocker", ['require'], function (require) {
    "use strict";

    var wrmContextPath = require('wrm/context-path')();
    var FakeServer = require('jira/projects/test-utils/fakeserver');
    var URLHelper = require('jira/projectissuenavigator/services/urlhelper');
    var FocusShifter = require('jira/projectissuenavigator/services/focusshifter');
    var Metrics = require('jira/projectissuenavigator/services/metrics');
    var JIRADialog = require('jira/dialog/dialog');
    var JIRAFlag = require('jira/flag');
    var ProjectIssueNavigator = require('jira/projectissuenavigator');
    var Backbone = require('jira/projectissuenavigator/libs/backbone');
    var _ = require('jira/projectissuenavigator/libs/underscore');
    var WrmData = require('wrm/data');
    var jQuery = require('jquery');

    var World = function(sandbox) {
        this.data = {
            issues: [],
            state: {},
            el: jQuery("<div></div>")
        };
        this.sandbox = sandbox;
        this.clock = sandbox.useFakeTimers();

        sandbox.stub(URLHelper, "extractStateFromCurrentUrl");
        sandbox.stub(URLHelper, 'isBrowseUrl');
        sandbox.stub(WrmData, 'claim');
        sandbox.stub(JIRAFlag, "showErrorMsg");
        sandbox.stub(FocusShifter.prototype, "show");
        sandbox.stub(Metrics);
    };

    World.prototype = {
        withData: function(data) {
            this.data.state.issue = data.issue;
            this.data.state.filter = data.filter;
            this.data.issues = data.issues.map(function (issue) {
                return jQuery.extend(true, {}, issue);
            });

            if (data.editableFields) {
                var originalIssue = _.findWhere(data.issues, {key: data.issue});
                originalIssue.isEditable = true;
                originalIssue.editableFields = data.editableFields;
            }
            return this;
        },
        usingBrowseUrl: function(flag) {
            URLHelper.isBrowseUrl.returns(flag === undefined ? true : Boolean(flag));
            return this;
        },
        build: function () {
            var issues = this.data.issues;
            var state = this.data.state;
            var sandbox = this.sandbox;

            // Mock the server
            this.server = FakeServer.create(sandbox, issues);

            // Mock the URL manager
            URLHelper.extractStateFromCurrentUrl.returns(state);

            // Don't start the history management, it will mess up with our QUnit page
            sandbox.stub(Backbone.history, "start", function () {
                Backbone.history.loadUrl("/issues");
            });
            Backbone.history.root = wrmContextPath + "/issues";

            // Don't show flags
            this.JIRAFlag = JIRAFlag;

            // Don't show the FocusShifter
            this.FocusShifter = FocusShifter;

            // Don't trigger any metrics
            this.Metrics = Metrics;

            // Don't show dialogs. Instead, save the markup to this.dialogEl
            var module = this;
            sandbox.stub(JIRADialog.prototype, "show", function() {
                this.options.content(function(el) {
                    module.dialogEl = el;
                });
            });

            WrmData.claim.withArgs("com.atlassian.jira.jira-projects-issue-navigator:generic-filters").returns([{
                    id: 'allissues',
                    jql: 'project = "{0}" ORDER BY {1}',
                    defaultOrderby: 'priority DESC, updated DESC',
                    label: AJS.I18n.getText("common.filters.all"),
                    requiresUser: false,
                    fields: [],
                    supportsInlineIssueCreate: true
                }]);

            var projectIssueNavigator = new ProjectIssueNavigator();
            projectIssueNavigator.start(this.data.el);
            this.projectIssueNavigator = projectIssueNavigator;
            return this;
        },
        restore: function() {
            this.sandbox.restore();
            if (this.projectIssueNavigator) {
                this.projectIssueNavigator.destroy();
                delete this.projectIssueNavigator;
            }
        },
        waitForSearch: function () {
            this.server.respond();
            return this;
        },
        waitForPageLoad: function () {
            this.server.respond();
            return this;
        },
        waitForEditorToStartLoading: function () {
            // The editor has some throttled load, make sure we advance the clock
            this.clock.tick(200);
            return this;
        },
        waitForEditor: function () {
            this.waitForEditorToStartLoading();
            this.server.respond();
            return this;
        },
        withUpdatedIssue: function(issueKey, issue) {
            var oldIssue = _.findWhere(this.data.issues, {key: issueKey});
            _.extend(oldIssue, issue);
            return this;
        },
        waitForEditorFields: function () {
            this.server.respond();
            return this;
        },
        waitForEditorSave: function () {
            this.server.respond();
            // Some of the success events/callbacks are async (facepalm), so we advance the clock to make sure
            // everything has a change to finish.
            this.clock.tick(200);
            return this;
        },
        goToIssueView: function() {
            jQuery(this.projectIssueNavigator.issueSearch.el).find('.expand').click();
            return this;
        },
        waitForIssueSearch: function() {
            return this
                .waitForSearch()
                .waitForPageLoad()
                .waitForEditor();
        }
    };

    return World;
});
