define("jira/projectissuenavigator/services/urlhelper", ['require'], function (require) {
    "use strict";

    var wrmContextPath = require('wrm/context-path')();
    var Browser = require('jira/projectissuenavigator/services/browser');
    var URI = require('jira/projectissuenavigator/libs/uri');

    function updateStateWithFilter(uri, state) {
        var filter = uri.search(true).filter;
        if (filter) state.filter = filter;
        else state.filter = null;
    }

    function updateStateWithOrderBy(uri, state) {
        var orderby = uri.search(true).orderby;
        if (orderby) state.orderby = orderby;
        else state.orderby = null;
    }

    function updateStateWithIssue(uri, state) {
        var issue = uri.filename();
        if (issue) state.issue = issue;
    }

    function updateStateWithStartIndex(uri, state) {
        var startIndex = uri.search(true).startIndex;
        if (startIndex) state.startIndex = startIndex;
        else state.startIndex = null;
    }

    /**
     * Generates the application part of the URL, if needed.
     *
     * For a URL like '/projects/DEMO/issues/DEMO-1?filter=a', the root part is '/DEMO-1?filter=a'.
     *
     * @returns {String} The application part of the URL
     */
    function generateApplicationURLPart(options) {
        var uri = new URI("");

        if (options.issue) {
            uri.filename(options.issue);
        }

        if (options.filter) {
            uri.setSearch("filter", options.filter);
        } else {
            uri.removeSearch("filter");
        }

        if (options.orderby) {
            uri.setSearch("orderby", options.orderby);
        } else {
            uri.removeSearch("orderby");
        }

        return uri.toString();
    }

    /**
     * Generates the root part of the URL, if needed.
     *
     * For a URL like '/projects/DEMO/issues/DEMO-1', the root part is '/projects/DEMO/issues'.
     *
     * @param {string} project The project to use in the URL.
     * @returns {string} The root part of the URL
     */
    function generateRootURLPart(project) {
        var uri = new URI("");

        if (project) {
            uri.path(wrmContextPath + "/projects/" + project + "/issues");
        }

        return uri.toString();
    }


    return {
        extractProjectKeyFromCurrentUrl: function() {
            var encodedPath = new URI(Browser.getCurrentUrl()).pathname();

            var match;

            if(this.isBrowseUrl(encodedPath)) {
                // Subtract project key from issue key
                match = encodedPath.match(/\/([A-Z0-9]+)\-[0-9]+$/);
            } else {
                // Look for /projects/<something>/issues
                match = encodedPath.match(/\/projects\/([^\/]+?)\/issues/);
            }

            if (match) {
                return match[1];
            } else {
                return null;
            }
        },

        extractIssueKeyFromCurrentUrl: function() {
            var encodedPath = new URI(Browser.getCurrentUrl()).pathname();

            var match = encodedPath.match(/\/([A-Z0-9]+\-[0-9]+)$/);
            if (match) {
                return match[1];
            } else {
                return null;
            }
        },

        generateUrl: function(options) {
            var root = generateRootURLPart(options.project);
            var app = generateApplicationURLPart(options);
            var glue = Browser.hasPushState() ? "/" : "#";

            return root + glue + app;
        },

        generateBrowseUrl: function(options) {
            var root = new URI("");
            root.path(wrmContextPath + "/browse/");

            var app = generateApplicationURLPart(options);

            return root.toString() + app;
        },

        extractStateFromCurrentUrl: function() {
            var uri = new URI(this.getApplicationUrl());

            // If this is a hashed URL, everything comes in the fragment
            if (this.isHashUrl()) {
                uri = new URI(uri.fragment());
            }

            var state = {};
            updateStateWithFilter(uri, state);
            updateStateWithOrderBy(uri, state);
            updateStateWithIssue(uri, state);
            updateStateWithStartIndex(uri, state);
            state.project = this.extractProjectKeyFromCurrentUrl();
            return state;
        },

        generateRootUrl: function(project) {
            project = project || this.extractProjectKeyFromCurrentUrl();
            return generateRootURLPart(project);
        },

        getApplicationUrl: function() {
            var fullUrl = Browser.getCurrentUrl();

            var regExp = new RegExp(this.isBrowseUrl(fullUrl) ?
                "^.*/browse/" :
                "^.*" + this.generateRootUrl() + ""
            );

            return fullUrl.replace(regExp, "");
        },

        isBrowseUrl: function(url) {
            url = url || Browser.getCurrentUrl();
            return url.indexOf("/browse/") > -1;
        },

        isHashUrl: function() {
            var appUrl = this.getApplicationUrl();
            var match = appUrl.match(/^#/);
            return !!match;
        },

        getAnchor: function () {
            return new URI(Browser.getCurrentUrl()).fragment();
        }
    };
});
