AJS.test.require("com.atlassian.jira.jira-projects-issue-navigator:application-test", function() {
    "use strict";

    require([
        "jira/projectissuenavigator/services/browser",
        "jira/projectissuenavigator/services/urlhelper"
    ], function(
        Browser,
        URLHelper
    ) {

        module("jira/projectissuenavigator/services/urlhelper", {

            setup: function() {
                sinon.stub(Browser, "getCurrentUrl");
                sinon.stub(Browser, "hasPushState");
            },

            teardown: function() {
                Browser.getCurrentUrl.restore();
                Browser.hasPushState.restore();
            },

            assertProjectKeyFromUrl: function(url, expectedKey) {
                Browser.getCurrentUrl.returns(url);
                var actualKey = URLHelper.extractProjectKeyFromCurrentUrl();
                strictEqual(actualKey, expectedKey, "Key from " + url);
            },

            assertStateFromUrl: function(url, expectedState) {
                url = AJS.contextPath() + url;
                Browser.getCurrentUrl.returns(url);
                var actualKey = URLHelper.extractStateFromCurrentUrl();
                deepEqual(actualKey, expectedState, "State from " + url);
            },

            assertGeneratedUrl: function(options, expecedUrl) {
                var actualUrl = URLHelper.generateUrl(options);
                equal(actualUrl, expecedUrl);
            }

        });

        test("It extracts the project key from the URL", function() {
            this.assertProjectKeyFromUrl("/projects/DEMO/issues", "DEMO");
            this.assertProjectKeyFromUrl("1/2/3/projects/DEMO/issues", "DEMO");
            this.assertProjectKeyFromUrl("/projects/DEMO/issues/DEMO-123", "DEMO");

            this.assertProjectKeyFromUrl("/projects/issues", null); // Missing project
            this.assertProjectKeyFromUrl("projects/DEMO/issues", null); // Missing starting slash
            this.assertProjectKeyFromUrl("projects/DEMO/test/issues", null); // Extra path
            this.assertProjectKeyFromUrl("/projects/DEMO/test", null); // Missing /issues
            this.assertProjectKeyFromUrl("/projects//test", null); // Empty project
        });

        test("It extracts the state from the URL", function() {
            this.assertStateFromUrl("/projects/DEMO/issues", {
                project: "DEMO",
                filter: null,
                orderby: null,
                startIndex: null
            });
            this.assertStateFromUrl("/projects/DEMO/issues/", {
                project: "DEMO",
                filter: null,
                orderby: null,
                startIndex: null
            });
            this.assertStateFromUrl("/projects/DEMO/issues#", {
                project: "DEMO",
                filter: null,
                orderby: null,
                startIndex: null
            });
            this.assertStateFromUrl("/projects/DEMO/issues/DEMO-123", {
                project: "DEMO",
                issue: "DEMO-123",
                filter: null,
                orderby: null,
                startIndex: null
            });
            this.assertStateFromUrl("/projects/DEMO/issues#DEMO-123", {
                project: "DEMO",
                issue: "DEMO-123",
                filter: null,
                orderby: null,
                startIndex: null
            });
            this.assertStateFromUrl("/projects/DEMO/issues?filter=myfilter", {
                project: "DEMO",
                filter: "myfilter",
                orderby: null,
                startIndex: null
            });
            this.assertStateFromUrl("/projects/DEMO/issues#?filter=myfilter", {
                project: "DEMO",
                filter: "myfilter",
                orderby: null,
                startIndex: null
            });
            this.assertStateFromUrl("/projects/DEMO/issues?orderby=key+DESC", {
                project: "DEMO",
                orderby: "key DESC",
                filter: null,
                startIndex: null
            });
            this.assertStateFromUrl("/projects/DEMO/issues#?orderby=key+DESC", {
                project: "DEMO",
                orderby: "key DESC",
                filter: null,
                startIndex: null
            });
            this.assertStateFromUrl("/projects/DEMO/issues?startIndex=3", {
                project: "DEMO",
                startIndex: "3",
                filter: null,
                orderby: null
            });
            this.assertStateFromUrl("/projects/DEMO/issues#?startIndex=3", {
                project: "DEMO",
                startIndex: "3",
                filter: null,
                orderby: null
            });
            this.assertStateFromUrl("/projects/DEMO/issues?test=1&test=2", {
                project: "DEMO",
                filter: null,
                orderby: null,
                startIndex: null
            });
            this.assertStateFromUrl("/projects/DEMO/issues#?test=1&test=2", {
                project: "DEMO",
                filter: null,
                orderby: null,
                startIndex: null
            });
            this.assertStateFromUrl("/projects/DEMO/issues/DEMO-123?test=1&test=2&filter=myFilter&orderby=key+DESC%2C+priority+ASC&startIndex=3", {
                project: "DEMO",
                issue: "DEMO-123",
                filter: "myFilter",
                orderby: "key DESC, priority ASC",
                startIndex: "3"
            });
            this.assertStateFromUrl("/projects/DEMO/issues#DEMO-123?test=1&test=2&filter=myFilter&orderby=key+DESC%2C+priority+ASC&startIndex=3", {
                project: "DEMO",
                issue: "DEMO-123",
                filter: "myFilter",
                orderby: "key DESC, priority ASC",
                startIndex: "3"
            });
            this.assertStateFromUrl("/browse/DEMO-123", {
                project: "DEMO",
                issue: "DEMO-123",
                filter: null,
                orderby: null,
                startIndex: null
            });
            this.assertStateFromUrl("/browse/DEMO-123?test=1&test=2&filter=myFilter&orderby=key+DESC%2C+priority+ASC&startIndex=3", {
                project: "DEMO",
                issue: "DEMO-123",
                filter: "myFilter",
                orderby: "key DESC, priority ASC",
                startIndex: "3"
            });

        });

        test("It generates the URL root with the default project", function() {
            Browser.getCurrentUrl.returns("/projects/BULK/issues");
            equal(AJS.contextPath() + "/projects/BULK/issues", URLHelper.generateRootUrl());
        });

        test("It generates the URL root for any custom project", function() {
            equal(AJS.contextPath() + "/projects/MYPROJECT/issues", URLHelper.generateRootUrl("MYPROJECT"));
        });

        test("It extracts the application URL", function() {
            Browser.hasPushState.returns(true);

            Browser.getCurrentUrl.returns(AJS.contextPath() + "/projects/DEMO/issues/a/b/c?a=b#c");
            var url = URLHelper.getApplicationUrl();
            equal(url, "/a/b/c?a=b#c");
        });

        test("It generate URLs for pushState browsers", function() {
            Browser.hasPushState.returns(true);
            this.assertGeneratedUrl({project: "DEMO"}, AJS.contextPath() + "/projects/DEMO/issues/");
            this.assertGeneratedUrl({project: "DEMO", issue: "DEMO-123"}, AJS.contextPath() + "/projects/DEMO/issues/DEMO-123");
            this.assertGeneratedUrl({issue: "DEMO-123"}, "/DEMO-123");
            this.assertGeneratedUrl({issue: "DEMO-123", filter: 'myfilter'}, "/DEMO-123?filter=myfilter");
            this.assertGeneratedUrl({orderby: "key DESC"}, "/?orderby=key+DESC");
            this.assertGeneratedUrl({orderby: "key DESC, priority ASC"}, "/?orderby=key+DESC%2C+priority+ASC");
        });

        test("It generate URLs for non-pushState browsers", function() {
            Browser.hasPushState.returns(false);
            this.assertGeneratedUrl({project: "DEMO"}, AJS.contextPath() + "/projects/DEMO/issues#");
            this.assertGeneratedUrl({project: "DEMO", issue: "DEMO-123"}, AJS.contextPath() + "/projects/DEMO/issues#DEMO-123");
            this.assertGeneratedUrl({issue: "DEMO-123"}, "#DEMO-123");
            this.assertGeneratedUrl({issue: "DEMO-123", filter: 'myfilter'}, "#DEMO-123?filter=myfilter");
            this.assertGeneratedUrl({orderby: "key DESC"}, "#?orderby=key+DESC");
            this.assertGeneratedUrl({orderby: "key DESC, priority ASC"}, "#?orderby=key+DESC%2C+priority+ASC");
        });

        test("It returns the anchor from the URL", function () {
            Browser.getCurrentUrl.returns(AJS.contextPath() + "/projects/DEMO/issues/a/b/c?a=b#c");

            equal(URLHelper.getAnchor(), 'c');
        });
    });
});
