/**
 * This module add supports for KeyboardShortcuts that are NOT supported by the regular Keyboard Shortcuts Plugin
 * (see https://developer.atlassian.com/display/JIRADEV/Keyboard+Shortcut+Plugin+Module). In particular, kb shortcuts
 * that use a key that is not a letter, like the arrows.
 *
 * The regular kb shortcuts are defined in jira-global-issue-navigator-plugin/src/main/resources/atlassian-plugin.xml.
 * Try to add new shortcuts to that XML file first, and if they are not supported for any reason, then fallback to this
 * module.
 *
 * Note that we do have a description for the arrow keys in atlassian-plugin.xml. That's because we want these keyboard
 * shortcuts to appear in the Help screen. The description is in the XML file, but the actual handling of the key press
 * is in this module.
 */
define("jira/projectissuenavigator/services/keyboardshortcuts", ["require"], function(require) {
    "use strict";

    var Browser = require("jira/projectissuenavigator/services/browser");
    var jQuery = require("jquery");
    var KeyboardShortcutsToggle = require("jira/ajs/keyboardshortcut/keyboard-shortcut-toggle");
    var Marionette = require("jira/projectissuenavigator/libs/marionette");

    function isDialogVisible() {
        return jQuery(Browser.getDocument()).find("div.aui-blanket").length > 0;
    }

    function isSupportedKey(e) {
        return (
            e.which === jQuery.ui.keyCode.LEFT ||
            e.which === jQuery.ui.keyCode.RIGHT
        );
    }

    function areKeyboardShortcutsEnabled() {
        return !KeyboardShortcutsToggle.areKeyboardShortcutsDisabled();
    }

    function onKeyDown(e) {
        if (areKeyboardShortcutsEnabled() && !isDialogVisible() && isSupportedKey(e)) {
            var target = jQuery(e.target);
            var targetIsValid = target.is(":not(:input)");

            if (target == null || targetIsValid) {
                if (e.which === jQuery.ui.keyCode.LEFT) {
                    this.trigger("focusIssueList");
                } else if (e.which === jQuery.ui.keyCode.RIGHT) {
                    this.trigger("focusIssueEditor");
                }
            }
        }
    }

    return Marionette.Object.extend({
        initialize: function() {
            jQuery(Browser.getDocument()).on('keydown', onKeyDown.bind(this));
        }
    });
});
