AJS.test.require("com.atlassian.jira.jira-projects-issue-navigator:application-test", function(){
    "use strict";

    require([], function() {
        var Analytics;

        module("jira/projectissuenavigator/services/analytics", {
            setup: function () {
                this.sandbox = sinon.sandbox.create();
                this.analyticsSpy = this.sandbox.spy();
                AJS.bind('analyticsEvent', this.analyticsSpy);

                this.context = AJS.test.mockableModuleContext();
                Analytics = this.context.require('jira/projectissuenavigator/services/analytics');
            },

            teardown: function() {
                AJS.unbind('analyticsEvent', this.analyticsSpy);
                this.sandbox.restore();
            },

            assertAnalytics: function(eventName, eventAttributes) {
                sinon.assert.calledOnce(this.analyticsSpy);

                var payload = {
                    name: eventName
                };
                if (eventAttributes) {
                    payload.data = eventAttributes;
                }

                sinon.assert.calledWithMatch(this.analyticsSpy, sinon.match.any, payload);
            }
        });

        test("Should emit analytics when a filter is selected", function(){
            Analytics.filterSelect({filterId: 'filter-id', source: 'event-source'});

            this.assertAnalytics("jira.project.sidebar.issue.filter.select", {
                filterId: 'filter-id',
                source: 'event-source'
            });
        });

        test("Should emit analytics when the user goes to the full issue navigator", function(){
            Analytics.goToFullIssueNavigator();

            this.assertAnalytics("jira.project.sidebar.issue.load.full.issue.nav");
        });

        test("Should emit analytics when the expand button is clicked", function(){
            Analytics.expandIssue({
                source: "ui-button"
            });

            this.assertAnalytics("jira.project.sidebar.issue.expand");
        });

        test("Should emit analytics when the collapse button is clicked", function(){
            Analytics.collapseIssue({
                source: "ui-button"
            });

            this.assertAnalytics("jira.project.sidebar.issue.collapse");
        });

        test("Should emit analytics when the user selects an issue in the issue list", function(){
            Analytics.issueSelectedInList({key: 'DEMO-1'});

            this.assertAnalytics("jira.project.sidebar.issue.select.issue", {
                issueKey: 'DEMO-1',
                source: "list",
                sameProject: true
            });
        });

        test("Should emit view issue analytics when an issue is loaded", function(){
            Analytics.triggerProjectNavViewIssue({
                issueId: 10000,
                issueKey: "TEST-123",
                projectId: 1,
                projectType: "business",
                projectKey: 'TEST'
            });

            this.assertAnalytics("jira.project.sidebar.issue.view.issue", {
                issueId: 10000,
                projectId: 1,
                projectType: "business",
                issueMatchesProjectContext: true
            });
        });

        test("Should emit view analytics when an issue is loaded and the issue doesn't match the project context", function(){
            Analytics.triggerProjectNavViewIssue({
                issueId: 10000,
                issueKey: "ANOTHER-123",
                projectId: 1,
                projectType: "business",
                projectKey: 'TEST'
            });

            this.assertAnalytics("jira.project.sidebar.issue.view.issue", {
                issueId: 10000,
                projectId: 1,
                projectType: "business",
                issueMatchesProjectContext: false
            });
        });

        test("issueMatchesProjectContext should be false when issueKey is null", function(){
            Analytics.triggerProjectNavViewIssue({
                issueId: 10000,
                issueKey: null,
                projectId: 1,
                projectType: "business",
                projectKey: 'TEST'
            });

            this.assertAnalytics("jira.project.sidebar.issue.view.issue", {
                issueId: 10000,
                projectId: 1,
                projectType: "business",
                issueMatchesProjectContext: false
            });
        });

        test("Should emit analytics when the user manage the filters", function(){
            Analytics.manageFilters();

            this.assertAnalytics("jira.project.sidebar.issue.filter.manage");
        });

        test("Should emit analytics when user uses paginates to the next page", function() {
            Analytics.pager('next', {current: 2, total: 4});

            this.assertAnalytics("jira.project.sidebar.issue.pager", {
                current: 2,
                total: 4,
                direction: 'next'
            });
        });

        test("Should emit analytics when user uses paginates to the previous page", function() {
            Analytics.pager('previous', {current: 1, total: 4});

            this.assertAnalytics("jira.project.sidebar.issue.pager", {
                current: 1,
                total: 4,
                direction: 'previous'
            });
        });

        test("Should emit analytics when inline edit is saved", function(){
            Analytics.editorFieldSaved({
                issueId: '123456',
                savedFieldIds: ['fieldId-1'],
                savedFieldTypes: ['fieldType-1']
            });

            this.assertAnalytics("kickass.inlineEdit", {
                issueId: '123456',
                savedFieldIds: ['fieldId-1'],
                savedFieldTypes: ['fieldType-1'],
                source: "jira.projectissuenav.global"
            });
        });

        test("Should emit analytics when inline edit is started", function(){
            Analytics.editFieldStarted({
                issueId: '123456',
                fieldId: 'fieldId-1',
                fieldType: 'fieldType-1'
            });

            this.assertAnalytics("kickass.inlineEdit.start", {
                issueId: '123456',
                fieldId: 'fieldId-1',
                fieldType: 'fieldType-1',
                source: "jira.projectissuenav.global"
            });
        });

        test("Should emit analytics when inline edit is cancelled", function(){
            Analytics.editFieldCancelled({
                issueId: '123456',
                fieldId: 'fieldId-1',
                fieldType: 'fieldType-1'
            });

            this.assertAnalytics("kickass.inlineEdit.cancel", {
                issueId: '123456',
                fieldId: 'fieldId-1',
                fieldType: 'fieldType-1',
                source: "jira.projectissuenav.global"
            });
        });
    });
});
