define("jira/projectissuenavigator/services/api", ['require'], function (require) {
    "use strict";

    var _ = require('jira/projectissuenavigator/libs/underscore');

    var NOOP = function(){};
    var DEFAULT_IMPLEMENTATION = {
        "nextIssue": NOOP,
        "previousIssue": NOOP,
        "getActiveIssueId": NOOP,
        "getActiveIssueKey": NOOP,
        "refreshActiveIssue": NOOP,
        "refreshIssue": NOOP,
        "editField": NOOP,
        "getFields": NOOP,
        "isSaving": NOOP,
        "isLoading": NOOP,
        "hasAccessibleIssue": NOOP,
        "showLoadError": NOOP,
        "waitForSavesToComplete": NOOP,
        "viewSelectedIssue": NOOP,
        "returnToSearch": NOOP,
        "focusSearch": NOOP,
        "toggleFilterPanel": NOOP,
        "isQueryValid": NOOP,
        "isFullScreenIssueVisible": NOOP,
        "switchLayouts": NOOP,
        "openFocusShifter": NOOP,
        "toggleFullscreenIssue": NOOP
    };

    var activeImplementation = DEFAULT_IMPLEMENTATION;

    /**
     * Creates a new object with the methods specified in "methods", and each
     * method delegates to the method with the same name in "activeImplementation"
     * In other words, this is implementing the Proxy pattern.
     *
     * This is equivalent to:
     *
     *     var delegate = {
     *          method1: function() {
     *              return activeImplementation.method1();
     *          },
     *              method2: function(a,b,c) {
     *              return activeImplementation.method2(a,b,c);
     *          }
     *      }
     *
     * The difference between using a delegate (delegate.method1()) instead of
     * a direct call to activeImplementation (activeImplementation.method1()) is
     * that we can change the implementation later. For example, this is how
     * this method could be used:
     *
     *      ////api.js
     *      var activeImplementation = DEFAULT_IMPLEMENTATION;
     *      var api = {
     *          delegate: createDelegate([
     *              //methods here
     *          ]),
     *          setImplementation: function(impl) {
     *              activeImplementation = impl;
     *          }
     *      }
     *
     * Now, let's imagine we have an "API server" that generates a delegate and
     * passes it to some "API clients"
     *
     *      ////server.js
     *      var impl = api.delegate();
     *
     *      client1.useAPI(impl);
     *      client2.useAPI(impl);
     *
     * When client1 and client2 call a method in `impl`, they are actually
     * calling a method in `DEFAULT_IMPLEMENTATION`. At any point in time,
     * the server could change that implementation to something else:
     *
     *      ////server.js
     *      api.setImplementation(somethingElse);
     *
     * From now on, when a client calls a method in `impl`, they will actually
     * call a method in `somethingElse`.
     *
     * @param {string[]} methods List of methods to create
     */
    function createDelegates(methods) {
        var delegates = {};
        methods.forEach(function(method){
            delegates[method] = function () {
                return activeImplementation[method].apply(activeImplementation, arguments);
            };
        });
        return delegates;
    }

    /**
     * Creates a global object "JIRA.API.IssueSearch" that delegates all
     * method calls to "activeImplementation"
     */
    function initAPI() {
        var api = createDelegates([
            "nextIssue",
            "previousIssue",
            "getActiveIssueId",
            "getActiveIssueKey",
            "refreshActiveIssue",
            "refreshIssue",
            "editField",
            "getFields",
            "isSaving",
            "isLoading",
            "hasAccessibleIssue",
            "showLoadError",
            "waitForSavesToComplete",
            "viewSelectedIssue",
            "returnToSearch",
            "focusSearch",
            "toggleFilterPanel",
            "isQueryValid",
            "isFullScreenIssueVisible",
            "switchLayouts",
            "openFocusShifter",
            "toggleFullscreenIssue"
        ], activeImplementation);
        return api;
    }

    /**
     * Creates a global object "JIRA.Issues.Api" (aka legacy API) that delegates
     * all the calls to the global API in "JIRA.API.IssueSearch".
     *
     * We can't alias the two global objects because the name of the methods has
     * changed.
     */
    function initLegacyAPI(issueSearchAPI) {
        var legacyAPI = {
            viewSelectedIssue: function () {
                issueSearchAPI.viewSelectedIssue();
            },
            nextIssue: function () {
                issueSearchAPI.nextIssue();
            },
            returnToSearch: function () {
                issueSearchAPI.returnToSearch();
            },
            focusSearch: function () {
                issueSearchAPI.focusSearch();
            },
            toggleFilterPanel: function () {
                issueSearchAPI.toggleFilterPanel();
            },
            isQueryValid: function () {
                return issueSearchAPI.isQueryValid();
            },
            isFullScreenIssueVisible: function () {
                return issueSearchAPI.isFullScreenIssueVisible();
            },
            switchLayouts: function () {
                issueSearchAPI.switchLayouts();
            },
            openFocusShifter: function () {
                issueSearchAPI.openFocusShifter();
            },
            waitForSavesToComplete: function () {
                issueSearchAPI.waitForSavesToComplete();
            },
            prevIssue: function () {
                issueSearchAPI.previousIssue();
            },
            getSelectedIssueId: function () {
                return issueSearchAPI.getActiveIssueId();
            },
            getSelectedIssueKey: function () {
                return issueSearchAPI.getActiveIssueKey();
            },
            refreshSelectedIssue: function () {
                return issueSearchAPI.refreshActiveIssue();
            },
            editFieldOnSelectedIssue: function (fieldId) {
                return issueSearchAPI.editField(fieldId);
            },
            getFieldsOnSelectedIssue: function () {
                return issueSearchAPI.getFields();
            },
            hasSavesInProgress: function () {
                return issueSearchAPI.isSaving();
            },
            isCurrentlyLoadingIssue: function () {
                return issueSearchAPI.isLoading();
            },
            isSelectedIssueAccessible: function () {
                return issueSearchAPI.hasAccessibleIssue();
            },
            issueIsVisible: function () {
                return issueSearchAPI.hasAccessibleIssue();
            },
            showInlineIssueLoadError: function () {
                return issueSearchAPI.showLoadError();
            },
            updateIssue: function (issue) {
                issue = issue || {};
                return issueSearchAPI.refreshIssue(issue.id);
            },
            toggleFullscreenIssue: function() {
                issueSearchAPI.toggleFullscreenIssue();
            }
        };
        AJS.namespace("JIRA.Issues.Api", null, legacyAPI);
    }

    return {
        getDefaultImplementation: function(){
            return _.clone(DEFAULT_IMPLEMENTATION);
        },

        init: function() {
            var IssueSearchAPI = initAPI();
            initLegacyAPI(IssueSearchAPI);
            AJS.namespace("JIRA.API.IssueSearch", null, IssueSearchAPI);
        },

        useImplementation: function(implementation) {
            activeImplementation = implementation;
        }
    };
});
