AJS.test.require("com.atlassian.jira.jira-projects-issue-navigator:application-test", function () {
    "use strict";

    require([
        "jira/components/issuecomponentresolver",
        "jira/components/issueeditor",
        "jira/projectissuenavigator/pages/issueview/views/issue",
        "jira/projectissuenavigator/services/notifications",
        "jira/projectissuenavigator/services/dialogsoverrider",
        "jira/projects/test-utils/mockutils",
        "jira/projects/test-utils/marionettemocker",
        "jira/projectissuenavigator/libs/marionette",
        "jira/projectissuenavigator/entities/navigatorstate",
        "jira/components/search/results",
        "jira/components/search/result",
        "jira/components/pager",
        "jquery",
        "jira/components/issueviewer/services/darkfeatures",
        "wrm/data"
    ], function(
        IssueComponentResolver,
        IssueEditor,
        IssueView,
        Notifications,
        DialogsOverrider,
        MockUtils,
        MarionetteMocker,
        Marionette,
        State,
        SearchResults,
        SearchResult,
        Pager,
        jQuery,
        DarkFeatures,
        WRMData
    ) {
        var ISSUES = [
            { id: 1, key: 'KEY-1'},
            { id: 2, key: 'KEY-2'},
            { id: 3, key: 'KEY-3'},
            { id: 4, key: 'KEY-4'}
        ];

        module("jira/projectissuenavigator/pages/issueview", {
            setup: function() {
                this.sandbox = sinon.sandbox.create();

                this.issueComponent = MarionetteMocker.createEventedMock(sinon, IssueEditor);
                this.dialogsOverrider = MarionetteMocker.createEventedMock(this.sandbox, DialogsOverrider);
                this.pager = MarionetteMocker.createEventedMock(this.sandbox, Pager);

                this.issueComponentResolver = this.sandbox.stub(IssueComponentResolver);
                this.issueComponentResolver.resolve.returns(this.issueComponent.constructor);

                this.internalView = MarionetteMocker.createEventedMock(this.sandbox, IssueView);
                this.internalView.issueContainer = new Marionette.Region({el: jQuery('<div>')});
                this.internalView.pager = new Marionette.Region({el: jQuery('<div>')});

                this.searchResults = MarionetteMocker.createEventedMock(sinon, SearchResults);
                this.searchResults = new SearchResults(ISSUES, {
                    issues: ISSUES
                });

                this.state = new State({
                    searchResults: this.searchResults
                });

                this.sandbox.stub(Notifications, "show");
                this.IssueViewConstructor = MockUtils.spyAll(this.sandbox, MockUtils.requireWithMocks("jira/projectissuenavigator/pages/issueview", {
                    "jira/components/issuecomponentresolver": this.issueComponentResolver,
                    "jira/projectissuenavigator/services/dialogsoverrider": this.dialogsOverrider.constructor,
                    "jira/projectissuenavigator/services/notifications": Notifications,
                    "jira/projectissuenavigator/pages/issueview/views/issue": this.internalView.constructor,
                    "jira/components/pager": this.pager.constructor,
                    "jira/components/issueviewer/services/darkfeatures": DarkFeatures
                }));

                this.internalView.render = sinon.spy(function() {this.$el = jQuery("<div>");});

                this.issueView = new this.IssueViewConstructor({
                    state: this.state
                });
            },
            teardown: function() {
                this.sandbox.restore();
            }
        });

        test("When deleting an issue, it gets the correct issue id", function() {
            var issueId = 1337;
            var dialog = {
                $activeTrigger: jQuery('<a/>')
            };

            this.issueComponent.getIssueId.returns(issueId);

            var getIssueIdStrategy = this.dialogsOverrider.constructor.firstCall.args[0].getIssueId;

            equal(getIssueIdStrategy(dialog), issueId, "It returns the correct issue id");
        });

        test("When deleting an issue, it gets the correct issue key", function() {
            var issueKey = "BAR-1";
            var dialog = {
                $activeTrigger: jQuery('<a/>')
            };

            this.issueComponent.getIssueKey.returns(issueKey);

            var getIssueKeyStrategy = this.dialogsOverrider.constructor.firstCall.args[0].getIssueKey;

            equal(getIssueKeyStrategy(dialog), issueKey, "It returns the correct issue key");
        });

        test("When deleting a subtask, it gets the correct issue id", function() {
            var issueId = 1337;
            var dialog = {
                $activeTrigger: jQuery('<a/>', {
                    'data-issueid': issueId
                })
            };

            var getIssueIdStrategy = this.dialogsOverrider.constructor.firstCall.args[0].getIssueId;

            equal(getIssueIdStrategy(dialog), issueId, "It returns the correct issue id");
        });

        test("When deleting a subtask, it gets the correct issue key", function() {
            var issueKey = "FOO-1";
            var dialog = {
                $activeTrigger: jQuery('<a/>', {
                    'data-issuekey': issueKey
                })
            };

            var getIssueKeyStrategy = this.dialogsOverrider.constructor.firstCall.args[0].getIssueKey;

            equal(getIssueKeyStrategy(dialog), issueKey, "It returns the correct issue key");
        });

        test("When destroying the IssueView, it stop listening for events from the pager", function() {
            this.issueView.render();

            this.issueView.destroy();

            sinon.assert.calledWith(this.issueView.stopListening, this.pager);
        });

        test("When destroying the IssueView, it destroys the pager too", function() {
            this.issueView.render();

            this.issueView.destroy();

            sinon.assert.calledOnce(this.pager.destroy);
        });

        test("When destroying the IssueView, it doesn't fail if there is no pager", function() {
            this.state.unset('searchResults');
            this.issueView.load('KEY-1');
            this.pager.destroy.reset();
            this.issueView.render();

            this.issueView.destroy();

            sinon.assert.notCalled(this.pager.destroy);
        });

        test("When rendering the IssueView, it loads the internal view markup in the page", function() {
            this.internalView.$el = jQuery("<div/>");

            this.issueView.render();

            sinon.assert.calledOnce(this.internalView.render);
            ok(this.issueView.el === this.internalView.$el);
        });

        test("When rendering the IssueView, it can display the 'collapse' button", function() {
            this.internalView.constructor.reset();

            new this.IssueViewConstructor({
                state: this.state,
                showCollapse: true
            });

            sinon.assert.calledOnce(this.internalView.constructor);
            sinon.assert.calledWith(this.internalView.constructor, sinon.match({ showCollapse: true }));
        });

        test("When rendering the IssueView, it can hide the 'collapse' button", function() {
            this.internalView.constructor.reset();

            new this.IssueViewConstructor({
                state: this.state,
                showCollapse: false
            });

            sinon.assert.calledOnce(this.internalView.constructor);
            sinon.assert.calledWith(this.internalView.constructor, sinon.match({ showCollapse: false }));
        });

        test("When attaching the IssueView to existing markup, it loads the internal view markup in the page", function() {
            var fakeViewMarkup = jQuery("<div/>");
            this.internalView.$el = fakeViewMarkup;

            this.issueView.attach(fakeViewMarkup);

            sinon.assert.calledOnce(this.internalView.attach);
            sinon.assert.calledWith(this.internalView.attach, fakeViewMarkup);
            sinon.assert.notCalled(this.internalView.render);
            ok(this.issueView.el === fakeViewMarkup);
        });

        test("When loading an issue, it loads the issue in the editor", function() {
            this.issueView.load('KEY-1');

            sinon.assert.calledOnce(this.issueComponent.loadIssue);
            sinon.assert.calledWith(this.issueComponent.loadIssue, {
                key: 'KEY-1',
                detailView: false
            });
        });

        test("When loading an issue with the 'Redirect Global to Project' Dark Feature enabled, the IssueView triggers the 'updateUrl' event", function() {
            this.sandbox.stub(DarkFeatures.REDIRECT_FROM_GLOBAL_TO_PROJECT, 'enabled').returns(true);
            var updateUrl = this.spy();
            this.issueView.on("updateUrl", updateUrl);

            this.issueView.load('KEY-1');

            sinon.assert.calledOnce(updateUrl);
            sinon.assert.calledWith(updateUrl, {
                issueKey: 'KEY-1'
            });
        });

        test("When loading an issue that is not in the search results, it hides the pager", function() {
            this.issueView.load('KEY-9999');

            sinon.assert.calledOnce(this.pager.destroy);
        });

        test("When loading an issue without search results, it hides the pager", function() {
            this.state.unset('searchResults');
            this.issueView.load('KEY-1');

            sinon.assert.calledOnce(this.pager.destroy);
        });

        test("When loading an issue in the search results, it updates the pager", function() {
            this.issueView.load('KEY-1');

            sinon.assert.calledOnce(this.pager.update);
        });

        test("When loading an issue not selected in the search results, it selects the issue in the search results", function() {
            this.searchResults.select('KEY-1');

            this.issueView.load('KEY-2');

            equal(this.searchResults.selected.get('key'), 'KEY-2');
        });

        test("When loading an issue already selected in the search results, it does not select it again", function() {
            this.searchResults.select('KEY-1');
            this.spy(this.searchResults, 'select');

            this.issueView.load('KEY-1');

            sinon.assert.notCalled(this.searchResults.select);
        });

        test("When loading an issue, it updates the state to contain the issue", function() {
            this.issueView.load('KEY-2');

            equal(this.state.get('issue'), 'KEY-2');
        });

        test("When loading an issue present in the page, it loads the issue in the editor", function() {
            this.stub(WRMData, "claim")
                .withArgs('projectId').returns(1)
                .withArgs('projectKey').returns('KEY')
                .withArgs('projectType').returns('software');

            this.issueView.loadFromPage('KEY-1');

            sinon.assert.calledOnce(this.issueComponent.loadIssueRenderedByTheServer);
            sinon.assert.calledWith(this.issueComponent.loadIssueRenderedByTheServer, {
                key: 'KEY-1',
                detailView: false,
                project: {
                    id: 1,
                    key: 'KEY',
                    projectType: 'software'
                }
            });
        });

        test("When loading an issue present in the page with the 'Redirect Global to Project' Dark Feature enabled, the IssueView triggers the 'updateUrl' event", function() {
            this.sandbox.stub(DarkFeatures.REDIRECT_FROM_GLOBAL_TO_PROJECT, 'enabled').returns(true);
            var updateUrl = this.spy();
            this.issueView.on("updateUrl", updateUrl);

            this.issueView.loadFromPage('KEY-1');

            sinon.assert.calledOnce(updateUrl);
            sinon.assert.calledWith(updateUrl, {
                issueKey: 'KEY-1'
            });
        });

        test("When loading an issue present in the page that is not in the search results, it hides the pager", function() {
            this.issueView.loadFromPage('KEY-9999');

            sinon.assert.calledOnce(this.pager.destroy);
        });

        test("When loading an issue present in the page without search results, it hides the pager", function() {
            this.state.unset('searchResults');
            this.issueView.loadFromPage('KEY-1');

            sinon.assert.calledOnce(this.pager.destroy);
        });

        test("When loading an issue present in the page and in the search results, it updates the pager", function() {
            this.issueView.loadFromPage('KEY-1');

            sinon.assert.calledOnce(this.pager.update);
        });

        test("When loading an issue present in the page not selected in the search result, it selects the issue in the search results", function() {
            this.searchResults.select('KEY-1');

            this.issueView.loadFromPage('KEY-2');

            equal(this.searchResults.selected.get('key'), 'KEY-2');
        });

        test("When loading an issue present in the page already selected in the search results, it does not select it again", function() {
            this.searchResults.select('KEY-1');
            this.spy(this.searchResults, 'select');

            this.issueView.loadFromPage('KEY-1');

            sinon.assert.notCalled(this.searchResults.select);
        });

        test("When loading an issue present in the page, it updates the state to contain the issue", function() {
            this.issueView.loadFromPage('KEY-2');

            equal(this.state.get('issue'), 'KEY-2');
        });

        test("When the editor triggers a 'loadComplete', this page triggers an 'editorLoaded' event", function() {
            var onEditorLoaded = this.spy();
            this.issueView.on("editorLoaded", onEditorLoaded);
            var payload = {
                issueKey: 'KEY-2',
                showPager: true
            };

            this.issueComponent.trigger("loadComplete", null, payload);

            sinon.assert.calledOnce(onEditorLoaded);
            sinon.assert.calledWith(onEditorLoaded, payload);
        });

        test("When the editor triggers a 'loadComplete' event with the 'Redirect Global to Project' Dark Feature disabled, the IssueView does not trigger the 'updateUrl' event", function() {
            this.sandbox.stub(DarkFeatures.REDIRECT_FROM_GLOBAL_TO_PROJECT, 'enabled').returns(false);

            this.state.set('issue', 'KEY-1');
            var updateUrl = this.spy();
            this.issueView.on("updateUrl", updateUrl);
            this.issueComponent.trigger("loadComplete");

            sinon.assert.notCalled(updateUrl);
        });

        test("When the editor triggers a 'loadError', this page triggers an 'loadError' event", function() {
            var onLoadError = this.spy();
            this.issueView.on("loadError", onLoadError);
            var payload = {
                issueKey: 'KEY-2',
                showPager: true
            };

            this.issueComponent.trigger("loadError", payload);

            sinon.assert.calledOnce(onLoadError);
            sinon.assert.calledWith(onLoadError, payload);
        });

        test("When the internal view triggers a collapse event, the IssueView retriggers it", function() {
            var onCollapse = this.spy();
            this.issueView.on('collapse', onCollapse);

            this.internalView.trigger('collapse');

            sinon.assert.called(onCollapse);
        });

        test("When the editor triggers a 'linkToIssue' event, the IssueView loads it without the pager", function() {
            this.issueComponent.trigger('linkToIssue', {
                issueKey: 'KEY-123'
            });

            sinon.assert.called(this.pager.destroy);
        });

        test("When the pager selects the next issue, the IssueView loads that issue with the pager", function() {
            this.searchResults.select('KEY-1');

            this.pager.trigger('next');

            sinon.assert.calledOnce(this.issueComponent.loadIssue);
            sinon.assert.calledWith(this.issueComponent.loadIssue, {
                key: 'KEY-2',
                detailView: false
            });
        });

        test("When the pager selects the previous issue, the IssueView loads that issue with the pager", function() {
            this.searchResults.select('KEY-2');

            this.pager.trigger('previous');

            sinon.assert.calledOnce(this.issueComponent.loadIssue);
            sinon.assert.calledWith(this.issueComponent.loadIssue, {
                key: 'KEY-1',
                detailView: false
            });
        });

        test("When the pager selects the next issue, the IssueView triggers a 'pager:next' event", function() {
            var onPagerNext = this.spy();
            this.searchResults.select('KEY-1');
            this.issueView.on('pager:next', onPagerNext);

            this.pager.trigger('next');

            sinon.assert.calledOnce(onPagerNext);
            sinon.assert.calledWith(onPagerNext, {current: 1, total: 4}); //0-based index
        });

        test("When the pager selects the previous issue, the IssueView triggers a 'pager:previous' event", function() {
            var onPagerPrevious = this.spy();
            this.searchResults.select('KEY-4');
            this.issueView.on('pager:previous', onPagerPrevious);

            this.pager.trigger('previous');

            sinon.assert.calledOnce(onPagerPrevious);
            sinon.assert.calledWith(onPagerPrevious, {current: 2, total: 4}); //0-based index
        });

        test("When adjusting the size, it adjusts the size of the internal components", function() {
            this.issueView.adjustSize();

            sinon.assert.calledOnce(this.internalView.maximizeHeight);
            sinon.assert.calledOnce(this.issueComponent.applyResponsiveDesign);
        });

        test("When the internal view is rendered, the IssueEditor is rendered inside", function() {
            this.internalView.trigger('render');

            sinon.assert.calledOnce(this.issueComponent.setContainer);
            sinon.assert.calledWith(this.issueComponent.setContainer, this.internalView.issueContainer.$el);
        });

        test("When the internal view is rendered, the Pager is rendered inside", function() {
            this.internalView.trigger('render');

            sinon.assert.calledWith(this.pager.show);
            sinon.assert.calledOnce(this.pager.show, this.internalView.issueContainer.$el);
        });

        test("When the internal view is rendered and there is no pager, it doesn't break", function() {
            this.state.unset('searchResults');
            this.issueView.load('KEY-1');

            this.internalView.trigger('render');

            sinon.assert.notCalled(this.pager.show);
        });

        test("When the IssueEditor loads all the fields, IssueView triggers a 'editorFieldsLoaded' event", function() {
            var onEditorFieldsLoaded = this.spy();
            this.issueView.on('editorFieldsLoaded', onEditorFieldsLoaded);

            this.issueComponent.trigger('fieldsLoaded');

            sinon.assert.calledOnce(onEditorFieldsLoaded);
        });

        test("When the page is focused, it delegates the call in the IssueEditor", function() {
            this.issueView.focusIssueEditor();

            sinon.assert.calledOnce(this.issueComponent.focus);
        });

        test("When the IssueEditor refineViewer event is triggered, it reloads the IssueEditor with the new data", function () {
            var preventDefault = this.spy();
            var query = {property: "value"};
            var event = {query: query, preventDefault: preventDefault};
            this.issueComponent.trigger('refineViewer', event);

            sinon.assert.calledOnce(preventDefault);
            sinon.assert.calledOnce(this.issueComponent.updateIssueWithQuery);
            sinon.assert.calledWith(this.issueComponent.updateIssueWithQuery, query);
        });
    });
});
