define("jira/projectissuenavigator/pages/issuesearch", ['require'], function (require) {
    "use strict";

    var _ = require('jira/projectissuenavigator/libs/underscore');
    var DetailsLayout = require('jira/components/detailslayout');
    var DialogsOverrider = require('jira/projectissuenavigator/services/dialogsoverrider');
    var EmptyView = require('jira/projectissuenavigator/pages/issuesearch/views/empty');
    var Filters = require('jira/projectissuenavigator/services/filters');
    var FocusShifter = require('jira/projectissuenavigator/services/focusshifter');
    var InlineIssueCreate = require('jira/projectissuenavigator/services/inline-issue-create');
    var JIRAEvents = require('jira/util/events');
    var JIRAFlag = require('jira/flag');
    var jQuery = require('jquery');
    var LinksCapturer = require('jira/projectissuenavigator/services/linkscapturer');
    var Marionette = require('jira/projectissuenavigator/libs/marionette');
    var Message = require('jira/message');
    var Meta = require('jira/util/data/meta');
    var Metrics = require('jira/projectissuenavigator/services/metrics');
    var Notifications = require('jira/projectissuenavigator/services/notifications');
    var SmartAjax = require('jira/ajs/ajax/smart-ajax');
    var Templates = require("jira/projectissuenavigator/pages/issuesearch/templates");
    var Types = require('jira/components/util/events');
    var URI = require('jira/projectissuenavigator/libs/uri');
    var URLHelper = require('jira/projectissuenavigator/services/urlhelper');

    var Subnavigator = JIRA.Projects.Subnavigator;

    return Marionette.Object.extend({
        /**
         * Constructor
         *
         * @param {object} options Configuration options
         * @param {module:jira/projectissuenavigator/entities/navigatorstate} options.state State with the navigation details
         * @param {boolean} options.canCreateIssues Whether this instance should include the InlineIssueCreate
         * @param {module:jira/components/search} options.search Search service used by this page. Might have been
         *                                                       pre-populated with results.
         */
        initialize: function(options) {
            this.el = jQuery(Templates.issueSearch());

            this._buildState(options.state);
            this._buildSearch(options.search);
            this._buildSubnavigation();
            if (options.canCreateIssues) {
                this._buildInlineIssueCreate();
            }
            this._buildDetailsLayout();
            this._buildLinksCapturer();
            this._buildDialogsOverrider();
            this._buildFocusShifter();
        },

        _buildState: function(state) {
            this.state = state;
        },

        _setTitleText: function (filterId) {
            var filters = Filters.getAvailableFilters(true);
            var filter = _.chain(filters)
                .flatten()
                .findWhere({ id: filterId })
                .value();

            var title = this.el.find("#issues-subnavigation-title");
            title.text((filter && filter.label) || "");
        },

        _buildSubnavigation: function () {
            var defaultFilterId = this.state.get('filter') || Filters.getDefaultFilterId();
            this._setTitleText(defaultFilterId);

            this.subnavigation = new Subnavigator({
                triggerPlaceholder: this.el.find("#issues-subnavigation-trigger"),
                contentPlaceholder: this.el.find("#issues-subnavigation-content"),
                itemGroups: Filters.getAvailableFilters(true),
                selectedItem: defaultFilterId,
                hideSelectedItem: false,
                changeViewText: AJS.I18n.getText("sidebar.issue.nav.change.filter")
            });

            this.listenTo(this.subnavigation, {
                "itemSelected": function (event) {
                    if (event.item.id === "manage") {
                        this.trigger('manageFilters');
                    } else if (event.item.jql) {
                        Metrics.startIssueSearch();

                        event.preventDefault();

                        this._setTitleText(event.item.id);

                        // Do not update the URL, it will be updated when the results are loaded.
                        // If not, we would end up creating two entries in the history for the same action.
                        this.search({
                            filter: event.item.id,
                            issue: null,
                            orderby: null
                        });
                        this.trigger('filterSelected', {
                            filterId: event.item.id,
                            source: 'subnavigator'
                        });
                    }
                }
            });
        },

        _buildInlineIssueCreate: function() {
            this.inlineIssueCreate = new InlineIssueCreate();
            this.listenTo(this.inlineIssueCreate, {
                "issueCreated": function(issueInfo) {
                    Metrics.startIssueSearch();
                    this.search({
                        issue: issueInfo.issue.issueKey
                    });
                }
            });
        },

        _buildDetailsLayout: function() {
            var options = {
                baseURL: URLHelper.generateRootUrl(this.state.get('project')),
                useExpand: true
            };

            if (this.options.canCreateIssues) {
                options.displayInlineIssueCreate = true;
                options.emptyViewFactory = _.bind(function() {
                    this.emptyView = new EmptyView({
                        emptyViewContent: this.options.emptyViewContent
                    });
                    return this.emptyView;
                }, this);
            }

            this.detailsLayout = new DetailsLayout(options);
            this.listenTo(this.detailsLayout, {
                "select": function(issueData) {
                    this.state.set('issue', issueData.key);
                    this.trigger("issue:select", {
                        issue: issueData.key
                    });
                },
                "empty": function() {
                    this.state.unset('issue');
                    if (this.options.canCreateIssues) {
                        this.emptyView.inlineIssueCreateContainer._ensureElement();
                        this.inlineIssueCreate.show(this.emptyView.inlineIssueCreateContainer.$el);
                        this.inlineIssueCreate.activateWithJQL(this.state.get('searchResults').jql);
                    }
                    this.trigger("issue:empty");
                },
                "list:update": function() {
                    this.trigger("list:update");
                },
                "list:sort": function(jql) {
                    Metrics.startIssueSearch();

                    var match = jql.match(/ORDER +BY +(.*)$/i);
                    if (!match) return;
                    this.search({
                        orderby: match[1]
                    });
                },
                "list:refresh": function() {
                    // Triggered *only* when the user clicks in the refresh icon
                    Metrics.startIssueSearch();

                    // When refreshing, forget the current selected issue, as it might not be
                    // part of the new results.
                    this.search({
                        issue: null
                    });
                },
                "list:select": function(issueData) {
                    // Triggered *only* when the user clicks in an issue in the list
                    Metrics.startIssueSearch();

                    this.trigger("issueSelectedInList", issueData);
                },
                "render": function() {
                    JIRAEvents.trigger(Types.LAYOUT_RENDERED);
                },
                "editorLoadedFromCache editorLoaded": function(event) {
                    Metrics.endIssueSearch();
                    Meta.set("viewissue-permlink", this._getFullScreenURL(event.issueKey));
                    this.trigger('editorLoaded', {issueRefreshedEvent: (event.issueEditorOptions || {}).issueRefreshedEvent || false});
                },
                "linkToIssue": function(options) {
                    this.trigger("linkToIssue", options);
                },
                "list:pagination": function() {
                    // Triggered *only* when the user clicks in a pagination link
                    Metrics.startIssueSearch();
                },
                "error:loadpage": function(errorInfo) {
                    this.showErrorFlagFromResponse(errorInfo);
                },
                "editor:saveSuccess": function(eventData) {
                    this.trigger("editor:saveSuccess", {
                        issueId: eventData.event,
                        savedFieldIds: eventData.savedFieldIds,
                        savedFieldTypes: eventData.savedFieldTypes,
                        duration: eventData.duration
                    });
                },
                "editor:editField": function(eventData) {
                    this.trigger("editor:editField", eventData);
                },
                "editor:editFieldCancel": function(eventData) {
                    this.trigger("editor:editFieldCancel", eventData);
                },
                "pager:next": function(data) {
                    Metrics.startIssueSearch();
                    this.trigger("pager:next", data);
                },
                "pager:previous": function (data) {
                    Metrics.startIssueSearch();
                    this.trigger("pager:previous", data);
                },
                "expand": function () {
                    this.trigger("expand", "ui-button");
                },
                "editorError": function() {
                    this.trigger("editorError");
                },
                "editorFieldsLoaded": function() {
                    this.trigger("editorFieldsLoaded");
                },
                "individualPanelRendered": function(renderedPanel) {
                    this.trigger("individualPanelRendered", renderedPanel);
                }
            });

            if (this.options.canCreateIssues) {
                this.listenTo(this.detailsLayout, {
                    "issueCreated": function(issueInfo) {
                        if (Filters.filterSupportsInlineIssueCreate(this.state.get('filter'))) {
                            Metrics.startIssueSearch();
                            this.search({
                                issue: issueInfo.issue.issueKey
                            });
                        } else {
                            Message.showSuccessMsg(Templates.issueCreatedFlag({
                                issueKey: issueInfo.issue.issueKey,
                                projectKey: this.state.get('project'),
                                summary: issueInfo.fields.summary
                            }), {
                                closeable: true
                            });
                        }
                    }
                });
            }
        },

        _buildSearch: function(search) {
            this.searchService = search;
        },

        _buildLinksCapturer: function() {
            this.linksCapturer = new LinksCapturer({el: this.el});
            this.listenTo(this.linksCapturer, {
                "linkToFullIssueNavigator": function() {
                    this.trigger("goToFullIssueNavigator");
                }
            });
        },

        _buildDialogsOverrider: function() {
            function dataExtractor(dataKey, orElse, dialog) {
                return dialog && dialog.$activeTrigger && dialog.$activeTrigger.data(dataKey) || (typeof orElse === 'function') && orElse();
            }
            this.dialogsOverrider = new DialogsOverrider({
                getIssueId: _.partial(dataExtractor, "issueid", _.bind(this.detailsLayout.getActiveIssueId, this.detailsLayout)),
                getIssueKey: _.partial(dataExtractor, "issuekey", _.bind(this.detailsLayout.getActiveIssueKey, this.detailsLayout))
            });
            this.listenTo(this.dialogsOverrider, {
                "quickEdit quickCreateSubtask issueUpdate": function(data) {
                    this.detailsLayout.refreshIssue(data.issueId).done(_.bind(function () {
                        Notifications.show(data.message, data.issueKey);
                    }, this));
                },
                "issueDelete": function(data) {
                    this.removeIssue(data.issueId).done(_.bind(function () {
                        Notifications.show(data.message, data.issueKey);
                    }, this));
                }
            });
        },

        _buildFocusShifter: function() {
            this.focusShifter = new FocusShifter(this.detailsLayout.issueComponent);
        },

        _getFullScreenURL: function (issueKey) {
            var baseUrl = Meta.get("jira-base-url");
            return new URI(baseUrl + "/browse/" + issueKey).normalize().toString();
        },

        /**
         * Loads the data stored in the internal state
         */
        load: function() {
            var searchResults = this.state.get('searchResults');
            var filter = this.state.get('filter');
            var isAllFilter = Filters.getAllIssuesFilterId() === filter;
            var issue = this.state.get('issue');

            var issueFoundInFilter = searchResults.hasIssueInSearch(issue);

            this._setTitleText(filter);
            this.subnavigation.selectItem(filter);

            // No issue specified
            if (!issue) {
                this.detailsLayout.load(searchResults);
                return;
            }

            // Issue in the current filter, load it!
            if (issueFoundInFilter) {
                this.detailsLayout.load(searchResults, issue);
                return;
            }

            // Issue not in the filter: retry with 'All issues' filter
            if (!isAllFilter) {
                this.search({
                    filter: Filters.getAllIssuesFilterId()
                });

                this.trigger('filterSelected', {
                    filterId: Filters.getAllIssuesFilterId(),
                    source: 'issueFilterMismatch'
                });
                return;
            }

            this.detailsLayout.load(searchResults);
        },

        /**
         * Does a search based on the params
         *
         * When an item is missing in params, it will use the value from the internal state. Passing `null` in any
         * parameter will clear that parameter from the internal state.
         *
         * @param {object} params Parameters of the search
         * @param {string} params.issue Issue key to load
         * @param {string} params.filter FilterId to load
         * @param {string} params.orderby Order clause to use in the JQL search
         * @param {number} params.startIndex Load the issue in this position (only used if params.issue is not defined)
         */
        search: function(params) {
            params = params || {};
            this.state.update({
                issue: params.issue,
                filter: params.filter,
                orderby: params.orderby
            });

            this.detailsLayout.showLoading();

            var jql = Filters.getJQL(this.state.get('filter'), this.state.get('project'), this.state.get('orderby'));
            this.searchService.search(jql).done(_.bind(function(searchResults) {
                this.state.update({
                    searchResults: searchResults
                });

                if (!this.state.has('issue') && params.startIndex) {
                    this.state.update({
                        issue: searchResults.getIssueKeyForIndex(params.startIndex)
                    });
                }

                this.load();
            }, this)).fail(_.bind(function(response) {
                this.detailsLayout.hideLoading();
                this.showErrorFlagFromResponse(response);
            }, this));
        },

        show: function() {
            this.subnavigation.show();
            this.detailsLayout.show(jQuery(this.el).find(".details-layout"));
        },

        onDestroy: function() {
            this.subnavigation.destroy();
            if (this.inlineIssueCreate) {
                this.inlineIssueCreate.destroy();
            }
            this.detailsLayout.destroy();
            this.linksCapturer.destroy();
            this.dialogsOverrider.destroy();
            this.focusShifter.destroy();
        },

        adjustSize: function() {
            this.detailsLayout.adjustSize();
        },

        showErrorFlagFromResponse: function(response) {
            if (typeof response === 'undefined' ||
                response.statusText === SmartAjax.SmartAjaxResult.TIMEOUT ||
                response.status === 401 ||
                response.hasData) {
                return this.showErrorFlag(SmartAjax.buildSimpleErrorContent(response));
            } else {
                var responseText = JSON.parse(response.responseText);

                if (response.status === 400 && responseText.errorMessages.length > 0) {
                    return this.showErrorFlag(responseText.errorMessages[0]);
                } else {
                    return this.showErrorFlag(SmartAjax.buildSimpleErrorContent(response));
                }
            }
        },

        showErrorFlag: function(msg) {
            return JIRAFlag.showErrorMsg('', msg, {
                persistent: false
            });
        },

        /**
         * Removes the issue with the given issueId from the search results.
         * @param issueId ID of the issue to remove
         * @returns {jQuery.Deferred} when the issue has been removed
         */
        removeIssue: function(issueId) {
            return this.detailsLayout.removeIssue(issueId);
        },

        getIssueId: function() {
            return this.detailsLayout.getActiveIssueId();
        },

        getIssueKey: function() {
            return this.detailsLayout.getActiveIssueKey();
        },

        getProjectId: function() {
            return this.detailsLayout.getActiveProjectId();
        },

        getProjectKey: function() {
            return this.detailsLayout.getActiveProjectKey();
        },

        getProjectType: function() {
            return this.detailsLayout.getActiveProjectType();
        },

        getAPIImplementation: function() {
            var page = this;
            return {
                "nextIssue": function() {
                    Metrics.startIssueSearch();
                    return page.detailsLayout.selectNext();
                },
                "previousIssue": function() {
                    Metrics.startIssueSearch();
                    return page.detailsLayout.selectPrevious();
                },
                "getActiveIssueId": function() {
                    return page.getIssueId();
                },
                "getActiveIssueKey": function() {
                    return page.getIssueKey();
                },
                "refreshActiveIssue": function() {
                    Metrics.startIssueSearch();
                    var activeIssueId = page.detailsLayout.getActiveIssueId();
                    return page.detailsLayout.refreshIssue(activeIssueId);
                },
                "refreshIssue": function(issueId) {
                    Metrics.startIssueSearch();
                    return page.detailsLayout.refreshIssue(issueId);
                },
                "editField": function(fieldId) {
                    var fields = page.detailsLayout.issueComponent.getFields();
                    var field = fields && fields.get(fieldId);
                    var permitted = field && field.isEditable();

                    if (permitted) {
                        page.detailsLayout.issueComponent.editField(field);
                    }

                    return permitted;
                },
                "getFields": function() {
                    var fields = page.detailsLayout.issueComponent.getFields();
                    return fields.length ? fields : undefined;
                },
                "isSaving": function() {
                    return page.detailsLayout.issueComponent.hasSavesInProgress();
                },
                "isLoading": function() {
                    return page.detailsLayout.isLoading();
                },
                "hasAccessibleIssue": function() {
                    return Boolean(
                        page.detailsLayout.searchResults &&
                        page.detailsLayout.searchResults.length &&
                        page.detailsLayout.searchResults.selected &&
                        !page.detailsLayout.searchResults.selected.get('inaccessible')
                    );
                },
                "showLoadError": function() {
                    return page.showErrorFlag(JIRA.Components.IssueViewer.Templates.Body.errorsLoading());
                },
                "waitForSavesToComplete": function() {
                    var d = new jQuery.Deferred();
                    if (!page.detailsLayout.issueComponent.hasSavesInProgress()) {
                        d.resolve();
                    } else {
                        var success = function() {
                            _.defer(function() {
                                if (!page.detailsLayout.issueComponent.hasSavesInProgress()) {
                                    d.resolve();
                                }
                            });
                        };
                        page.once("editor:saveSuccess", success);
                    }
                    return d.promise();
                },
                "openFocusShifter": function() {
                    page.focusShifter.show();
                },
                "toggleFullscreenIssue": function() {
                    page.trigger("expand", "kb-shortcut");
                }
            };
        },

        focusIssueList: function() {
            this.detailsLayout.focusList();
        },

        focusIssueEditor: function() {
            this.detailsLayout.focusEditor();
        }
    });
});
