define("jira/projectissuenavigator", ["require"], function(require) {
    "use strict";

    var _ = require("jira/projectissuenavigator/libs/underscore");
    var Analytics = require("jira/projectissuenavigator/services/analytics");
    var API = require("jira/projectissuenavigator/services/api");
    var Filters = require("jira/projectissuenavigator/services/filters");
    var DialogsCleaner = require('jira/projectissuenavigator/services/dialogscleaner');
    var IssueSearch = require("jira/projectissuenavigator/pages/issuesearch");
    var IssueView = require("jira/projectissuenavigator/pages/issueview");
    var jQuery = require("jquery");
    var Marionette = require("jira/projectissuenavigator/libs/marionette");
    var Metrics = require("jira/projectissuenavigator/services/metrics");
    var Router = require("jira/projectissuenavigator/services/router");
    var State = require("jira/projectissuenavigator/entities/navigatorstate");
    var Traces = require("jira/projectissuenavigator/services/traces");
    var JIRAEvents = require("jira/projectissuenavigator/services/jiraevents");
    var WRMData = require("wrm/data");
    var Search = require("jira/components/search");
    var KeyboardShortcuts = require("jira/projectissuenavigator/services/keyboardshortcuts");
    var URLHelper = require('jira/projectissuenavigator/services/urlhelper');
    var Browser = require('jira/projectissuenavigator/services/browser');

    return Marionette.Object.extend({
        _initializedFromPageLoad: true,

        initialize: function(options) {
            options = _.defaults({}, options, {
                el: jQuery('<div></div>'),
                canCreateIssues: false,
                emptyViewContent: ''
            });
            this.el = options.el;
            this.canCreateIssues = options.canCreateIssues;
            this.emptyViewContent = options.emptyViewContent;
        },

        /**
         * Creates the Search service used by IssueView and IssueSearch to get the current search results.
         *
         * If the there is pre-propulated data (via WRM), we'll create a 'searchResults' object straight away
         */
        _buildSearch: function() {
            this.searchService = new Search();

            var embeddedData = WRMData.claim("com.atlassian.jira.jira-issue-navigator-components:search-data");
            if (embeddedData) {
                var searchResults = this.searchService.setPreloadedData(embeddedData);

                var data = WRMData.claim("com.atlassian.jira.jira-issue-navigator-components:search-results");
                if (data) searchResults.setPreloadedData(data);

                this.state.set('searchResults', searchResults);
            }
        },

        _buildIssueView: function() {
            this.issueView = new IssueView({
                state: this.state,
                search: this.searchService,
                showCollapse: this.state.has('filter')
            });
            this.listenTo(this.issueView, {
                "collapse": function(source) {
                    Metrics.startIssueSearch();
                    Analytics.collapseIssue({
                        source: source
                    });
                    this._showIssueSearch();
                },
                "linkToIssue": function() {
                    Metrics.startIssueView();
                },
                "issueDelete": function(data) {
                    Metrics.startIssueSearch();

                    // We need to delete the issue before showing the IssueSearch, if not we will have some race
                    // conditions regarding network load. Also, we can't use issueSearch.removeIssue because it
                    // assumes the issue has been already shown, so we remove it directly from the searchResults.
                    var searchResults = this.state.get('searchResults');
                    if (!searchResults) {
                        this._destroyIssueView();
                        this._showIssueSearch();
                    } else {
                        var model = searchResults.get(data.issueId);
                        searchResults.removeAndUpdateSelectionIfNeeded(model).done(function () {
                            if (searchResults.selected) {
                                this.state.set('issue', searchResults.selected.get('key'));
                            }
                            this._destroyIssueView();
                            this._showIssueSearch();
                        }.bind(this));
                    }
                },
                "updateUrl": function(options) {
                    this.router.updateURL({
                        project: this.state.get('project'),
                        issue: options.issueKey,
                        filter: this.state.get('filter'),
                        orderby: this.state.get('orderby'),
                        replaceHistory: this._initializedFromPageLoad,
                        keepParameters: this._initializedFromPageLoad,
                        isIssueView: true
                    });
                    this._initializedFromPageLoad = false;
                },
                "editorFieldsLoaded": this._notifyIssueViewEditorFieldsLoaded.bind(this),
                "loadError": function(properties) {
                    properties = properties || [];
                    this.jiraEvents.triggerRefreshIssuePage(properties.issueId);
                },
                "individualPanelRendered": function(renderedPanel) {
                    this.jiraEvents.triggerNewContentAddedFromPanelRefreshed(renderedPanel);
                },
                "viewRendered": function($el, issueId) {
                    this.jiraEvents.triggerNewContentAddedFromPageLoad($el);
                    this.jiraEvents.triggerRefreshToggleBlocks(issueId);
                },
                "panelRendered": function(panelId, $ctx, $existing) {
                    this.jiraEvents.triggerPanelRefreshed(panelId, $ctx, $existing);
                }
            });
        },

        _buildKeyboardShortuts: function() {
            var keyboardShortcuts = new KeyboardShortcuts();
            this.listenTo(keyboardShortcuts, {
                "focusIssueList": function() {
                    if (this.issueSearch) {
                        this.issueSearch.focusIssueList();
                    }
                },
                "focusIssueEditor": function() {
                    if (this.issueSearch) {
                        this.issueSearch.focusIssueEditor();
                    } else if (this.issueView) {
                        this.issueView.focusIssueEditor();
                    }
                }
            });
        },

        _notifyIssueViewEditorLoadedAndInThePage: function(options) {
            options = options || {};
            this.jiraEvents.triggerNewContentAddedFromPanelRefreshed(this.issueView.el);
            if (!options.existingMarkup) {
                this.jiraEvents.triggerRefreshIssuePage(this.issueView.getIssueId());
            }
            Traces.traceEditorLoaded();
            Analytics.triggerProjectNavViewIssue({
                issueId: this.issueView.getIssueId(),
                issueKey: this.issueView.getIssueKey(),
                projectId: this.issueView.getProjectId(),
                projectKey: this.issueView.getProjectKey(),
                projectType: this.issueView.getProjectType()
            });

            Metrics.endIssueView();
        },

        _notifyIssueViewEditorFieldsLoaded: function() {
            Traces.traceEditorFieldsLoaded();
        },

        _notifyIssueSearchEditorLoadedAndInThePage: function(options) {
            options = options || {};
            this.jiraEvents.triggerNewContentAddedFromPanelRefreshed(this.issueSearch.el);
            if (!options.issueRefreshedEvent) {
                this.jiraEvents.triggerRefreshIssuePage(this.issueSearch.getIssueId());
            }
            Analytics.triggerProjectNavViewIssue({
                issueId: this.issueSearch.getIssueId(),
                issueKey: this.issueSearch.getIssueKey(),
                projectId: this.issueSearch.getProjectId(),
                projectKey: this.issueSearch.getProjectKey(),
                projectType: this.issueSearch.getProjectType()
            });

            Traces.traceEditorLoaded();
        },

        _showIssueView: function (options) {
            this._buildIssueView();
            API.useImplementation(this._getAPIImplementation(this.issueView));

            function onIssueLoaded (options) {
                // From now on, when the editor loads a different issue, we want to notify external JIRA components
                // (like JIRA Traces or JIRA Events).
                //
                // We can't do that when we build the page because then the handler for `editorLoaded` could be called
                // before the page is presented to the user, creating potential flaky tests (traces being fired before
                // the page is injected into the document) or problems in plugins (if they assume the content that
                // triggered the JIRA Event is already on the document)
                this.listenTo(this.issueView, {
                    'editorLoaded': this._notifyIssueViewEditorLoadedAndInThePage.bind(this)
                });
                options = _.defaults({}, options, {existingMarkup: false});
                this._notifyIssueViewEditorLoadedAndInThePage(options);

                this.issueView.adjustSize();
                this.issueView.focusIssueEditor();
            }

            if (options.existingMarkup === true) {
                this.issueView.attach(this.el.find(".issue-view"));
                this.issueView.loadFromPage(options.issueKey).always(function () {
                    onIssueLoaded.call(this, {existingMarkup: true});
                }.bind(this));
            } else {
                this.issueView.render();
                this.issueView.load(options.issueKey).always(function () {
                    this._activateIssueViewPage();
                    onIssueLoaded.call(this, {existingMarkup: false});
                }.bind(this));
            }
        },

        _destroyIssueView: function() {
            if (this.issueView) {
                this.issueView.destroy();
                this.stopListening(this.issueView);
                delete this.issueView;
            }
        },

        _buildIssueSearch: function() {
            this.issueSearch = new IssueSearch({
                canCreateIssues: this.canCreateIssues,
                emptyViewContent: this.emptyViewContent,
                state: this.state,
                search: this.searchService
            });
            this.listenTo(this.issueSearch, {
                "linkToIssue": function(options) {
                    Metrics.startIssueView();
                    this.state.unset('searchResults');
                    this.state.unset('filter');
                    this._showIssueView({
                        issueKey: options.issueKey
                    });
                },
                "expand": function(source) {
                    Metrics.startIssueView();
                    Analytics.expandIssue({
                        source: source
                    });

                    // IssueView can destroy the searchResults if the user navigates to an issue from another project
                    // using a link in the page. In that case, when the user collapses that external issue we want to
                    // restore the IssueSearch to its initial state: same search, same filter, same selected issue.
                    this._stateBeforeExpand = new State({
                        filter: this.state.get('filter'),
                        searchResults: this.state.get('searchResults'),
                        issue: this.state.get('issue')
                    });
                    this._showIssueView({
                        issueKey: this.issueSearch.getIssueKey()
                    });
                },
                'filterSelected': function(filterData) {
                    Analytics.filterSelect(filterData);
                },
                'goToFullIssueNavigator': function() {
                    Analytics.goToFullIssueNavigator();
                },
                'issueSelectedInList': function(issueData) {
                    Analytics.issueSelectedInList(issueData);
                },
                'manageFilters': function() {
                    Analytics.manageFilters();
                },
                'list:update': function() {
                    Traces.traceListLoaded();
                },
                'editorLoaded': function (options) {
                    this._notifyIssueSearchEditorLoadedAndInThePage(options);
                },
                "editorFieldsLoaded": function() {
                    this._notifyIssueViewEditorFieldsLoaded();
                },
                "editorError": function() {
                    Traces.traceEditorError();
                },
                "pager:next": function(data) {
                    Analytics.pager('next', data);
                },
                "pager:previous": function (data) {
                    Analytics.pager('previous', data);
                },
                "issue:select issue:empty": function() {
                    // If _initializedFromPageLoad is true, this is the first time we update the URL. We want to replace
                    // the existing URL, because that URL is "wrong". The most common case is:
                    //
                    //  Original URL: /project/PKEY/issues (eg. when using the header navigation)
                    //  URL to replace with:  /project/PKEY/issues/KEY?filter=X  (if there are search results)
                    //                        /project/PKEY/issues?filter=X      (if the search is empty)
                    //
                    // There could be the case where the original URL is ok. It doesn't matter because Backbone.Route
                    // will not do anything in that case.
                    //
                    // Another way to look at this: when the Project Issue Navigator is loaded, we might need to do
                    // some URL rewriting in order to represent the final state (i.e. the selected issue). That new URL
                    // should replace the existing one, if not the back-button will bring the user back to the 'wrong'
                    // state. By replacing it, the back-button will take the user to the page before the Project Issue
                    // Navigator. But from now on, we control the URL, and we know all the states are valid, so we don't
                    // need to replace it anymore, just keep adding the URLs to the History stack.
                    this.router.updateURL({
                        project: this.state.get('project'),
                        issue: this.state.get('issue'),
                        filter: this.state.get('filter'),
                        orderby: this.state.get('orderby'),
                        replaceHistory: this._initializedFromPageLoad
                    });
                    this._initializedFromPageLoad = false;
                },
                'editor:saveSuccess': function(eventData) {
                    Analytics.editorFieldSaved(eventData);
                },
                'editor:editField': function(eventData) {
                    Analytics.editFieldStarted(eventData);
                },
                'editor:editFieldCancel': function(eventData) {
                    Analytics.editFieldCancelled(eventData);
                }
            });
        },

        _activateIssueSearchPage: function () {
            // We have to destroy the existing page before rendering the new one, if not the existing page
            // can't clean up itself properly because its markup is no longer in the document.
            this._destroyIssueView();
            this._activatePage(this.issueSearch);
        },

        _activateIssueViewPage: function () {
            // We have to destroy the existing page before rendering the new one, if not the existing page
            // can't clean up itself properly because its markup is no longer in the document.
            this._destroyIssueSearch();
            this._activatePage(this.issueView);
        },

        _showIssueSearch: function (params) {
            params = params || {};
            //Issue Search is not happy with in-memory rendering (mainly because the
            //jQuery draggable sidebar), so unfortunately we need to inject this component
            //in the document even if it still have to load data.
            if (!this.issueSearch) {
                this._buildIssueSearch();
                this._activateIssueSearchPage();
                API.useImplementation(this._getAPIImplementation(this.issueSearch));
                this.issueSearch.show();
            }

            if (this.state.has('searchResults')) {
                // If we have searchResults, it means that we are still in the same project and the stable search is
                // still valid, so we can load it.
                this.issueSearch.load();
            } else if (this._stateBeforeExpand) {
                // If we don't have searchResults but we have stateBeforeExpand, it means that the IssueView navigated
                // to an issue outside our project. In that case, restore the original search results and load them.
                this.state.set({
                    'searchResults': this._stateBeforeExpand.get('searchResults'),
                    'issue': this._stateBeforeExpand.get('issue'),
                    'filter': this._stateBeforeExpand.get('filter')
                });
                delete this._stateBeforeExpand;
                this.issueSearch.load();
            } else {
                // We don't have any previous state to restore, so just do a new search.
                this.issueSearch.search({
                    filter: params.filter || Filters.getDefaultFilterId(),
                    issue: params.issue || null,
                    orderby: params.orderby || null,
                    startIndex: params.startIndex || null
                });
            }
        },

        _destroyIssueSearch: function() {
            if (this.issueSearch) {
                this.issueSearch.destroy();
                this.stopListening(this.issueSearch);
                delete this.issueSearch;
            }
        },

        _activatePage: function(page) {
            this.el.find(".content").empty().append(page.el);
        },

        _scrollToAnchor: function () {
            var anchor = URLHelper.getAnchor();
            Browser.scrollIntoView(anchor);
        },

        _buildRouter: function () {
            this.router = new Router();

            this.listenTo(this.router, {
                "route:browse:initial": function(params) {
                    Metrics.startIssueViewFromPageLoad();
                    this.state.update(params);
                    this._showIssueView({
                        existingMarkup: WRMData.claim("com.atlassian.jira.jira-projects-issue-navigator:server-rendered"),
                        issueKey: params.issue
                    });
                    this._scrollToAnchor();
                },
                "route:browse": function(params) {
                    Metrics.startIssueView();
                    if (!params.filter) {
                        this.state.unset('searchResults');
                    }
                    this.state.update(params);
                    this._showIssueView({
                        existingMarkup: false,
                        issueKey: params.issue
                    });
                    DialogsCleaner.clean();
                    this._scrollToAnchor();
                },
                "route:issues:initial": function (params) {
                    Metrics.startIssueSearchFromPageLoad();
                    params.filter = Filters.sanitiseFilter(params.filter);
                    this.state.update(params);
                    this._showIssueSearch(params);
                    Analytics.filterSelect({
                        filterId: params.filter,
                        source: 'url'
                    });
                },
                "route:issues": function (params) {
                    Metrics.startIssueSearch();
                    this.state.unset('searchResults');
                    params.filter = Filters.sanitiseFilter(params.filter);
                    this.state.update(params);
                    this._showIssueSearch(params);
                    Analytics.filterSelect({
                        filterId: params.filter,
                        source: 'url'
                    });
                    DialogsCleaner.clean();
                }
            });
        },

        _getAPIImplementation: function(page) {
            var app = this;
            var api = {
                viewSelectedIssue: function() {
                    // If the IssueView is already visible, this is a NOOP.
                    if (!app.issueView) {
                        Metrics.startIssueView();
                        app.state.unset('searchResults');
                        app.state.unset('filter');
                        app._showIssueView({
                            issueKey: app.issueSearch.getIssueKey()
                        });
                    }
                },
                returnToSearch: function() {
                    // If the IssueSearch is already visible, this is a NOOP.
                    if (!app.issueSearch) {
                        Metrics.startIssueSearch();
                        app._showIssueSearch({
                            issue: app.state.get('issue')
                        });
                    }
                },
                isFullScreenIssueVisible: function() {
                    return Boolean(app.issueView);
                },
                isQueryValid: function() {
                    // As there are no custom queries, all queries are valid
                    return true;
                }
            };

            return _.extend({},
                API.getDefaultImplementation(),
                page.getAPIImplementation(),
                api
            );
        },

        start: function() {
            jQuery(document.body).addClass("page-type-split page-issue-navigator");

            // Build components
            this._buildRouter();
            this.state = new State();
            this.jiraEvents = new JIRAEvents();
            this._buildSearch();
            this._buildKeyboardShortuts();

            // Initialize services
            Filters.initialize();
            API.init();
            this.router.start();

            // Adjust components size on scroll/resize
            var pendingAdjustSize;
            jQuery(window).on('scroll resize', _.bind(function () {
                if (!pendingAdjustSize) {
                    pendingAdjustSize = requestAnimationFrame(_.bind(function () {
                        this.adjustSize();
                        pendingAdjustSize = null;
                    }, this));
                }
            }, this));
            this.adjustSize();
        },

        adjustSize: function() {
            if (this.issueView) this.issueView.adjustSize();
            if (this.issueSearch) this.issueSearch.adjustSize();
        }
    });
});
