AJS.test.require("com.atlassian.jira.jira-projects-issue-navigator:application-test", function () {
    "use strict";

    require([
        "jira/components/search/results",
        "jira/projectissuenavigator/services/router",
        "jira/projects/test-utils/marionettemocker",
        "jira/projects/test-utils/mockutils",
        "jira/projectissuenavigator/entities/navigatorstate",
        "jira/projectissuenavigator/pages/issuesearch",
        "jira/projectissuenavigator/pages/issueview",
        "jira/projectissuenavigator/services/filters",
        "jira/projectissuenavigator/services/metrics",
        "jira/projectissuenavigator/services/analytics",
        "jira/projectissuenavigator/services/traces",
        "jira/projectissuenavigator/services/keyboardshortcuts",
        "jira/projectissuenavigator/services/jiraevents",
        "jira/projectissuenavigator/services/urlhelper",
        'jira/projectissuenavigator/services/browser',
        'jira/projectissuenavigator/services/dialogscleaner',
        "wrm/data",
        "jquery"
    ], function (
        Results,
        Router,
        MarionetteMocker,
        MockUtils,
        State,
        IssueSearch,
        IssueView,
        Filters,
        Metrics,
        Analytics,
        Traces,
        KeyboardShortcuts,
        JIRAEvents,
        URLHelper,
        Browser,
        DialogsCleaner,
        WrmData,
        $
    ) {
        var WRM_ISSUE_NAV_SERVER_RENDERED_KEY = "com.atlassian.jira.jira-projects-issue-navigator:server-rendered";

        module("jira/projectissuenavigator", {
            setup: function() {
                this.sandbox = sinon.sandbox.create();

                this.router = MarionetteMocker.createEventedMock(this.sandbox, Router);
                this.issueSearch = MarionetteMocker.createEventedMock(this.sandbox, IssueSearch);
                this.issueView = MarionetteMocker.createEventedMock(this.sandbox, IssueView);
                this.metrics = MarionetteMocker.createEventedMock(this.sandbox, Metrics);
                this.filters = this.sandbox.stub(Filters);
                this.analytics = this.sandbox.stub(Analytics);
                this.traces = this.sandbox.stub(Traces);
                this.jiraEvents = MarionetteMocker.createEventedMock(this.sandbox, JIRAEvents);
                this.keyboardShortcuts = MarionetteMocker.createEventedMock(this.sandbox, KeyboardShortcuts);
                this.urlHelper = this.sandbox.stub(URLHelper);
                this.browser = this.sandbox.stub(Browser);

                // This function will be called with 'new', so .bind() will not have effect here.
                // Only save the first instance of the State
                var that = this;
                this.stateConstructor = function(options) {
                    if (that.state) {
                        return new State(options);
                    } else {
                        that.state = new State(options);
                        return that.state;
                    }
                };

                this.ProjectIssueNavigatorContructor = MockUtils.spyAll(this.sandbox, MockUtils.requireWithMocks("jira/projectissuenavigator", {
                    "jira/projectissuenavigator/services/router": this.router.constructor,
                    "jira/projectissuenavigator/pages/issuesearch": this.issueSearch.constructor,
                    "jira/projectissuenavigator/pages/issueview": this.issueView.constructor,
                    "jira/projectissuenavigator/services/metrics": this.metrics,
                    "jira/projectissuenavigator/services/analytics": this.analytics,
                    "jira/projectissuenavigator/services/traces": this.traces,
                    "jira/projectissuenavigator/services/filters": this.filters,
                    "jira/projectissuenavigator/services/jiraevents": this.jiraEvents.constructor,
                    "jira/projectissuenavigator/services/keyboardshortcuts": this.keyboardShortcuts.constructor,
                    "jira/projectissuenavigator/entities/navigatorstate": this.stateConstructor,
                    "jira/projectissuenavigator/services/urlhelper": this.urlHelper,
                    "jira/projectissuenavigator/services/browser": this.browser,
                    "jira/projectissuenavigator/services/dialogscleaner": DialogsCleaner
                }));

                this.filters.getDefaultFilterId.returns('allopenissues');

                this.projectIssueNavigator = new this.ProjectIssueNavigatorContructor();
                this.projectIssueNavigator.start();

                this.issueView.getIssueKey.returns("DEMO-123");
                this.issueSearch.getIssueKey.returns("DEMO-123");

                this.issueView.load.returns(new $.Deferred().resolve().promise());
                this.issueView.loadFromPage.returns(new $.Deferred().resolve().promise());

                this.issueSearch.constructor.reset();
                this.issueSearch.show.reset();

                this.sandbox.stub(WrmData, "claim");
            },

            teardown: function() {
                this.sandbox.restore();
                $("body").removeClass("page-type-split");
                $("body").removeClass("page-issue-navigator");
            },

            assertSearch: function(params) {
                sinon.assert.calledOnce(this.issueSearch.search);
                sinon.assert.calledWith(this.issueSearch.search, params);
            },

            goToIssue: function(issueKey, filter) {
                this.router.trigger("route:browse", {
                    issue: issueKey,
                    filter: filter || null
                });
            },

            goToSearch: function() {
                this.router.trigger("route:issues", {
                    filter: 'allissues',
                    issue: 'DEMO-1',
                    orderby: 'key DESC',
                    startIndex: "3"
                });
                this.state.set('issue', 'DEMO-1');
                this.state.set('filter', 'allissues');
                this.state.set('orderby', 'key DESC');
                this.state.update({searchResults: new Results([
                    {id: 1, key: 'DEMO-1'}
                ], {})});
            },

            returnToSearch: function() {
                this.issueView.trigger('collapse');
            },

            deleteIssue: function(issueKey, issueId) {
                this.issueView.trigger('issueDelete', { issueKey: issueKey, issueId: issueId });
            },

            assertAnalytics: function(eventName, eventAttributes) {
                sinon.assert.calledOnce(this.analyticsSpy);

                var payload = {
                    name: eventName
                };
                if (eventAttributes) {
                    payload.data = eventAttributes;
                }

                sinon.assert.calledWithMatch(this.analyticsSpy, sinon.match.any, payload);
            }
        });


        test("When the router triggers the event 'route:browse:initial', the PIN records metrics for loading the IssueView from page load", function() {
            this.router.trigger("route:browse:initial", {});

            sinon.assert.calledOnce(this.metrics.startIssueViewFromPageLoad);
            sinon.assert.calledOnce(this.metrics.endIssueView);
            sinon.assert.notCalled(this.metrics.startIssueView);
        });

        test("When the router triggers the event 'route:browse', the PIN records metrics for loading the IssueView from a single page app interaction", function() {
            this.router.trigger("route:browse", {});

            sinon.assert.calledOnce(this.metrics.startIssueView);
            sinon.assert.calledOnce(this.metrics.endIssueView);
            sinon.assert.notCalled(this.metrics.startIssueViewFromPageLoad);
        });

        test("When the router triggers the event 'route:issues:initial', the PIN records metrics for loading the IssueSearch from page load", function() {
            this.router.trigger("route:issues:initial", {});

            sinon.assert.calledOnce(this.metrics.startIssueSearchFromPageLoad);
            sinon.assert.notCalled(this.metrics.startIssueSearch);
        });

        test("When the router triggers the event 'route:issues', the PIN records metrics for loading the IssueSearch from a single page app interaction", function() {
            this.router.trigger("route:issues", {});

            sinon.assert.calledOnce(this.metrics.startIssueSearch);
            sinon.assert.notCalled(this.metrics.startIssueSearchFromPageLoad);
        });

        test("When the Router triggers a 'route:issues' event with filter, issue and orderby, it loads new issues with the parameters from the event", function() {
            this.filters.sanitiseFilter.returns('allissues');
            this.router.trigger("route:issues", {
                filter: 'allissues',
                issue: 'DEMO-1',
                orderby: 'key DESC',
                startIndex: "3"
            });

            this.assertSearch({
                filter: 'allissues',
                issue: 'DEMO-1',
                orderby: 'key DESC',
                startIndex: "3"
            });
        });

        test("When the Router triggers a 'route:issues' event without a filter, it loads new issues with the default filter", function() {
            this.filters.sanitiseFilter.returns('allopenissues');
            this.router.trigger("route:issues", {
                issue: 'DEMO-1',
                orderby: 'key DESC',
                startIndex: "3"
            });

            this.assertSearch({
                filter: 'allopenissues',
                issue: 'DEMO-1',
                orderby: 'key DESC',
                startIndex: "3"
            });
        });

        test("When the Router triggers a 'route:issues' event without a issuekey, it loads new issues with the rest of the parameters", function() {
            this.filters.sanitiseFilter.returns('allissues');
            this.router.trigger("route:issues", {
                filter: 'allissues',
                orderby: 'key DESC',
                startIndex: "3"
            });

            this.assertSearch({
                filter: 'allissues',
                issue: null,
                orderby: 'key DESC',
                startIndex: "3"
            });
        });

        test("When the Router triggers a 'route:issues' event without an orderby, it loads new issues with the rest of the parameters", function() {
            this.filters.sanitiseFilter.returns('allissues');
            this.router.trigger("route:issues", {
                filter: 'allissues',
                issue: 'DEMO-1',
                startIndex: "3"
            });

            this.assertSearch({
                filter: 'allissues',
                orderby: null,
                issue: 'DEMO-1',
                startIndex: "3"
            });
        });

        test("When the Router triggers a 'route:issues' event and the IssueView page is selected, it loads the IssueSearch", function() {
            this.goToIssue('DEMO-123');
            var onFilterSelected = this.spy();
            this.projectIssueNavigator.on("filterSelected", onFilterSelected);

            this.router.trigger("route:issues", {
                filter: 'allissues',
                issue: 'DEMO-1',
                orderby: 'key DESC',
                startIndex: "3"
            });

            sinon.assert.calledOnce(this.issueView.destroy);
            sinon.assert.calledOnce(this.issueSearch.constructor);
            sinon.assert.calledOnce(this.issueSearch.show);
        });

        test("When the Router triggers a 'route:issues' event, it cleans the dialogs", function() {
            this.stub(DialogsCleaner, "clean");

            this.router.trigger("route:issues", {});

            sinon.assert.calledOnce(DialogsCleaner.clean);
        });

        test("When the Router triggers a 'route:browse' event it scrolls to the anchor selected in the URL", function () {
            this.urlHelper.getAnchor.returns('comment-10000');

            this.router.trigger("route:browse", {});

            sinon.assert.calledOnce(this.browser.scrollIntoView);
            sinon.assert.calledWith(this.browser.scrollIntoView, 'comment-10000');
        });

        test("When the Router triggers a 'route:browse' event, it cleans the dialogs", function() {
            this.stub(DialogsCleaner, "clean");

            this.router.trigger("route:browse", {});

            sinon.assert.calledOnce(DialogsCleaner.clean);
        });

        test("When the Router triggers a 'route:browse:initial' event it scrolls to the anchor selected in the URL", function () {
            this.urlHelper.getAnchor.returns('comment-10000');

            this.router.trigger("route:browse:initial", {});

            sinon.assert.calledOnce(this.browser.scrollIntoView);
            sinon.assert.calledWith(this.browser.scrollIntoView, 'comment-10000');
        });

        test("When the Router triggers a 'route:browse:initial' event with com.atlassian.jira.jira-projects-issue-navigator:server-rendered unset then ISSUE_REFRESHED event is triggered", function () {
            WrmData.claim.withArgs(WRM_ISSUE_NAV_SERVER_RENDERED_KEY).returns(undefined);
            this.router.trigger("route:browse:initial", {});
            sinon.assert.calledOnce(this.jiraEvents.triggerRefreshIssuePage);
        });

        test("When the Router triggers a 'route:browse:initial' event with com.atlassian.jira.jira-projects-issue-navigator:server-rendered set to true no ISSUE_REFRESHED event is triggered", function () {
            WrmData.claim.withArgs(WRM_ISSUE_NAV_SERVER_RENDERED_KEY).returns(true);
            this.router.trigger("route:browse:initial", {});
            sinon.assert.notCalled(this.jiraEvents.triggerRefreshIssuePage);
        });

        test("When linking to an issue from IssueSearch, it records browser metrics for an issue load", function() {
            this.goToSearch();
            this.issueSearch.trigger('linkToIssue', {issueKey: 'DEMO-123'});

            sinon.assert.calledOnce(this.metrics.startIssueView);
            sinon.assert.calledOnce(this.metrics.endIssueView);
        });

        test("When linking to an issue from IssueView, it starts recording the browser metrics for an issue load", function() {
            this.goToIssue("DEMO-122");
            this.metrics.startIssueView.reset();

            this.issueView.trigger('linkToIssue', {issueKey: 'DEMO-123'});

            sinon.assert.calledOnce(this.metrics.startIssueView);
        });

        test("When IssueView triggers the 'updateUrl' event, it updates the URL", function() {
            var options = {issueKey: 'DEMO-123'};

            this.goToIssue('DEMO-123');
            this.issueView.trigger('linkToIssue', options);

            this.issueView.trigger('updateUrl', options);

            sinon.assert.calledOnce(this.router.updateURL);
            sinon.assert.calledWith(this.router.updateURL, sinon.match({
                issue: options.issueKey
            }));
        });

        test("When IssueView triggers the 'updateUrl' event first time, it doesn't pushState", function() {
            var options = {issueKey: 'DEMO-123'};

            this.goToIssue('DEMO-123');
            this.issueView.trigger('linkToIssue', options);

            this.issueView.trigger('updateUrl', options);

            sinon.assert.calledOnce(this.router.updateURL);
            sinon.assert.calledWith(this.router.updateURL, sinon.match({
                issue: options.issueKey,
                replaceHistory: true
            }));
        });

        test("When IssueView triggers the 'updateUrl' event second time, it does pushState", function() {
            var options = {issueKey: 'DEMO-123'};
            var options2 = {issueKey: 'DEMO-234'};

            this.goToIssue('DEMO-123');
            this.issueView.trigger('linkToIssue', options);

            this.issueView.trigger('updateUrl', options);

            sinon.assert.calledOnce(this.router.updateURL);
            sinon.assert.calledWith(this.router.updateURL, sinon.match({
                issue: options.issueKey,
                replaceHistory: true
            }));

            this.issueView.trigger('linkToIssue', options2);
            this.issueView.trigger('updateUrl', options2);

            sinon.assert.calledTwice(this.router.updateURL);
            sinon.assert.calledWith(this.router.updateURL, sinon.match({
                issue: options2.issueKey,
                replaceHistory: false
            }));
        });

        test("When loading an issue in IssueView, it stops recording the browser metrics for an issue load", function() {
            this.goToIssue("DEMO-122");
            this.issueView.trigger('linkToIssue', {issueKey: 'DEMO-123'});
            this.metrics.endIssueView.reset();

            this.issueView.trigger('editorLoaded');

            sinon.assert.calledOnce(this.metrics.endIssueView);
        });

        test("When the editor is ready in the IssueSearch, it triggers a trace", function() {
            this.goToSearch();
            this.issueSearch.trigger("editorLoaded");

            sinon.assert.calledOnce(this.traces.traceEditorLoaded);
        });

        test("When the editor is ready in the IssueSearch, it emits the JIRA Events used by other plugins", function() {
            this.goToSearch();
            this.issueSearch.getIssueId.returns(10001);

            this.issueSearch.trigger("editorLoaded");

            sinon.assert.calledOnce(this.jiraEvents.triggerNewContentAddedFromPanelRefreshed);
            sinon.assert.calledOnce(this.jiraEvents.triggerRefreshIssuePage);
            sinon.assert.calledWith(this.jiraEvents.triggerRefreshIssuePage, 10001);
        });

        test("When the editor is ready in the IssueSearch and ISSUE_REFRESHED event is triggered by lower layers, it emits the JIRA Events used by other plugins without triggerRefreshIssuePage", function() {
            this.goToSearch();

            this.issueSearch.trigger("editorLoaded", {issueRefreshedEvent: true});

            sinon.assert.calledOnce(this.jiraEvents.triggerNewContentAddedFromPanelRefreshed);
            sinon.assert.notCalled(this.jiraEvents.triggerRefreshIssuePage);
        });

        test("When the editor is ready in the IssueSearch, it should trigger project nav view issue analytics event", function() {
            this.issueSearch.getIssueId.returns(10001);
            this.issueSearch.getIssueKey.returns("DEMO-123");
            this.issueSearch.getProjectKey.returns("DEMO");
            this.issueSearch.getProjectId.returns(2);
            this.issueSearch.getProjectType.returns('software');
            this.goToSearch();

            this.issueSearch.trigger("editorLoaded");

            sinon.assert.calledOnce(this.analytics.triggerProjectNavViewIssue);
            sinon.assert.calledWith(this.analytics.triggerProjectNavViewIssue, {
                issueId: 10001,
                issueKey: "DEMO-123",
                projectId: 2,
                projectKey: "DEMO",
                projectType: 'software'
            });
        });

        test("When the user enters non-existent filterId through the URL, then the default filterId is used", function() {
            this.filters.sanitiseFilter.returns('allopenissues');
            this.router.trigger("route:issues", {
                filter: 'invalidFilter'
            });

            this.assertSearch({
                filter: 'allopenissues',
                orderby: null,
                issue: null,
                startIndex: null
            });
        });

        test("When going to an issue, it loads the IssueView for that issue", function() {
            this.goToSearch();
            this.goToIssue('DEMO-123');

            sinon.assert.calledOnce(this.issueSearch.destroy);
            sinon.assert.calledOnce(this.issueView.constructor);
            sinon.assert.calledOnce(this.issueView.render);
            sinon.assert.calledOnce(this.issueView.load);
            sinon.assert.calledWith(this.issueView.load, "DEMO-123");
        });

        test("When going to an issue, it triggers a trace when the IssueView is ready ", function() {
            this.issueView.getIssueId.returns(10001);
            this.goToIssue('DEMO-123');

            sinon.assert.calledOnce(this.traces.traceEditorLoaded);
        });

        test("When going to an issue, it emits the JIRA Events used by other plugins", function() {
            this.issueView.getIssueId.returns(10001);
            this.goToIssue('DEMO-123');

            sinon.assert.calledOnce(this.jiraEvents.triggerNewContentAddedFromPanelRefreshed);
            sinon.assert.calledOnce(this.jiraEvents.triggerRefreshIssuePage);
            sinon.assert.calledWith(this.jiraEvents.triggerRefreshIssuePage, 10001);
        });

        test("When the editor is loaded after going to an issue, it triggers a trace when the IssueView is ready", function() {
            this.issueView.getIssueId.returns(10001);
            this.goToIssue('DEMO-123');
            this.traces.traceEditorLoaded.reset();

            this.issueView.trigger("editorLoaded");

            sinon.assert.calledOnce(this.traces.traceEditorLoaded);
        });

        test("When the editor is loaded after going to an issue, it should trigger project nav view issue analytics event", function() {
            this.issueView.getIssueId.returns(10001);
            this.issueView.getProjectKey.returns("DEMO");
            this.issueView.getProjectId.returns(2);
            this.issueView.getProjectType.returns('software');
            this.goToIssue('DEMO-123');
            this.analytics.triggerProjectNavViewIssue.reset();

            this.issueView.trigger("editorLoaded");

            sinon.assert.calledOnce(this.analytics.triggerProjectNavViewIssue);
            sinon.assert.calledWith(this.analytics.triggerProjectNavViewIssue, {
                issueId: 10001,
                issueKey: "DEMO-123",
                projectId: 2,
                projectKey: "DEMO",
                projectType: "software"
            });
        });

        test("When the editor is loaded after going to an issue, it emits the JIRA Events used by other plugins", function() {
            this.issueView.getIssueId.returns(10001);
            this.goToIssue('DEMO-123');
            this.jiraEvents.triggerNewContentAddedFromPanelRefreshed.reset();
            this.jiraEvents.triggerRefreshIssuePage.reset();

            this.issueView.trigger("editorLoaded");

            sinon.assert.calledOnce(this.jiraEvents.triggerNewContentAddedFromPanelRefreshed);
            sinon.assert.calledOnce(this.jiraEvents.triggerRefreshIssuePage);
        });

        test("When returning to search, it loads the IssueSearch", function() {
            this.goToIssue('DEMO-123');

            this.returnToSearch();

            sinon.assert.calledOnce(this.issueView.destroy);
            sinon.assert.calledOnce(this.issueSearch.constructor);
            sinon.assert.calledOnce(this.issueSearch.show);
        });

        test("When returning to search, it records browser metrics for a normal search load", function() {
            this.goToIssue('DEMO-123');

            this.returnToSearch();

            sinon.assert.calledOnce(this.metrics.startIssueSearch);
        });

        test("When going to an issue, it records browser metrics for an issue load", function() {
            this.goToIssue('DEMO-123');

            sinon.assert.calledOnce(this.metrics.startIssueView);
            sinon.assert.calledOnce(this.metrics.endIssueView);
        });

        test("When the Issue Search triggers a 'filterSelected' event it triggers an analytics event", function() {
            this.goToSearch();
            this.analytics.filterSelect.reset();
            this.issueSearch.trigger("filterSelected", {
                filterId: 'allissues',
                source: 'subnavigator'
            });

            sinon.assert.calledOnce(this.analytics.filterSelect);
            sinon.assert.calledWith(this.analytics.filterSelect, {
                filterId: 'allissues',
                source: 'subnavigator'
            });
        });

        test("When the Router triggers a 'route:issues' event, it triggers an analytics event", function() {
            this.filters.sanitiseFilter.returns('allissues');
            this.router.trigger("route:issues", {
                filter: 'allissues',
                issue: 'DEMO-1',
                orderby: 'key DESC',
                startIndex: "3"
            });

            sinon.assert.calledOnce(this.analytics.filterSelect);
            sinon.assert.calledWith(this.analytics.filterSelect, {
                filterId: 'allissues',
                source: 'url'
            });
        });

        test("When the Issue Search triggers a 'goToFullIssueNavigator' event it triggers an analytics event", function() {
            this.goToSearch();
            this.issueSearch.trigger("goToFullIssueNavigator");

            sinon.assert.calledOnce(this.analytics.goToFullIssueNavigator);
        });

        test("When the Issue Search triggers a 'issueSelectedInList' event it triggers an analytics event", function() {
            this.goToSearch();
            this.issueSearch.trigger("issueSelectedInList", {
                key: 'KEY-123'
            });

            sinon.assert.calledOnce(this.analytics.issueSelectedInList);
            sinon.assert.calledWith(this.analytics.issueSelectedInList, {
                key: 'KEY-123'
            });
        });

        test("When the Issue Search triggers a 'manageFilters' event it triggers an analytics event", function() {
            this.goToSearch();
            this.issueSearch.trigger("manageFilters");

            sinon.assert.calledOnce(this.analytics.manageFilters);
        });

        test("When the Issue Search triggers a 'pager:next' event it triggers an analytics event", function() {
            this.goToSearch();
            this.issueSearch.trigger("pager:next", {
                current: 2,
                total: 4
            });

            sinon.assert.calledOnce(this.analytics.pager);
            sinon.assert.calledWith(this.analytics.pager, 'next', {
                current: 2,
                total: 4
            });
        });

        test("When the Issue Search triggers a 'pager:previous' event it triggers an analytics event", function() {
            this.goToSearch();
            this.issueSearch.trigger("pager:previous", {
                current: 2,
                total: 4
            });

            sinon.assert.calledOnce(this.analytics.pager);
            sinon.assert.calledWith(this.analytics.pager, 'previous', {
                current: 2,
                total: 4
            });
        });

        test("When issue has been deleted in IssueView, it shows IssueSearch and removes the deleted issue from it", function() {
            var searchResults = new Results([
                {id: 1, key: 'DEMO-1'}
            ], {});
            this.state.update({searchResults: searchResults});
            this.stub(searchResults, "removeAndUpdateSelectionIfNeeded").returns(new $.Deferred().resolve().promise());

            this.goToIssue('DEMO-1', 'myopenissues');
            this.deleteIssue('DEMO-1', 1);

            sinon.assert.calledOnce(this.metrics.startIssueSearch);
            sinon.assert.calledOnce(this.issueSearch.constructor);
            sinon.assert.calledOnce(searchResults.removeAndUpdateSelectionIfNeeded);
            sinon.assert.calledWith(searchResults.removeAndUpdateSelectionIfNeeded, searchResults.get(1));
        });

        test("When issue has been deleted in navSearch, it removes the deleted issue from it", function() {
            this.state.update({searchResults: null});

            this.goToIssue('DEMO-1');
            this.deleteIssue('DEMO-1', 1);

            sinon.assert.calledOnce(this.metrics.startIssueSearch);
            sinon.assert.calledOnce(this.issueSearch.constructor);
        });

        test("When the IssueSearch emits a issue:select event, the URL is updated to reflect the new issue", function() {
            this.goToSearch();
            this.state.update({
                filter: 'myfilter',
                orderby: 'priority DESC',
                issue: "KEY-123"
            });

            this.issueSearch.trigger("issue:select", {
                issue: "KEY-123"
            });

            sinon.assert.calledOnce(this.router.updateURL);
            sinon.assert.calledWith(this.router.updateURL, sinon.match({
                filter: 'myfilter',
                orderby: 'priority DESC',
                issue: "KEY-123"
            }));
        });

        test("When the IssueSearch emits a issue:empty event, the URL is updated to discard the issue", function() {
            this.goToSearch();
            this.state.update({
                filter: 'myfilter',
                orderby: 'priority DESC',
                issue: null
            });

            this.issueSearch.trigger("issue:empty");

            sinon.assert.calledOnce(this.router.updateURL);
            sinon.assert.calledWith(this.router.updateURL, sinon.match({
                filter: 'myfilter',
                orderby: 'priority DESC'
            }).and(sinon.match(function(obj) {
                return !(obj.issue);
            })));
        });
        test("When the IssueSearch emits a issue:select event, the new URL replaces the History", function() {
            this.goToSearch();

            this.issueSearch.trigger("issue:select", {issue: "KEY-123"});

            sinon.assert.calledOnce(this.router.updateURL);
            sinon.assert.calledWith(this.router.updateURL, sinon.match({
                replaceHistory: true
            }));
        });

        test("When the IssueSearch emits a issue:select event for the second time, the new URL does not replace the History", function() {
            this.goToSearch();
            this.issueSearch.trigger("issue:select", {issue: "KEY-123"});
            this.router.updateURL.reset();

            this.issueSearch.trigger("issue:select", {issue: "KEY-123"});

            sinon.assert.calledOnce(this.router.updateURL);
            sinon.assert.calledWith(this.router.updateURL, sinon.match({
                replaceHistory: false
            }));
        });

        test("When the IssueSearch emits a issue:empty event, the new URL replaces the History", function() {
            this.goToSearch();

            this.issueSearch.trigger("issue:empty");

            sinon.assert.calledOnce(this.router.updateURL);
            sinon.assert.calledWith(this.router.updateURL, sinon.match({
                replaceHistory: true
            }));
        });

        test("When the IssueSearch emits a issue:empty event for the second time, the new URL does not replace the History", function() {
            this.goToSearch();
            this.issueSearch.trigger("issue:empty");
            this.router.updateURL.reset();

            this.issueSearch.trigger("issue:empty");

            sinon.assert.calledOnce(this.router.updateURL);
            sinon.assert.calledWith(this.router.updateURL, sinon.match({
                replaceHistory: false
            }));
        });

        test("When the Issue Search triggers a 'editor:saveSuccess' event it triggers an analytics event", function() {
            this.goToSearch();

            this.issueSearch.trigger("editor:saveSuccess", {
                issueId: '123456',
                savedFieldIds: ['fieldId-1'],
                duration: 123
            });

            sinon.assert.calledOnce(this.analytics.editorFieldSaved);
            sinon.assert.calledWith(this.analytics.editorFieldSaved, {
                issueId: '123456',
                savedFieldIds: ['fieldId-1'],
                duration: 123
            });
        });

        test("When the Issue Search triggers a 'editor:editField' event it triggers an analytics event", function() {
            this.goToSearch();

            this.issueSearch.trigger("editor:editField", {
                issueId: '123456',
                fieldId: 'somefield-1',
                fieldType: 'fieldtype-1'
            });

            sinon.assert.calledOnce(this.analytics.editFieldStarted);
            sinon.assert.calledWith(this.analytics.editFieldStarted, {
                issueId: '123456',
                fieldId: 'somefield-1',
                fieldType: 'fieldtype-1'
            });
        });

        test("When the Issue Search triggers a 'editor:editFieldCancel' event it triggers an analytics event", function() {
            this.goToSearch();

            this.issueSearch.trigger("editor:editFieldCancel", {
                issueId: '123456',
                fieldId: 'somefield-1',
                fieldType: 'fieldtype-1'
            });

            sinon.assert.calledOnce(this.analytics.editFieldCancelled);
            sinon.assert.calledWith(this.analytics.editFieldCancelled, {
                issueId: '123456',
                fieldId: 'somefield-1',
                fieldType: 'fieldtype-1'
            });
        });

        test("When the Issue View is collapsed, it triggers an analytics event", function () {
            this.goToIssue("DEMO-122");

            this.issueView.trigger("collapse");

            sinon.assert.calledOnce(this.analytics.collapseIssue);
        });

        test("When the Issue Detail is expanded, it triggers an analytics event", function () {
            this.goToSearch();

            this.issueSearch.trigger("expand");

            sinon.assert.calledOnce(this.analytics.expandIssue);
        });

        test("When the Issue Detail is expanded, it triggers an issue started event", function () {
            this.goToSearch();

            this.issueSearch.trigger("expand");

            sinon.assert.calledOnce(this.metrics.startIssueView);
        });

        test("When the IssueView navigates away of the project, the current state is preserved on collapse", function() {
            this.goToSearch();
            this.issueSearch.search.reset();
            var originalSearchResults = this.state.get('searchResults');
            var originalIssue = this.state.get('issue');
            var originalFilter = this.state.get('filter');

            this.issueSearch.trigger("expand");
            this.state.unset('searchResults');
            this.state.unset('filter');
            this.issueView.trigger("collapse");

            sinon.assert.notCalled(this.issueSearch.search);
            sinon.assert.calledOnce(this.issueSearch.load);
            equal(this.state.get('searchResults'), originalSearchResults);
            equal(this.state.get('issue'), originalIssue);
            equal(this.state.get('filter'), originalFilter);
        });

        test("When the IssueView renders a individual panel, it triggers a JIRA.Event", function(){
            var panel = jQuery("<div/>");
            this.goToIssue("DEMO-122");
            this.jiraEvents.triggerNewContentAddedFromPanelRefreshed.reset();

            this.issueView.trigger("individualPanelRendered", panel);

            sinon.assert.calledOnce(this.jiraEvents.triggerNewContentAddedFromPanelRefreshed);
            sinon.assert.calledWith(this.jiraEvents.triggerNewContentAddedFromPanelRefreshed, panel);
        });

        test("When the IssueView renders the main view, it triggers two JIRA.Event", function(){
            var panel = jQuery("<div/>");
            var issueId = 10001;
            this.goToIssue("DEMO-122");
            this.issueView.trigger("viewRendered", panel, issueId);

            sinon.assert.calledOnce(this.jiraEvents.triggerNewContentAddedFromPageLoad);
            sinon.assert.calledWith(this.jiraEvents.triggerNewContentAddedFromPageLoad, panel);

            sinon.assert.calledOnce(this.jiraEvents.triggerRefreshToggleBlocks);
            sinon.assert.calledWith(this.jiraEvents.triggerRefreshToggleBlocks, issueId);
        });

        test("When the KeyboardShortcuts trigger a 'focusIssueList' in the IssueSearch page, it focus the list", function() {
            this.goToSearch();

            this.keyboardShortcuts.trigger('focusIssueList');

            sinon.assert.calledOnce(this.issueSearch.focusIssueList);
        });

        test("When the KeyboardShortcuts trigger a 'focusIssueList' in the IssueView page, it does nothing", function() {
            this.goToIssue("DEMO-122");

            this.keyboardShortcuts.trigger('focusIssueList');

            sinon.assert.notCalled(this.issueSearch.focusIssueList);
        });

        test("When the KeyboardShortcuts trigger a 'focusIssueEditor' in the IssueSearch page, it focus the editor", function() {
            this.goToSearch();

            this.keyboardShortcuts.trigger('focusIssueEditor');

            sinon.assert.calledOnce(this.issueSearch.focusIssueEditor);
        });

        test("When the KeyboardShortcuts trigger a 'focusIssueEditor' in the IssueView page, it focus the editor", function() {
            this.goToIssue("DEMO-122");
            this.issueView.focusIssueEditor.reset();

            this.keyboardShortcuts.trigger('focusIssueEditor');

            sinon.assert.calledOnce(this.issueView.focusIssueEditor);
        });
    });
});
