define("jira/issues/views/details", [
    'jira/components/lib/marionette-2.1.0',
    'jira/components/detailslayout',
    'jira/components/search/results',
    'jira/issues/views/details/metrics',
    "jira/issues/views/details/analytics",
    "jira/issues/views/details/traces",
    "jira/issues/views/details/application-adapter",
    'jira/components/libs/uri'
], function(
    Marionette,
    DetailsLayout,
    Results,
    metrics,
    analytics,
    traces,
    applicationAdapter,
    URI
) {
    "use strict";

    return Marionette.Object.extend({
        _buildSearchResults: function(search) {
            this._destroySearchResults();

            var searchResults = search.getResults();
            var keys = searchResults._getIssueIdsToKeys();
            var ids = searchResults.get('issueIds');
            var issues = _.map(ids, function (id) { return {id: id, key: keys[id]}; });

            this.searchResults = new Results([], {
                totalRecordsInDB: searchResults.get('total'),
                pageSize: searchResults.get('pageSize'),
                totalRecordsInSearch: searchResults.get('issueIds').length,
                issues: issues,
                jql: search.getEffectiveJql(),
                allowNoSelection: true
            });
        },

        _destroySearchResults: function() {
            if (this.searchResults) {
                this.stopListening(this.searchResults);
                delete this.searchResults;
            }
        },

        _buildDetailsLayout: function(searchPageModule) {
            var currentQuery = new URI().query();
            var url = new URI(AJS.contextPath() + "/issues/").query(currentQuery).removeQuery("startIndex");
            this.detailsLayout = new DetailsLayout({
                baseURL: url.toString(),
                shouldUpdateCurrentProject: false
            });
            var boundAdjustSize = _.bind(this.detailsLayout.adjustSize, this.detailsLayout);

            this.listenTo(this.detailsLayout, {
                "select": function(issueData) {
                    if (issueData.id) {
                        var shouldReplaceCurrentURLInTheHistory = false;
                        var isThereAnIssueInTheCurrentURL = this.search.getResults().getSelectedIssue().getKey();
                        if (!isThereAnIssueInTheCurrentURL) {
                            shouldReplaceCurrentURLInTheHistory = true;
                        }
                        this.search.getResults().selectIssueById(issueData.id, {replace: shouldReplaceCurrentURLInTheHistory});
                    }
                },
                "list:refresh": function() {
                    metrics.start();
                    analytics.triggerIssueTableRefresh();
                    this.search.refresh();
                },
                "list:sort": function(jql) {
                    this.search.doSort(jql);
                },
                "list:update": function() {
                    traces.triggerSearchFinished();
                    traces.triggerStableUpdate();
                },
                "list:render": function() {
                    this.trigger("render");
                },
                "empty": function() {
                    traces.triggerSearchFinished();
                    this.trigger("render");
                },
                "destroy": function() {
                    JIRA.Issues.offHorizontalResize(boundAdjustSize);
                    JIRA.Issues.offVerticalResize(boundAdjustSize);
                },
                "list:select": function(event) {
                    metrics.start();
                    analytics.triggerOpenIssueFromTable({
                        issueId: event.id,
                        absolutePosition: event.absolutePosition,
                        relativePosition: event.relativePosition
                    });
                },
                "list:pagination": function() {
                    metrics.start();
                },
                "editorLoaded": function(event) {
                    metrics.end();
                    analytics.triggerViewIssue(event);
                },
                "editorLoadedFromCache": function() {
                    metrics.end();
                },
                "editor:saveSuccess": function(event) {
                    analytics.triggerInlineEdit({
                        issueId: event.event,
                        savedFieldIds: event.savedFieldIds,
                        savedFieldTypes: event.savedFieldTypes,
                        duration: event.duration
                    });
                },
                "editor:editField": function(event) {
                    analytics.triggerInlineEditStart({
                        issueId: event.issueId,
                        fieldId: event.fieldId,
                        fieldType: event.fieldType
                    });
                },
                "editor:editFieldCancel": function(event) {
                    analytics.triggerInlineEditCancel({
                        issueId: event.issueId,
                        fieldId: event.fieldId,
                        fieldType: event.fieldType
                    });
                },
                "linkToIssue": function(event) {
                    metrics.viewIssue();

                    searchPageModule.reset({
                        selectedIssueKey: event.issueKey
                    });
                }
            });

            JIRA.Issues.onHorizontalResize(boundAdjustSize);
            JIRA.Issues.onVerticalResize(boundAdjustSize);
        },

        _destroyDetailsLayout: function() {
            this.detailsLayout.destroy();
            delete this.detailsLayout;
        },

        _buildSearch: function(search) {
            this.search = search;
            this.listenTo(this.search.getResults(), {
                "change:resultsId": this._loadSearch,
                "issueDeleted": function(issueData) {
                    var deletedIssue = this.searchResults.get(issueData.id);
                    this.searchResults.removeAndUpdateSelectionIfNeeded(deletedIssue);
                }
            });

            // These is *not* a regular Backbone event, we can't use listenTo.
            this.search.getResults().onIssueUpdated(this._handleRefreshIssue);

            this.listenTo(this.search.getResults().getSelectedIssue(), {
                "change": function() {
                    var newIssue = this.search.getResults().getSelectedIssue().get('id');
                    if (!newIssue) return;
                    if (!this.searchResults) return;

                    // The selected issue in this.search.getResults() has changed and it isn't
                    // changed in our model: this means it has been changed 'outside' the DetailsLayout
                    // (e.g. push state or API). In this case, we want reload our model to get the
                    // correct page and select the issue.
                    if (this.searchResults.selected && this.searchResults.selected.get('id') !== newIssue) {
                        this.detailsLayout.load(this.searchResults, newIssue);
                    }
                }
            });
        },

        _destroySearch: function(){
            this.search.getResults().setTable(null, {silent: true});
            this.search.getResults().offIssueUpdated(this._handleRefreshIssue);
        },

        _loadSearch: function() {
            var issueKey;

            this._buildSearchResults(this.search);
            if (this.search.getResults().hasSelectedIssue()) {
                issueKey = this.search.getResults().getSelectedIssue().get('key');
            } else if (this.search.getResults().get('startIndex') > 0) {
                issueKey = this.searchResults.getIssueKeyForIndex(this.search.getResults().get('startIndex'));
            } else if (this.search.getResults().hasHighlightedIssue()) {
                issueKey = this.search.getResults().getHighlightedIssue().get('key');
            }

            this.detailsLayout.load(this.searchResults, issueKey);
        },

        _handleRefreshIssue: function(issueId) {
            this.detailsLayout.refreshIssue(issueId);
        },

        initialize: function(options) {
            this._handleRefreshIssue = _.bind(this._handleRefreshIssue, this);

            this._buildDetailsLayout(options.searchPageModule);
            this._buildSearch(options.search);

            // Create a container, we don't want to replace the existing markup
            var container = jQuery("<div></div>");
            options.searchContainer.find(".navigator-content").html('').append(container);
            this.container = this.detailsLayout.show(container);
            jQuery("body").addClass("page-type-split");

            options.fullScreenIssue.hide();

            applicationAdapter.init(this.detailsLayout);
        },

        render: function() {
            // This method is NOT called when the DetailsLayout is loaded initially. It will get called
            // if the uses switches from another layout (i.e. List View) to Details View.

            this._loadSearch();
        },

        close: function() {
            this.destroy();
        },

        onDestroy: function() {
            jQuery("body").removeClass("page-type-split");
            this._destroyDetailsLayout();
            this._destroySearch();
            this._destroySearchResults();
            applicationAdapter.destroy();
        },

        nextIssue: function() {
            this.detailsLayout.selectNext();
        },

        prevIssue: function() {
            this.detailsLayout.selectPrevious();
        },

        isLoading: function() {
            return this.detailsLayout.isLoading();
        },

        isIssueViewActive: function() {
            return false;
        },

        focusIssueList: function() {
            this.detailsLayout.focusList();
        },

        focusIssueEditor: function() {
            this.detailsLayout.focusEditor();
        }
    });
});
