(function(require) {
    "use strict";

    var $ = require('jquery');
    var _ = require('underscore');
    var Backbone = require('backbone');
    var Meta = require('jira/util/data/meta');
    var issueNavLoader = require('jira/issues/viewissue/issuenavloader');
    var NavigatorMetrics = require('jira/issues/navigator/metrics');
    var URI = require('jira/components/libs/uri');

    NavigatorMetrics.notifyIssueView();

    $(function() {
        var initialState = JIRA.Issues.GlobalIssueNavCreator.readInitialData();

        JIRA.Issues.Application.start({
            useLog: Meta.get("dev-mode") === true,
            showReturnToSearchOnError: function() {
                return JIRA.Issues.LayoutPreferenceManager.getPreferredLayoutKey() !== "split-view";
            }
        });

        var viewIssueComponents = JIRA.Issues.ViewIssueCreator.create($(document), initialState);
        var issueModuleIssue;
        var returnToSearchRequested = false;

        //initialize issues API with a SearchPageModule
        var searchPageModuleReplacement = {
            getEffectiveIssue: function getEffectiveIssue() {
                return _.clone(this.getIssueModel());
            },
            getEffectiveIssueId: function getEffectiveIssueId() {
                return this.getIssueModel().getId();
            },
            getEffectiveIssueKey: function getEffectiveIssueKey() {
                return this.getIssueModel().getKey();
            },
            isCurrentlyLoadingIssue: function isCurrentlyLoadingIssue() {
                return false; //we do not load issue - we use server rendered version
            },
            isHighlightedIssueAccessible: function isHighlightedIssueAccessible() {
                return true;
            },
            isIssueVisible: function isIssueVisible() {
                return true; //fullScreenIssue is always visible
            },
            nextIssue: function nextIssue() {
                return false; //no-op for standalone issue
            },
            openFocusShifter: function openFocusShifter() {
                JIRA.Issues.FocusShifter.show();
            },
            prevIssue: function prevIssue() {
                return false; //no-op for standalone issue
            },
            returnToSearch: function returnToSearch() {
                returnToSearchRequested = true;
            },
            showInlineIssueLoadError: function showInlineIssueLoadError() {
                //no-op - can we have issue load errors?
            },
            isFullScreenIssueVisible: function isFullScreenIssueVisible() {
                return true;
            },
            getCurrentLayout: function getCurrentLayout() {
                return null; //no current layout for fullScreenIssue
            },
            getSortedLayouts: function getSortedLayouts() {
                return [];
            },
            changeLayout: function changeLayout() {
                //no-op for view issue
            },
            update: function update() {
                //this is a no-op for a server rendered issue
            },
            toggleFilterPanel: function toggleFilterPanel() {
                return false; //no-op for standalone issue
            },
            updateIssue: function updateIssue(issueUpdate) {
                return viewIssueComponents.fullScreenIssue.updateIssue(issueUpdate);
            },
            getIssueModel: function getIssueModel() {
                return issueModuleIssue || new JIRA.Issues.SimpleIssue({
                    id: JIRA.Issues.Application.request("issueEditor:getIssueId"),
                    key: JIRA.Issues.Application.request("issueEditor:getIssueKey")
                });
            }
        };

        JIRA.Issues.Api.initialize({
            searchPageModule: searchPageModuleReplacement
        });

        JIRA.Issues.overrideIssueDialogs({
            getIssueId: _.bind(searchPageModuleReplacement.getEffectiveIssueId, searchPageModuleReplacement),
            isNavigator: true,
            updateIssue: function(dialog) {
                var issueUpdate = JIRA.Issues.Utils.getUpdateCommandForDialog(dialog);
                return searchPageModuleReplacement.updateIssue(issueUpdate);
            }
        });

        JIRA.Issues.Application.once('navigation:stateChanged', function(state) {
            viewIssueComponents.fullScreenIssue.show({
                key: state.selectedIssueKey
            });
        });

        JIRA.Issues.Application.on("issueEditor:close", function() {
            returnToSearchRequested = true;
        });

        // save the fragment before it's being replaced by the router
        var fragment = new URI().fragment();

        Backbone.history.start({
            root: JIRA.Issues.IssueNavRouter.getBackboneRoot(),
            pushState: true
        });

        issueNavLoader.initialize(_.extend(initialState, {viewIssueComponents: viewIssueComponents, standaloneIssue: true}));
        issueNavLoader.loadIssueNav()
            .then(function() {
                if (returnToSearchRequested) {
                    JIRA.Issues.Api.returnToSearch();
                }
            });

        // scroll the focused comment into view using the fragment saved before the router changed the URL
        if (fragment) {
            var focusedElement = document.getElementById(fragment);
            if (focusedElement) {
                focusedElement.scrollIntoView();
                // this is needed because the stalker obscures the view. The stalker is usually 135px tall,
                // and 10px for some nice margins
                document.body.scrollTop -= 145;
            }
        }
    });
}(window.require));
