define('jira/issues/viewissue/issuenavloader', [
    'jquery',
    'jira/util/browser'
],
function(
    jQuery,
    Browser
) {
    "use strict";

    /**
     * Utility object for fetching issue search resources. It will handle application state changes (i.e it will remember
     * last state change and repeat it when issue search components are initialized). In case of problems with loading
     * issue search components it will perform a full page reload to given URL.
     */
    return {
        /**
         * Number of ms that IssueNavLoader will wait for initilizing issue search components until it will perform a
         * full page reload.
         *
         * @const
         * @type {Number}
         * @default
         */
        SPA_NAVIGATION_TIMEOUT: 6 * 1000,
        /**
         * Last state that application navigated to.
         *
         * @type {State}
         */
        stateChange: undefined,
        /**
         * Initialize IssueNavLoader.
         *
         * @param {Object} issueNavOptions - parameters that will be passed to IssueNavCreator
         */
        initialize: function initialize(issueNavOptions) {
            this.issueNavOptions = issueNavOptions;
        },
        /**
         * Asynchronously download issue search resources
         *
         * @returns {jQuery.Deferred} promise, that is resolved when web resources are downloaded
         */
        requireIssueNav: function requireIssueNav() {
            var deferred = new jQuery.Deferred();

            WRM.require(['wrc!com.atlassian.jira.jira-issue-nav-plugin:issuenav'], function () {
                deferred.resolve();
            });

            return deferred.promise();
        },
        /**
         * Load issue search components. Any application state change that will happen during loading will be repeated
         * after issue search is initialized, or after certain timeout, will be loaded as a full page reload.
         *
         * @returns {jQuery.Deferred} promise, that is resolved when issue nav is initialized and all previous state changes performed.
         */
        loadIssueNav: function loadIssueNav() {
            JIRA.Issues.Application.on('navigation:stateChanged', this._onStateChanged, this);
            return this.requireIssueNav()
                    .then(function() {
                        JIRA.Issues.GlobalIssueNavCreator.create(jQuery(document), this.issueNavOptions);
                    }.bind(this))
                    .then(this._onIssueNavInitialized.bind(this));
        },
        _onIssueNavInitialized: function onIssueNavInitialized() {
            if (this.stateChange) {
                JIRA.Issues.Application.execute('navigation:updateState', this.stateChange,
                        {replace: true, reset: true, forceRefresh: true});
                this.stateChange = undefined;
            }
            JIRA.Issues.Application.off('navigation:stateChanged', this._onStateChanged);
        },
        _onStateChanged: function _onStateChanged(state) {
            this.stateChange = state; //keep only the last one
            this._triggerNavigationAnalyticsEvent();
            setTimeout(function () {
                //if we haven't navigated yet, let's do a full page reload
                if (this.stateChange) {
                    JIRA.Issues.Application.execute("analytics:trigger", "kickass.viewIssueNavigationPageReload", {
                        navigationTimeout: this.SPA_NAVIGATION_TIMEOUT
                    });
                    var url = AJS.contextPath() + "/" + this.stateChange.toUrl();

                    Browser.reloadViaWindowLocation(url);
                }
            }.bind(this), this.SPA_NAVIGATION_TIMEOUT);
        },
        _triggerNavigationAnalyticsEvent: function _triggerNavigationAnalyticsEvent() {
            var eventTime;
            eventTime = parseInt(window.performance.now() - window.performance.timing.loadEventEnd + window.performance.timing.navigationStart);
            JIRA.Issues.Application.execute("analytics:trigger", "kickass.viewIssueNavigationRequested", {
                navigationTime: eventTime
            });
        }
    };
});
