AJS.test.require([
    'com.atlassian.jira.jira-issue-nav-plugin:viewissue'
], function() {
    "use strict";

    module('IssueNavLoader', {
        setup: function() {
            this.sandbox = sinon.sandbox.create();
            this.context = AJS.test.mockableModuleContext();
            this.reloadSpy = this.sandbox.spy();
            this.context.mock('jira/util/browser', {
                reloadViaWindowLocation: this.reloadSpy
            });
            this.issueNavLoader = this.context.require('jira/issues/viewissue/issuenavloader');
            this.requireCallback = function() {};
            this.sandbox.stub(WRM, 'require', function(deps, cb) {
                if (deps[0] === 'wrc!com.atlassian.jira.jira-issue-nav-plugin:issuenav') {
                    this.requireCallback = cb;
                }
            }.bind(this));
            this.navigationCallback = function() {};
            this.sandbox.stub(JIRA.Issues.Application, 'on', function(event, callback, context) {
                context = context || this;
                if (event === 'navigation:stateChanged') {
                    this.navigationCallback = callback.bind(context);
                }
            }.bind(this));
            this.sandbox.spy(JIRA.Issues.Application, 'execute');
            this.issueNavCreateSpy = this.sandbox.stub(JIRA.Issues.GlobalIssueNavCreator, 'create');
            this.clock = sinon.useFakeTimers();
        },
        teardown: function() {
            this.sandbox.restore();
            this.clock.restore();
        }
    });

    test('Issue Search is initialized after resources are fetched', function() {
        this.issueNavLoader.loadIssueNav();

        ok(this.issueNavCreateSpy.notCalled, 'Issue Search Creator is called');
        this.requireCallback();
        ok(this.issueNavCreateSpy.calledOnce, 'Issue Search Creator is called');
    });

    test('Navigation is repeated after resources are fetched', function() {
        var state = {selectedIssueType: 'TEST-1'};
        var options = {reset: true};
        this.issueNavLoader.loadIssueNav();

        this.navigationCallback(state, options);

        this.requireCallback();
        ok(this.issueNavCreateSpy.calledOnce, 'Issue Search Creator is called');
        ok(JIRA.Issues.Application.execute.calledTwice, 'Navigation is repeated');
        ok(JIRA.Issues.Application.execute.calledWith('analytics:trigger', 'kickass.viewIssueNavigationRequested'), 'Analytics event is triggered');
        ok(JIRA.Issues.Application.execute.calledWithExactly('navigation:updateState', state, {replace: true, forceRefresh: true, reset: true}), 'Navigation is repeated');
    });

    test('Only last navigation is repeated', function() {
        var state = {selectedIssueType: 'TEST-1'};
        var state2 = {selectedIssueType: 'TEST-2', jql: 'JQL'};

        this.issueNavLoader.loadIssueNav();
        this.navigationCallback(state);
        this.navigationCallback(state2);
        this.requireCallback();
        ok(JIRA.Issues.Application.execute.calledThrice, 'Navigation is repeated');
        ok(JIRA.Issues.Application.execute.calledWith('analytics:trigger', 'kickass.viewIssueNavigationRequested'), 'Analytics event is triggered');
        ok(JIRA.Issues.Application.execute.calledWith('navigation:updateState', state2), 'Navigation is repeated');
    });

    test('Full page reload is performed if resources aren\'t fetched', function() {
        var contextPath = 'http://jira';
        var path = 'browse/TEST-1';
        var state = {
            selectedIssueType: 'TEST-1',
            toUrl: this.sandbox.stub().returns(path)
        };

        this.sandbox.stub(AJS, 'contextPath').returns(contextPath);

        this.issueNavLoader.loadIssueNav();
        this.navigationCallback(state);
        this.clock.tick(this.issueNavLoader.SPA_NAVIGATION_TIMEOUT + 100);
        ok(this.reloadSpy.calledOnce, 'Location is changed after timeout');
        ok(this.reloadSpy.calledWithExactly(contextPath + '/' + path), 'Location is changed to proper url');
    });

    test('Full page reload is performed if resources aren\'t fetched', function() {
        var state = {
            selectedIssueType: 'TEST-1',
            toUrl: this.sandbox.stub().returns('')
        };

        this.issueNavLoader.loadIssueNav();
        this.clock.tick(this.issueNavLoader.SPA_NAVIGATION_TIMEOUT + 100);
        ok(this.reloadSpy.notCalled, 'Location is changed only when user requests state change');
        this.navigationCallback(state);
        this.clock.tick(this.issueNavLoader.SPA_NAVIGATION_TIMEOUT - 1);
        ok(this.reloadSpy.notCalled, 'Location is changed only when user requests state change');
        this.clock.tick(1);
        ok(this.reloadSpy.calledOnce, 'Location is changed only when user requests state change');
    });

});
