define('jira/issues/navigator/metrics', ['internal/browser-metrics'], function (metrics) {
    "use strict";

    /**
     * Adds a class to elements matching a selector.
     *
     * Why?
     * ====
     *
     * For single-page-app transitions the content elements *already exist* because they were present for the
     * previous page. To work around this we add a "marker" class that indicates the element is "stale",
     * and use a :not(stale) pattern in our ready selector to ensure we don't match those.
     *
     * When the page is actually updated, the stale elements will be replaced with fresh elements that don't
     * have our marker class.
     *
     * @param selector
     * @param className
     */
    function markAsStale(selector, className) {
        var elements = document.querySelectorAll(selector);
        for (var i = 0; i < elements.length; i++) {
            elements[i].classList.add(className);
        }
    }

    return {
        /**
         * @param isInitial whether to measure metrics from initial page load
         */
        notifyIssueView: function (isInitial) {
            isInitial = Boolean(isInitial);
            var projectKeySelector = "#key-val";
            var contentSelector = ".issue-body-content:not(.browser-metrics-stale)";

            if(!isInitial) {
                markAsStale(contentSelector, "browser-metrics-stale");
            }

            metrics.start({
                isInitial: isInitial,
                key: "jira.issue.view",
                ready: [
                    {selector: projectKeySelector, requireUpdate: false},
                    {selector: contentSelector, requireUpdate: false}
                ]
            });
        },

        notifyIssueNavList: function () {
            var containerSelector = ".navigator-content";
            var contentSelector = "#issuetable:not(.browser-metrics-stale), .error:not(.browser-metrics-stale), .no-results-hint:not(.browser-metrics-stale), #errorPageContainer:not(.browser-metrics-stale)";

            markAsStale(contentSelector, "browser-metrics-stale");

            metrics.start({
                key: "jira.issue.nav-list",
                ready: [
                    {selector: containerSelector, requireUpdate: false},
                    {selector: contentSelector, requireUpdate: false}
                ]
            });
        }
    };
});
