AJS.test.require(["com.atlassian.jira.jira-issue-nav-plugin:testutils", "com.atlassian.jira.jira-issue-nav-plugin:issuenav-page-changes"], function() {
    "use strict";

    var $ = require('jquery');

    //test dialogs are hidden on navigate event
    module("JIRA.Dialog.DialogCleanup", {
        setup: function() {
            this.sandbox = sinon.sandbox.create();
            this.context = AJS.test.mockableModuleContext();

            this.navigationCallback = function() {};
            this.fullPageNavigation = function() {
                this.navigationCallback({}, {fullPageLoad: true});
            };
            this.sandbox.stub(JIRA.Issues.Application, 'on', function(event, callback, context) {
                context = context || this;
                if (event === 'navigation:stateChanged') {
                    this.navigationCallback = callback.bind(context);
                }
            }.bind(this));

            //Mock the active dialog
            this.fakeDialog = new JIRA.Dialog();
            this.sandbox.spy(this.fakeDialog, "hide");
            JIRA.Dialog.current = this.fakeDialog;
        },
        teardown: function() {
            this.sandbox.restore();
            if (JIRA.Dialog.current && JIRA.Dialog.current.hide) {
                JIRA.Dialog.current.hide();
            }
            JIRA.Dialog.current = null;
        }
    });

    test("dialog is hidden on stateChange event", function() {
        //Init the dialogCleaner
        JIRA.Issues.dialogCleaner();

        this.navigationCallback();

        ok(this.fakeDialog.hide.calledOnce, "dialog hide() has been called");
        ok(JIRA.Dialog.current === null, "no dialog is current");
    });

    test("dialog is not hidden on initial stateChange event", function() {
        //Init the dialogCleaner
        JIRA.Issues.dialogCleaner();

        this.fullPageNavigation();

        equal(this.fakeDialog.hide.callCount, 0, "dialog hide() has been called");
        ok(JIRA.Dialog.current === this.fakeDialog, "the fake dialog is still current");
    });


    /**
     * In a BigPipe world, some dialogs would lazily evaluate and load their content.
     * Due to a bug in the WRM (see {@link https://extranet.atlassian.com/jira/browse/APDEX-896},
     * dialogs that load CSS may not complete until document readyState is 'complete' -- well after
     * DomContentLoaded. In the meantime, the view issue application can initialise,
     * set up the dialog cleaner, and then trigger a 'navigate' event, which would invoke the
     * dialog cleanup.
     *
     * We don't want the dialog cleaner to accidentally close dialogs that are legitimately still
     * opening, just because the user did something before DCL fired and the viewissue app started.
     *
     * @see {@link https://extranet.atlassian.com/jira/browse/APDEX-874}
     */
    test("dialog that builds its own content asynchronously should show itself eventually", function() {
        //
        // The setup
        //
        var builderDeferred = new $.Deferred();
        var dialogReady = this.sandbox.spy();
        var builderReady = this.sandbox.spy();
        var builderDone = this.sandbox.spy();

        var requireDeferred = new $.Deferred();
        var wrmRequire = this.sandbox.stub();
        wrmRequire.returns(requireDeferred);

        this.context.mock('wrm/require', wrmRequire);

        var Dialog = this.context.require('jira/dialog/dialog');
        var WRMDialog = Dialog.extend({
            defineResources: function() {
                this._super.apply(this, arguments);
                ['fake-webresource:a', 'fake-webresource:b'].forEach(function(resource) { this.requireResource(resource); }, this);
            }
        });
        var lazyDialog = new WRMDialog({
            id: "laziest-of-dialogs",
            content: function content(callback) {
                var dialog = this;
                builderDeferred.then(function () {
                    callback($('<p>The laziest of content</p>'));
                    builderDone();
                });
                dialog.onContentReady(dialogReady);
                builderReady();
            }
        });

        //
        // The test scenario
        //

        lazyDialog.show();
        JIRA.Dialog.current = lazyDialog; // Need to do this since the global JIRA.Dialog == production jira/dialog/dialog module definition and not our required one.

        ok(Dialog.current === lazyDialog, "(precondition) lazy dialog should be the current dialog, and attempting to open");
        ok(JIRA.Dialog.current === lazyDialog, "(precondition) lazy dialog should be the current dialog, and attempting to open");
        equal(builderReady.notCalled, true);
        equal(builderDone.notCalled, true);
        equal(dialogReady.notCalled, true);
        equal(lazyDialog.get$popup().is(":visible"), false);

        // Now the view issue application initialises,
        // and triggers the dialog cleanup code
        JIRA.Issues.dialogCleaner();
        this.fullPageNavigation();

        ok(JIRA.Dialog.current === lazyDialog, "the dialog should still be opening, because it was doing so before the view issue app started initialising");
        equal(builderReady.notCalled, true);
        equal(builderDone.notCalled, true);
        equal(dialogReady.notCalled, true);
        equal(lazyDialog.get$popup().is(":visible"), false);

        // The resources for the dialog finally load
        requireDeferred.resolve();

        ok(JIRA.Dialog.current === lazyDialog, "the dialog should still be current");
        equal(builderReady.callCount, 1);
        equal(builderDone.notCalled, true);
        equal(dialogReady.notCalled, true);
        equal(lazyDialog.get$popup().is(":visible"), false);

        // The asynchronous content creator finally finishes
        builderDeferred.resolve();

        ok(JIRA.Dialog.current === lazyDialog, "the dialog should still be current");
        equal(builderReady.callCount, 1);
        equal(builderDone.callCount, 1);
        equal(dialogReady.callCount, 1);
        equal(lazyDialog.get$popup().is(":visible"), true);
    });
});
