define('jira/issues/search/filters/filter-panel-view', ['require'], function(require) {
    "use strict";

    var BaseView = require('jira/issues/baseview');
    var Application = require('jira/issues/application');
    var Resize = require('jira/issues/utils/resize');
    var Events = require('jira/util/events');
    var Types = require('jira/util/events/types');
    var Reasons = require('jira/util/events/reasons');
    var KeyboardShortcut = require('jira/ajs/keyboardshortcut/keyboard-shortcut');
    var _ = require('underscore');
    var jQuery = require('jquery');
    var Templates = require('jira/issues/templates/issue-nav-filter');

    /**
     * The collapsible filters panel.
     * <p/>
     * Lists system filters and the user's favourite filters.
     */
    return BaseView.extend({
        events: {
            "simpleClick .system-filters a.filter-link:not(.requires-login)": "_onClickFilter",
            "simpleClick .new-search": "_onClickNewSearch",
            "click .ui-dock": "_onDockFilterPanelClick",
            "click .ui-undock": "_onUndockFilterPanelClick"
        },

        collapsedTemplate: Templates.collapsedFiltersPanel,

        template: Templates.filtersPanel,

        /**
         * @param {object} options
         */
        initialize: function(options) {

            _.bindAll(this, "adjustFavouriteFiltersHeight");
            if (options.easeOff) {
                this._adjustFavouriteFiltersHeight = _.debounce(this._adjustFavouriteFiltersHeight, options.easeOff);
            }
            Resize.onVerticalResize(this.adjustFavouriteFiltersHeight);

            // Resolved when system and favourite filters are rendered
            this.panelReady = new jQuery.Deferred();

            Events.bind(Types.LAYOUT_RENDERED, _.bind(function() {
                this._splitViewSidebarElement = jQuery(".list-results-panel:first");
            }, this));
        },

        /**
         * Adjust the height of the favourite filters list.
         *
         * @private
         */
        adjustFavouriteFiltersHeight: function() {
            var filterList = this.$(".filter-panel-wrapper");
            var filterListTop = filterList.length && filterList.offset().top;
            var windowHeight = window.innerHeight;

            filterList.css("max-height", windowHeight - filterListTop);
        },

        /**
         * When the dock filter link is clicked
         * @private
         */
        _onDockFilterPanelClick: function() {
            jQuery('.navigator-container').removeClass('navigator-sidebar-collapsed');
        },

        /**
         * When the undock filter link is clicked.
         * @private
         */
        _onUndockFilterPanelClick: function() {
            jQuery('.navigator-container').addClass('navigator-sidebar-collapsed');
        },

        toggleDockState: function() {
            this.$el.popoutSidebar("toggle");
        },

        _onClickFilter: function(e) {
            e.preventDefault();
            var filterId = jQuery(e.target).data("id");
            this.trigger("filterSelected", filterId);
        },

        _onClickNewSearch: function(e) {
            e.preventDefault();
            this.trigger("filterSelected", null);
        },

        render: function() {
            if (!this.model.isDocked()) {
                jQuery('.navigator-container').addClass('navigator-sidebar-collapsed');
                this._renderCollapsed();
            } else {
                jQuery('.navigator-container').removeClass('navigator-sidebar-collapsed');
                this._renderOpen();
            }

            var dockingShortcut = KeyboardShortcut.getKeyboardShortcutKeys('toggle.filter.panel');
            this.$el.popoutSidebar({
                isDocked: this.model.isDocked(),
                expandedWidth: this.model.getWidth(),
                renderExpanded: _.bind(this._renderOpen, this),
                renderCollapsed: _.bind(this._renderCollapsed, this),
                undockText: AJS.I18n.getText("issue.nav.filters.undock"),
                undockTitle: AJS.I18n.getText('issue.nav.filters.undock.help', dockingShortcut),
                dockText: AJS.I18n.getText("issue.nav.filters.dock"),
                dockTitle: AJS.I18n.getText('issue.nav.filters.dock.help', dockingShortcut),
                toggleTarget: ".filter-title:first",
                persist: _.bind(function(isDocked, width) {
                    this.model.setDocked(isDocked);
                    this.model.setWidth(width);
                }, this),
                resize: _.bind(function() {
                    if (this._splitViewSidebarElement) this._splitViewSidebarElement.sidebar("updatePosition");
                }, this)
            });

            if (this.model.shouldShowDockIntro()) {
                this.$el.popoutSidebar("collapse", false);
            }

            if (this.model.isDocked()) {
                this.$el.popoutSidebar("renderDockState");
            }
            this._fireDockAnalytics();
            this.panelReady.resolve();

            return this.$el;
        },

        _fireDockAnalytics: function() {
            if (localStorage.getItem("dockStatesAnalyticsEnabled")) {
                var docked = this.model.isDocked();
                if (localStorage.getItem("dockStateAnalytic") !== "" + docked) {
                    localStorage.setItem("dockStateAnalytic", docked);
                    Application.execute("analytics:trigger", "kickass.filters.docked", {
                        isDocked: this.model.isDocked()
                    });
                }
            }
        },

        /**
         * Render the collapsed filter panel.
         *
         * @return {JIRA.Issues.FilterPanelView} <tt>this</tt>
         * @private
         */
        _renderCollapsed: function() {
            var $feedback = jQuery(".jira-feedback-link-container.subtle-style").detach().addClass("left noborder");
            this.$el.addClass("collapsed").html(this.collapsedTemplate({isDocked: this.model.isDocked()}));
            this.trigger("renderCollapsed");
            Resize.triggerHorizontalResize();
            $feedback.appendTo("body");
            return this;
        },

        /**
         * Render the open filter panel.
         *
         * @return {JIRA.Issues.FilterPanelView} <tt>this</tt>
         * @private
         */
        _renderOpen: function() {
            var $feedback = jQuery(".jira-feedback-link-container.subtle-style").detach().removeClass("left noborder");
            this.$el.removeClass("collapsed");

            this.$el.width(this.model.getWidth());
            this.$el.html(this.template({}));

            this.trigger("renderOpen");
            $feedback.appendTo(this.$el);

            Events.trigger(Types.NEW_CONTENT_ADDED, [this.$el, Reasons.filterPanelOpened]);
            Resize.triggerHorizontalResize();
            return this;
        },

        _renderSelectedFilter: function() {
            //FIXME Disabled until both views are moved to FilterComponent
            //this.favouriteFiltersController.activateFilter(this.model.get("activeFilter"));
            //this.systemFilters.activateFilter(this.model.get("activeFilter"));
        }
    });
});
