(function() {
    "use strict";

    var URI = require('jira/components/libs/uri');
    /**
     * Enhance links outside KA, usually to load things via AJAX.
     * <p/>
     * Only works if the AJAX view issue dark feature is enabled.
     */
    JIRA.Issues.enhanceLinks = {

        /**
         * Search for links that should alter the page state via pushState
         *
         * @param options {Object} Options used by the enhancer
         * @param options.selector {String} (Optional) Selector to use, defaults to .push-state
         */
        withPushState: function(options) {
            options = _.defaults(options, {
                selector: ".push-state"
            });

            AJS.$(document).delegate(options.selector, "simpleClick", function(e) {
                if (this.href) {
                    e.preventDefault();

                    // Let's assume we're only going to be pushing relative links.
                    var path = new URI(this.href).fragment("").resource();

                    // Backbone's fragment doesn't include the history root.
                    path = path.replace(JIRA.Issues.IssueNavRouter.getBackboneRoot(), "");

                    JIRA.Issues.Application.execute('navigation:navigateToUrl', path,
                            {forceRefresh: true, checkPermission: true, replace: true, reset: true});
                }
            });
        },

        /**
         * Search for links to the the issue navigator and enhances them
         */
        toIssueNav: function() {
            var filterSelector = "a.filter-link[data-filter-id], a.filter-link[data-jql]";
            var newSelector = "#issues_new_search_link_lnk";

            AJS.$(document).on("simpleClick", filterSelector, function(e) {
                e.preventDefault();

                var $anchor = AJS.$(e.target).closest("a");
                var filterId = $anchor.data("filter-id");
                var jql = $anchor.data("jql");

                JIRA.Issues.Application.execute("navigation:navigate", {
                    filter: filterId,
                    jql: jql,
                    searchId: _.uniqueId()
                }, {reset: true, resetQuery: true});
            });

            // The "Search for Issues" link in the header.
            AJS.$(document).on("simpleClick", newSelector, function(e) {
                e.preventDefault();
                JIRA.Issues.Application.execute("navigation:navigate", {
                    jql: '',
                    searchId: _.uniqueId()
                }, {reset: true, resetQuery: true});
            });
        },

        isIssueTableDropdown: function() {
            if (AJS.InlineLayer.current) {
                var $offsetTarget = jQuery(AJS.InlineLayer.current.offsetTarget());
                if ($offsetTarget.closest(".list-view table#issuetable").size()) {
                    return true;
                }
            }
        },

        /**
         * Search for links to a particular issue and enhances them
         */
        toIssue: function() {
            // Make issue links load via AJAX.
            var issueSelector = "a.issue-link[data-issue-key]";
            var issueIsParentSelector = ".parentIssue";
            var issueFromTableSelector = ".list-view table#issuetable a.issue-link[data-issue-key]";
            AJS.$(document).on("simpleClick", issueSelector, function(e) {
                if (e.isDefaultPrevented()) return;

                e.preventDefault();
                e.stopPropagation();

                var issueKey = AJS.$(e.target).closest("a").data("issue-key");

                // Check if it is a link from the results table
                var isFromResultsTable = AJS.$(e.target).is(issueFromTableSelector);
                var isLinkToParentIssue = AJS.$(e.target).is(issueIsParentSelector);
                var isIssueTableDropdown = JIRA.Issues.enhanceLinks.isIssueTableDropdown(issueFromTableSelector);

                if (this.searchPageModule && ((isFromResultsTable && !isLinkToParentIssue) || isIssueTableDropdown)) {
                    var searchResults = this.searchPageModule.searchResults;
                    var issueIdAsString = searchResults._getIssueIdForKey(issueKey);
                    var issueId = parseInt(issueIdAsString, 10);

                    JIRA.Issues.Application.execute("analytics:trigger", "kickass.openIssueFromTable", {
                        issueId: issueIdAsString,
                        // these are 1-based indices
                        absolutePosition: searchResults.getPositionOfIssueInSearchResults(issueId) + 1,
                        relativePosition: searchResults.getPositionOfIssueInPage(issueId) + 1,
                        source: "jira.issuenav.global"
                    });
                    JIRA.Issues.Application.execute("navigation:navigate", {
                        selectedIssueKey: issueKey
                    });
                } else {
                    require('jira/issues/navigator/metrics').notifyIssueView();

                    JIRA.Issues.Application.execute("navigation:navigate", {
                        selectedIssueKey: issueKey
                    }, {reset: true});
                }
            }.bind(this));
        },

        registerSearchPageModule: function registerSearchPageModule(searchPageModule) {
            this.searchPageModule = searchPageModule;
        }
    };

}());
