(function(require) {
    "use strict";

    var $ = require("jquery");
    var Events = require("jira/util/events");
    var Metrics = require("jira/issues/navigator/metrics");

    // export some events
    JIRA.Events.LAYOUT_RENDERED = "layoutRendered";

    AJS.namespace("JIRA.Issues.ViewIssueCreator");

    JIRA.Issues.ViewIssueCreator.create = function create($el, options) {

        this.$issueContainer = $(".issue-container");
        this.$searchContainer = $(".navigator-container");

        this.fullScreenIssue = new JIRA.Issues.FullScreenIssue({
            issueContainer: this.$issueContainer,
            searchContainer: this.$searchContainer
        });

        // Router
        this.issueNavRouter = new JIRA.Issues.IssueNavRouter({
            initialSessionSearchState: options.initialSessionSearchState
        });

        JIRA.Issues.enhanceLinks.toIssueNav();
        JIRA.Issues.enhanceLinks.toIssue();
        JIRA.Issues.enhanceLinks.withPushState({});

        JIRA.Issues.Application.on("issueEditor:linkToIssue", function(event) {
            Metrics.notifyIssueView();
            JIRA.Issues.Application.execute('navigation:navigate', {
                selectedIssueKey: event.issueKey
            }, {reset: true});
        });

        // Initialize event bubbling
        JIRA.Issues.Application.on("issueEditor:saveSuccess", function(props) {
            Events.trigger(JIRA.Events.ISSUE_REFRESHED, [props.issueId]);
        });
        JIRA.Issues.Application.on("issueEditor:saveError", function(props) {
            if (!props.deferred) {
                Events.trigger(JIRA.Events.ISSUE_REFRESHED, [props.issueId]);
            }
        });
        JIRA.Issues.Application.on("issueEditor:render", function(regions) {
            JIRA.Issues.Application.execute("pager:render", regions);
        });

        /**
         * Used to defer the showing of issue dialogs until all promises are resolved.
         * We use this to ensure the dialog we are opening has the correct data.
         * If we are inline editing the summary then open the edit dialog, we want to be sure that the summary has been
         * updated on the server first, otherwise we will be showing stale data in the edit dialog.
         */
        JIRA.Dialogs.BeforeShowIssueDialogHandler.add(JIRA.Issues.Api.waitForSavesToComplete);
        return this;
    };
}(window.require));

(function(require) {
    "use strict";

    var $ = require("jquery");
    var _ = require("underscore");
    var Meta = require("jira/util/data/meta");
    var Events = require("jira/util/events");
    var LegacyIssueNavigator = require("jira/issues/search/legacyissuenavigator");
    var LegacyIssueNavigatorShortcuts = require("jira/issues/search/legacyissuenavigatorshortcuts");

    AJS.namespace("JIRA.Issues.IssueNavCreator");

    JIRA.Issues.IssueNavCreator.create = function($el, options) {
        options = _.defaults(options, {standaloneIssue: false});
        var viewIssueComponents = options.viewIssueComponents || JIRA.Issues.ViewIssueCreator.create($el, options);

        this.fullScreenIssue = viewIssueComponents.fullScreenIssue;

        var searchPageModule = this.searchPageModule = new JIRA.Issues.SearchPageModule({}, {
            initialIssueTableState: options.initialIssueTableStatea
        });

        searchPageModule.registerViewContainers({
            issueContainer: viewIssueComponents.$issueContainer,
            searchContainer: viewIssueComponents.$searchContainer
        });

        // Initialise Modules

        var searchHeaderModule = this.searchHeaderModule = new JIRA.Issues.SearchHeaderModule({
            searchPageModule: searchPageModule
        });

        var filterModule = new JIRA.Issues.FilterModule({
            searchPageModule: searchPageModule,
            systemFilters: searchPageModule.addOwnerToSystemFilters(options.systemFilters)
        });

        // TODO TF-693 - FullScreenIssue have detached this element, so we reach inside to get a reference
        var $queryForm = this.fullScreenIssue.searchContainer.find("form.navigator-search");
        var queryModule = JIRA.Components.Query.create({
            el: $queryForm,
            searchers: options.initialSearcherCollectionState,
            preferredSearchMode: Meta.get("user.search.mode"),
            layoutSwitcher: true,
            autocompleteEnabled: Meta.getBoolean("autocomplete-enabled"),
            basicAutoUpdate: Meta.getBoolean("hasCriteriaAutoUpdate")
        });


        Events.bind(JIRA.Events.ISSUE_TABLE_REORDER, function(e) {
            if (!JIRA.Issues.Application.request("issueEditor:canDismissComment")) {
                e.preventDefault();
            }
        });

        this.layoutSwitcherView = new JIRA.Issues.LayoutSwitcherView({
            searchPageModule: searchPageModule
        });
        // TODO TF-693 - FullScreenIssue have detached this element, so we reach inside to get a reference
        var $layoutSwitcherToggle = this.fullScreenIssue.searchContainer.find("#layout-switcher-toggle");
        this.layoutSwitcherView.setElement($layoutSwitcherToggle).render();

        var issueModule = JIRA.Issues.Application.request("issueEditor");

        JIRA.Issues.Application.on("issueEditor:refineViewer", function(event) {
            event.preventDefault();
            JIRA.Issues.Application.execute("issueEditor:updateIssueWithQuery", event.query);
        });
        JIRA.Issues.Application.commands.setHandler("returnToSearch", function() {
            JIRA.Issues.Application.execute("issueEditor:close");
        });

        JIRA.Issues.FocusShifter.init();

        var viewIssueData = issueModule.viewIssueData;

        var issueSearchManager = new JIRA.Issues.IssueSearchManager({
            initialIssueTableState: options.initialIssueTableState,
            initialIssueIds: options.initialIssueIds
        });

        var searchModule = new JIRA.Issues.SearchModule({
            searchPageModule: searchPageModule,
            queryModule: queryModule,
            issueSearchManager: issueSearchManager,
            initialSelectedIssue: options.initialSelectedIssue
        });

        var issueCacheManager = new JIRA.Issues.Cache.IssueCacheManager({
            searchResults: searchModule.getResults(),
            viewIssueData: viewIssueData
        });

        this.fullScreenIssue.registerIssueCacheManager(issueCacheManager);

        // TODO TF-693 - FullScreenIssue will detach these elements, so get a reference now before they're not discoverable.
        var issueNavToolsElement = this.fullScreenIssue.searchContainer.find(".saved-search-selector");
        // TODO TF-693 - Try to prevent FullScreenIssue from hacking and mutilating the DOM so much...

        // Register Modules
        searchPageModule.registerSearch(searchModule);
        searchPageModule.registerSearchHeaderModule(searchHeaderModule);
        searchPageModule.registerFilterModule(filterModule);
        searchPageModule.registerQueryModule(queryModule);

        searchPageModule.registerFullScreenIssue(viewIssueComponents.fullScreenIssue);

        searchPageModule.registerIssueSearchManager(issueSearchManager);
        searchPageModule.registerIssueCacheManager(issueCacheManager);
        searchPageModule.registerLayoutSwitcher(this.layoutSwitcherView);

        searchHeaderModule.registerSearch(searchModule);
        searchHeaderModule.createToolsView(issueNavToolsElement);

        if (options.standaloneIssue) {
            //Make sure that SPM knows that we are viewing standalone issue without resetting what is already rendered
            searchPageModule.standalone = true;
        }
        // Overrides

        JIRA.Issues.enhanceLinks.registerSearchPageModule(searchPageModule);

        JIRA.Issues.dialogCleaner();

        JIRA.Issues.Api.initialize({
            searchPageModule: searchPageModule
        });

        JIRA.Issues.overrideIssueDialogs({
            getIssueId: _.bind(searchPageModule.getEffectiveIssueId, searchPageModule),
            isNavigator: true,
            updateIssue: function(dialog) {
                var issueUpdate = JIRA.Issues.Utils.getUpdateCommandForDialog(dialog);
                return searchPageModule.updateIssue(issueUpdate);
            }
        });

        LegacyIssueNavigator.initialize();
        LegacyIssueNavigatorShortcuts.initialize();

        // Keyboard shortcuts ?

        $(document).keydown(function(e) {
            var dialogIsVisible = $("div.aui-blanket").length > 0;
            var wasSupportedKey = (e.which === $.ui.keyCode.ENTER || e.which === $.ui.keyCode.LEFT ||
                e.which === $.ui.keyCode.UP || e.which === $.ui.keyCode.RIGHT || e.which === $.ui.keyCode.DOWN);

            if (!dialogIsVisible && wasSupportedKey) {
                var target = $(e.target);
                var targetIsValid = target.is(":not(:input)");

                if (target == null || targetIsValid) {
                    if (e.which === $.ui.keyCode.ENTER) {
                        if (target == null || target.is(":not(a)")) {
                            JIRA.Issues.Api.viewSelectedIssue();
                        }
                    } else if (e.which === $.ui.keyCode.LEFT) {
                        searchPageModule.focusIssueList();
                    } else if (e.which === $.ui.keyCode.RIGHT) {
                        searchPageModule.focusIssueEditor();
                    } else if (e.which === $.ui.keyCode.UP) {
                        if (searchPageModule.handleUp()) {
                            e.preventDefault();
                        }
                    } else if (e.which === $.ui.keyCode.DOWN) {
                        if (searchPageModule.handleDown()) {
                            e.preventDefault();
                        }
                    }
                }
            }
        });

        // Not such a crash hot idea; should remove it.
        this.searchResults = searchModule.getResults();

        // Create the on change bindings for updating the login link.
        this.searchPageModule.on("change", changeLoginUrl);
        this.searchResults.on("change", changeLoginUrl);
        this.searchResults.getSelectedIssue().on("change", changeLoginUrl);

        return this;
    };

    //Change the login url to the current state.
    function changeLoginUrl() {
        var url = JIRA.Issues.LoginUtils.redirectUrlToCurrent();
        $('.login-link').attr('href', url);
    }
}(window.require));

(function(require) {
    "use strict";

    var $ = require("jquery");
    var Messages = require("jira/message");

    AJS.namespace("JIRA.Issues.GlobalIssueNavCreator");

    JIRA.Issues.GlobalIssueNavCreator.create = function($el, options) {

        // Construct base issue navigator
        var issuenav = JIRA.Issues.IssueNavCreator.create($el, options);

        // Create filter modules' views

        var filterModule = issuenav.searchPageModule.filterModule; // HACK: I'm reaching inside the issuenav because the filter module is depended on by base behaviour of the whole thing.
        // TODO TF-693 - FullScreenIssue have detached these elements, so we reach inside to get a reference
        var $filterPanelEl = issuenav.fullScreenIssue.searchContainer.find(".navigator-sidebar");
        var $filterHeaderEl = issuenav.fullScreenIssue.searchContainer.find("#search-header-view");

        filterModule.createView({
            $filterPanelEl: $filterPanelEl
        });
        filterModule.filtersComponent.showFilterHeader({
            el: $filterHeaderEl,
            model: issuenav.searchPageModule.getFilter(),
            isEdited: issuenav.searchPageModule.isDirty()
        });

        // Find the quickSearch message (if any) and convert it to a JIRA.Message
        var quickSearchMessaage = $("#quicksearch-message");
        if (quickSearchMessaage.length) {
            var $container = Messages.showWarningMsg(quickSearchMessaage.find("p").html(), {
                closeable: true
            });
            $("a", $container).click(function() {
                $container.remove();
            });
        }

        return issuenav;
    };
    /**
    * Read all the initial data in the DOM
    *
    * If the ColumnConfigState has been sent from the server we want to take the HTML from the table
    * and pop it onto its table property.
    *
    * This prevents us from having to populate the HTML twice in the dom. Once in the HTML and another time in the
    * JSON. It also prevents us needing to ensure there are no XSS vulnerabilities in the JSON HTML string.
    */
    JIRA.Issues.GlobalIssueNavCreator.readInitialData = function readInitialData() {
        var data = {};
        var $navigatorContent = $(".navigator-content");

        data.initialIssueTableState = $navigatorContent.data("issue-table-model-state");
        if (data.initialIssueTableState && !data.initialIssueTableState.table) {
            var wrapper = $("<div></div>").append($navigatorContent.children().clone());
            data.initialIssueTableState.issueTable.table = wrapper.html();
        }

        data.initialIssueIds = AJS.$('#stableSearchIds').data('ids');
        data.initialSelectedIssue = $navigatorContent.data("selected-issue");

        // jQuery.parseJSON gracefully returns null given an empty string.
        // Would be even nicer if the json was placed in a data- attribute, which jQuery will automatically parse with .data().
        data.initialSearcherCollectionState = $.parseJSON($("#criteriaJson").text());

        var DarkFeatures = require("jira/components/issueviewer/services/darkfeatures");
        var httpSessionDisabled = DarkFeatures.HTTP_SESSIONS_DISABLED.enabled();

        if (httpSessionDisabled) {

            var LatestSearchStore = require("jira/components/search/latestStorage");
            var latestSearch = LatestSearchStore.get();

            if (latestSearch && (latestSearch.jql || latestSearch.filter)) {
                data.initialSessionSearchState = latestSearch;
            }

            if (!data.initialSessionSearchState) {
                // We use the backup from the server. Despite the name, this doesn't actually look at the session any more
                // but rather uses a fallback filter definition from the server for a user's default filter. Particularly
                // useful when someone first uses JIRA.
                data.initialSessionSearchState = $navigatorContent.data("session-search-state");
            }
        } else {
            data.initialSessionSearchState = $navigatorContent.data("session-search-state");
        }
        data.systemFilters = $.parseJSON($("#systemFiltersJson").text());

        return data;
    };

})(window.require);
