AJS.test.require(["com.atlassian.jira.jira-issue-nav-plugin:pager-test"], function(){
    "use strict";

    var PagerModel = require("jira/issues/modules/pager/model");
    var PagerController = require("jira/issues/modules/pager/controller");
    var MockUtils = require("jira/components/test-utils/mockutils");
    var MarionetteMocker = require("jira/components/test-utils/marionettemocker");

    module('jira/issues/modules/pager', {
        setup: function() {
            this.sandbox = sinon.sandbox.create();

            this.PagerModel = MarionetteMocker.createEventedMock(this.sandbox, PagerModel);
            this.PagerController = MarionetteMocker.createEventedMock(this.sandbox, PagerController);

            this.callbacks = {
                goBack: this.sandbox.stub(),
                nextItem: this.sandbox.stub(),
                previousItem: this.sandbox.stub()
            };

            this.PagerConstructor = MockUtils.spyAll(this.sandbox, MockUtils.requireWithMocks("jira/issues/modules/pager", {
                "jira/issues/modules/pager/model": this.PagerModel.constructor,
                "jira/issues/modules/pager/controller": this.PagerController.constructor
            }));
        },
        teardown: function() {
            this.sandbox.restore();
        }
    });

    test("When the pager is initialised, it should set up the model", function() {
        var issuePager = new this.PagerConstructor();
        issuePager.initialize();

        sinon.assert.calledOnce(this.PagerModel.constructor);
    });

    test("When pager is updated with data, it should update the model", function() {
        var issuePager = new this.PagerConstructor();
        issuePager.initialize();
        var data = {foo: "bar"};

        issuePager.update(data);

        sinon.assert.calledWith(this.PagerModel.update, data);
    });

    test("When pager is updated, it should not clear the model if the view is an error", function() {
        this.PagerModel.get.withArgs("isError").returns(true);

        var issuePager = new this.PagerConstructor();
        issuePager.initialize();

        issuePager.update();

        sinon.assert.notCalled(this.PagerModel.clear);
    });

    test("When pager is updated, it should clear the model if the view is not an error", function() {
        this.PagerModel.get.withArgs("isError").returns(false);

        var issuePager = new this.PagerConstructor();
        issuePager.initialize();

        issuePager.update();

        sinon.assert.calledOnce(this.PagerModel.clear);
    });

    test("When the pager is shown, it should set up the controller if it doesn't exist", function() {
        var issuePager = new this.PagerConstructor(this.stub());
        issuePager.initialize();

        issuePager.show();

        sinon.assert.calledOnce(this.PagerController.constructor);
        sinon.assert.calledWith(this.PagerController.constructor, sinon.match({
            model: sinon.match(this.PagerModel)
        }));
    });

    test("When the pager is shown, it should set up controller events if it doesn't exist", function() {
        var issuePager = new this.PagerConstructor(this.callbacks);
        issuePager.initialize();

        issuePager.show();

        this.PagerController.trigger("goBack");
        sinon.assert.calledOnce(this.callbacks.goBack, "Go back handler is called");

        this.PagerController.trigger("nextItem");
        sinon.assert.calledOnce(this.callbacks.nextItem, "Next item handler is called");

        this.PagerController.trigger("previousItem");
        sinon.assert.calledOnce(this.callbacks.previousItem, "Previous item handler is called");
    });

    test("When the pager is shown, it should call the controller", function() {
        var container = "foo";
        var element = "bar";

        var issuePager = new this.PagerConstructor(this.stub());
        issuePager.initialize();

        issuePager.show(container, element);

        sinon.assert.calledWith(this.PagerController.show, container, element);
    });

    test("When the pager is closed, it should close the controller", function() {
        var issuePager = new this.PagerConstructor(this.stub());
        issuePager.initialize();

        // We call show() to ensure the controller is set.
        issuePager.show();

        issuePager.close();

        sinon.assert.calledOnce(this.PagerController.close);
    });

    test("When the pager is closed, it should clear the model", function() {
        var issuePager = new this.PagerConstructor();
        issuePager.initialize();

        issuePager.close();

        sinon.assert.calledOnce(this.PagerModel.clear);
    });
});
