AJS.test.require([
    "com.atlassian.jira.jira-issue-nav-plugin:testutils",
    "com.atlassian.jira.jira-issue-nav-plugin:navigation"
], function() {
    "use strict";

    var DarkFeatures = require("jira/components/issueviewer/services/darkfeatures");
    var MockUtils = require("jira/components/test-utils/mockutils");
    var State = require("jira/issues/modules/navigation/state");
    var NavigationController = require("jira/issues/modules/navigation/controller");
    var _ = require("underscore");
    var Application = require("jira/issues/application");
    var LayoutPreferenceManager = require("jira/issues/layout-preference-manager");

    module('NavigationController', {
        setup: function() {
            this.context = AJS.test.mockableModuleContext();
            this.context.mock('jira/issues/modules/navigation/state', function() {});
            this.sandbox = sinon.sandbox.create();
            this.canDismissCommentStub = this.sandbox.stub(Application, "request").withArgs("issueEditor:canDismissComment").returns(true);
            this.preferredLayoutStub = this.sandbox.stub(LayoutPreferenceManager, "getPreferredLayoutKey").returns("split-view");
            this.darkFeatureEnabledStub = this.sandbox.stub(DarkFeatures.REDIRECT_FROM_GLOBAL_TO_PROJECT, "enabled").returns(false);

            this.NavigationControllerContructor = MockUtils.requireWithMocks('jira/issues/modules/navigation/controller', {
                "jira/components/issueviewer/services/darkfeatures": DarkFeatures,
                "jira/issues/modules/navigation/state": State,
                "jira/issues/application": Application
            });

            this.navigationController = new this.NavigationControllerContructor({isFullPageLoad: false});

            this.stateChangedHandler = this.sandbox.spy();
            this.navigationController.on('stateChanged', this.stateChangedHandler);
        },
        teardown: function() {
            this.sandbox.restore();
        },
        createStateWith: function() {
            var states = _.toArray(arguments);
            // We need to _.clone() because cloning moves everything from the prototype to the object itself, which is
            // exactly what NavigationController does internally.
            var baseState = _.clone(new State());
            return _.extend.apply(null, [baseState].concat(states));
        }
    });

    test("stateChanged event contains correct properties", function() {
        var navigateOptions = {
           routerEvent: true,
           randomProperty: '1'
        };
        var state = {
           jql: "test",
           selectedIssueKey: "TEST-1"
        };
        var secondState = {
            filter: 12
        };

        this.navigationController.navigate(state, navigateOptions);

        sinon.assert.calledOnce(this.stateChangedHandler);
        sinon.assert.calledWithExactly(this.stateChangedHandler, this.createStateWith(state), navigateOptions);

        this.navigationController.navigate(secondState);

        sinon.assert.calledTwice(this.stateChangedHandler);
        sinon.assert.calledWithExactly(this.stateChangedHandler, this.createStateWith(state, secondState), {});

        this.navigationController.navigate(secondState, {reset: true});

        sinon.assert.calledThrice(this.stateChangedHandler);
        sinon.assert.calledWithExactly(this.stateChangedHandler, this.createStateWith(secondState), {reset: true});
    });

    test("When navigating and the REDIRECT_FROM_GLOBAL_TO_PROJECT DarkFeature is off, it does not do a full page load of the URL", function() {
        var navigateOptions = {};
        var state = {
            selectedIssueKey: "TEST-1"
        };

        this.navigationController.navigate(state, navigateOptions);

        sinon.assert.calledOnce(this.stateChangedHandler);
        sinon.assert.calledWith(this.stateChangedHandler, this.createStateWith(state), navigateOptions);
    });

    test("When navigating and the REDIRECT_FROM_GLOBAL_TO_PROJECT DarkFeature is on, it does a full page load of the URL", function() {
        this.darkFeatureEnabledStub.returns(true);

        var navigateOptions = {};
        var state = {
            selectedIssueKey: "TEST-1",
            isStandaloneIssue: function() { return true; },
            toUrl: function() { return "browse/TEST-1"; }
        };

        this.navigationController.navigate(state, navigateOptions);

        sinon.assert.calledOnce(this.stateChangedHandler);
        sinon.assert.calledWith(this.stateChangedHandler, this.createStateWith(state), _.extend(navigateOptions, { forceFullPageLoad: true}));
    });


    test("stateChanged event is fired only if state has changed or forceRefreshParameter was passed", function() {
        var state = {
            jql: "test",
            selectedIssueKey: "test"
        };

        this.navigationController.navigate(state);
        ok(this.stateChangedHandler.calledOnce, "State Changed event got triggered");

        this.navigationController.navigate(state);
        ok(this.stateChangedHandler.calledOnce, "State Changed event wasn't triggered for the second time");

        this.navigationController.navigate(state, {forceRefresh: true});
        ok(this.stateChangedHandler.calledTwice, "State Changed event was triggered");
        ok(this.stateChangedHandler.calledWithExactly(this.createStateWith(state), {}), "State Changed event has proper parameters");

    });

    test("Navigate to URL transforms state to URL with respect of the override parameter", function(){
        var stateFromUrl = {jql: "test", filter: 0};
        var url = "/jql=abce";
        var override = {filter: 1};

        var model = {
            getStateFromUrl: this.sandbox.stub().returns(stateFromUrl)
        };
        var navigationController = new NavigationController({model: model, isFullPageLoad: false});
        this.sandbox.spy(navigationController, "navigate");

        navigationController.navigateToUrl(url);
        ok(navigationController.navigate.calledOnce, "State Changed event got triggered");
        ok(navigationController.navigate.calledWith(stateFromUrl), "State Changed event contains proper arguments");

        var navigateOptions = {routerEvent: true};
        navigationController.navigateToUrl(url, navigateOptions, override);
        ok(navigationController.navigate.calledTwice, "State Changed event got triggered");
        ok(navigationController.navigate.calledWithExactly(_.defaults(override, stateFromUrl), navigateOptions), "State Changed event contains proper arguments");
    });

    test("fullPageLoad parameter is only passed down for the first navigation", function() {
        var navigationController = new this.NavigationControllerContructor();
        navigationController.on('stateChanged', this.stateChangedHandler);
        var state = {jql: "test"};
        var secondState = {jql: "2"};
        var options = {routerEvent: true};

        navigationController.navigate(state, options);
        ok(this.stateChangedHandler.calledOnce, "State Changed event got triggered");
        ok(this.stateChangedHandler.calledWithExactly(this.createStateWith(state), _.extend(options, {fullPageLoad: true})), "fullPageLoad is passed");

        navigationController.navigate(secondState, options);
        ok(this.stateChangedHandler.calledTwice, "State Changed event got triggered");
        ok(this.stateChangedHandler.calledWithExactly(this.createStateWith(secondState), options), "fullPageLoad isn't passed");
    });

    test("Dirty comment warning is respected when navigating", function() {
        var state = {
            jql: "test"
        };

        this.navigationController.navigate(state);
        ok(this.stateChangedHandler.calledOnce, "State changed event is fired");
        ok(this.stateChangedHandler.calledWith(this.createStateWith(state)), "State changed event is fired");

        this.canDismissCommentStub.returns(false);

        this.navigationController.navigate({jql: "test2"});
        ok(this.stateChangedHandler.calledOnce, "State changed event isn't fired when user doesn't want to dismiss the comment");
    });

    test("Event triggering is queued", function() {
        var states = [
            {jql: "state1"},
            {jql: "state2"}
        ];
        var stateChangedHandler = function() {
            this.navigationController.navigate(states[0]);
        }.bind(this);
        var stateChangedSpy = sinon.spy();
        this.navigationController.on("stateChanged", stateChangedHandler);
        this.navigationController.on("stateChanged", stateChangedSpy);

        this.navigationController.navigate(states[1]);

        sinon.assert.calledTwice(stateChangedSpy);
        deepEqual(stateChangedSpy.firstCall.args[0], this.createStateWith(states[1]));
        deepEqual(stateChangedSpy.secondCall.args[0], this.createStateWith(states[0]), "Second state changed has been triggered after all handlers for the first one finished");
    });
});
