define("jira/issues/components/search", ["require"], function(require){
    "use strict";

    var _ = require('underscore');
    var Application = require('jira/issues/application');
    var Controller = require('jira/components/libs/marionette-1.4.1/controller');
    var LatestSearchStore = require("jira/components/search/latestStorage");
    var metrics = require('jira/issues/navigator/metrics');

    /**
     * @class JIRA.Components.SearchService
     *
     * This service encapsulates all the search related stuff
     *
     * @extends JIRA.Marionette.Controller
     */
    return Controller.extend({
        /**
         * @event issueUpdated
         * When an issue has been updated
         */

        /**
         * @event search
         * When an new search has been done
         */

        /**
         * @event before:search
         * Before doing a new search
         */

        /**
         * @event error:search
         * When a new search has been done but it thrown an error
         */

        /**
         * @event issueHighlighted
         * When an issue has been marked as highlighted in the internal model
         */

        /**
         * @event selectedIssueChanged
         * When an issue has been marked as selected in the internal model
         */

        /**
         * @param {Object} options
         * @param {JIRA.Issues.SearchModule} options.searchModule
         * @param {JIRA.Issues.SearchResults} options.searchResults
         * @param {JIRA.Issues.ColumnConfigModel} options.columnConfig
         */
        initialize: function(options) {
            this.searchModule = options.searchModule;
            this.searchResults = options.searchResults;
            this.columnConfig = options.columnConfig;

            _.bindAll(this, "_onIssueUpdated", "_onHighlightedIssueChange", "_onSelectedIssueChange", "_onBeforeSearch");

            this.listenTo(this.searchResults, "change:resultsId change:startIndex stableUpdate issueDeleted", function() {
                this._doSearch();
            });

            // These are *not* regular Backbone events, we can't use listenTo.
            this.searchResults.onIssueUpdated(this._onIssueUpdated);
            this.searchResults.onHighlightedIssueChange(this._onHighlightedIssueChange);
            this.searchResults.onSelectedIssueChange(this._onSelectedIssueChange);

            this.searchModule.onBeforeSearch(this._onBeforeSearch);
        },

        _onSelectedIssueChange: function(issue) {
            this.trigger("selectedIssueChanged", issue, this.searchResults.getHighlightedIssue());
        },

        _onIssueUpdated: function(issueId, entity, reason) {
            this.trigger("issueUpdated", issueId, entity, reason);
        },

        _onHighlightedIssueChange: function(issue) {
            this.trigger("issueHighlighted", issue.getId());
        },

        _onBeforeSearch: function() {
            this.trigger("before:search");
        },

        close: function() {
            this.stopListening();
            this.searchResults.offIssueUpdated(this._onIssueUpdated);
            this.searchResults.offHighlightedIssueChange(this._onHighlightedIssueChange);
        },

        /**
         * Asks the SearchResults object to do a new search with the parameters already contained
         * in the SearchModule
         * @private
         */
        _doSearch: function() {
            this.searchInProgress = true;

            var filterId = this.searchModule.getFilterId();
            var isSystemFilter = filterId < 0;
            filterId = (isSystemFilter || _.isNull(filterId)) ? undefined : filterId;

            var search = {
                filter: filterId,
                jql: this.searchModule.getEffectiveJql()
            };
            LatestSearchStore.save(search);

            this.searchResults.getResultsForPage({
                jql: this.searchModule.getEffectiveJql(),
                filterId: filterId
            })
                .always(_.bind(function() {
                    this.searchInProgress = false;
                }, this))
                .done(_.bind(function(table) {
                    if (!this.searchResults.hasHighlightedIssue()) {
                        this.searchResults.highlightFirstInPage();
                    }
                    this.trigger("search", table, this.searchResults);
                }, this))
                .fail(_.bind(function() {
                    this.trigger("error:search");
                }, this));
        },

        /**
         * Loads a page of the current search results. This code expects the issue position,
         * not the page number. Example: using a pageSize of 25, passing startIndex=50 will load the
         * issues #50 to #74.
         *
         * This method will do nothing if a search is already in progress
         *
         * @param {number} startIndex Position of the issue in the page
         */
        goToPage: function(startIndex) {
            if (this.searchInProgress) return;
            this._onBeforeSearch();
            this.searchResults.goToPage(startIndex);
        },

        /**
         * Sorts the current search results by a field.
         *
         * This method will do nothing if a search is already in progress
         *
         * @param {string} fieldId ID of the field to sort by
         */
        sort: function(fieldId) {
            if (this.searchInProgress) return;

            var allSorts = this.searchModule.getResults().getColumnSortJql();
            var sortJql = allSorts[fieldId];
            if (sortJql) {
                this.searchModule.doSort(sortJql);
            }
        },

        /**
         * Re-runs the current search.
         *
         * This method will do nothing if a search is already in progress
         */
        runCurrentSearch: function() {
            if (this.searchInProgress) return;

            if (Application.request("issueEditor:canDismissComment")) {
                Application.execute("analytics:trigger", "kickass.issueTableRefresh");
                this.searchModule.refresh();
            }
        },

        /**
         * This method updates the existing set of results (aka Stable Search). It does not re-run the search again, so the
         * list of Issues will remain the same.
         *
         * This method will do nothing if a search is already in progress.
         */
        updateExitingResults: function() {
            if (this.searchInProgress) return;

            if (Application.request("issueEditor:canDismissComment")) {
                Application.execute("analytics:trigger", "kickass.issueTableRefresh");
                this._doSearch();
            }
        },

        highlightIssue: function(issueId) {
            this.searchResults.highlightIssueById(issueId);
        },

        getPager: function() {
            return this.searchResults.getPager();
        },

        selectNextIssue: function() {
            this.searchResults.highlightNextIssue();
            if (this.searchResults.hasSelectedIssue()) {
                require('jira/issues/navigator/metrics').notifyIssueView();
                this.searchResults.selectNextIssue();
            }
        },

        selectPreviousIssue: function () {
            this.searchResults.highlightPrevIssue();
            if (this.searchResults.hasSelectedIssue()) {
                metrics.notifyIssueView();
                this.searchResults.selectPrevIssue();
            }
        },

        unselectIssue: function() {
            this.searchResults.unselectIssue();
        },

        hasSelectedIssue: function() {
            return this.searchResults.hasSelectedIssue();
        },

        getHighlightedIssue: function() {
            return this.searchResults.getHighlightedIssue().id;
        }
    });
});
