define('jira/issues/components/filters/views/header', ['require'], function (require) {
    "use strict";

    var _ = require('underscore');
    var wrmContextPath = require('wrm/context-path');
    var jQuery = require('jquery');
    var Templates = require('jira/issues/components/filters/templates/header');
    var ItemView = require('jira/components/libs/marionette-1.4.1/itemview');
    var LoginUtils = require('jira/issues/utils/login');
    var Tipsy = require("jira/issues/tipsy");

    var contextPath = wrmContextPath();

    /**
     * @class JIRA.Components.Filters.Controllers.Header
     *
     * @extends JIRA.Marionette.ItemView
     */
    return ItemView.extend({
        operations: {
            save: {
                styleClass: "save-changes",
                label: AJS.I18n.getText("common.words.save")
            },
            saveAs: {
                styleClass: "save-as-new-filter",
                label: AJS.I18n.getText("search.save.new.filter")
            },
            discardChanges: {
                styleClass: "discard-filter-changes",
                label: AJS.I18n.getText("search.discard.filter.changes")
            },
            details: {
                styleClass: "show-filter-details",
                title: AJS.I18n.getText("issuenav.filters.details.button.title"),
                label: AJS.I18n.getText("issuenav.filters.details.label")
            },
            favouriteOn: {
                title: AJS.I18n.getText('issuenav.filters.fav.button.on'),
                label: AJS.I18n.getText('issuenav.filters.fav.button.on'),
                styleClass: "fav-link aui-icon aui-icon-small aui-iconfont-star"
            },
            favouriteOff: {
                title: AJS.I18n.getText('issuenav.filters.fav.button.off'),
                label: AJS.I18n.getText('issuenav.filters.fav.button.off'),
                styleClass: "fav-link aui-icon aui-icon-small aui-iconfont-unstar"
            }
        },

        template: Templates.Main,

        ui: {
            filterFavouriteButton: ".fav-link",
            filterEditedLabel: ".filter-edited-item",
            saveButton: '.save-changes',
            saveAsButton: '.save-as-new-filter',
            detailsTrigger: '.show-filter-details',
            dropdownContent: '#js-edited-content'
        },

        triggers: {
            'click .fav-link': "toogleFavourite",
            "click .save-as-new-filter:not([aria-disabled=true])": {
                event: "saveAs",
                preventDefault: true,
                stopPropagation: false
            },
            "click .save-changes:not([aria-disabled=true])": {
                event: "save",
                preventDefault: true,
                stopPropagation: false
            },
            "click .discard-filter-changes": {
                event: "discard",
                preventDefault: true,
                stopPropagation: false
            }
        },

        events: {
            "click .show-filter-details": function() {
                this.trigger("details", this.ui.detailsTrigger);
            }
        },

        _generatePrimaryOperations: function() {
            var operations = [];

            if (this.isEdited && this.model.isMyFilter()) {
                operations.push(this.operations.save);
            }

            if (!this.isDirty && !!LoginUtils.loggedInUserName()) {
                operations.push(this.operations.saveAs);
            }

            if (this.isEdited) {
                operations.push(this.operations.discardChanges);
            }

            return operations;
        },

        _generateSecondaryOperations: function() {
            var operations = [];

            if (!this.model.getIsSystem()) {
                operations.push(this.operations.details);
            }

            if (!!LoginUtils.loggedInUserName() && !this.model.getIsSystem()) {
                var operation = this.model.isFavouriteFilter() ? this.operations.favouriteOn : this.operations.favouriteOff;
                operations.href = contextPath + "/secure/EditFilter!default.jspa?filterId=" + this.model.id;
                operations.push(operation);
            }

            return operations;
        },

        serializeData: function() {
            if (!this.model) {
                return {
                    title: AJS.I18n.getText("search.search"),
                    primaryButton: this.operations.saveAs
                };
            }

            if (!this.model.getIsValid()) {
                return {
                    title: AJS.I18n.getText("search.search")
                };
            }

            var primaryOperations = this._generatePrimaryOperations();
            var secondaryOperations = this._generateSecondaryOperations();

            var renderData = {
                title: this.model.getName(),
                primaryButton: primaryOperations.shift(),
                primaryOps: primaryOperations,
                secondaryOps: secondaryOperations
            };

            if (this.model.getOwner()) {
                _.extend(renderData, {
                    owner: this.model.getOwner() || "",
                    ownerUserName: this.model.getOwnerUserName() || "",
                    avatarUrl: this.model.getAvatarUrl(),
                    ownerUrl: contextPath + '/secure/ViewProfile.jspa?name=' + encodeURIComponent(this.model.getOwnerUserName())
                });
            }

            return renderData;
        },

        render: function(options) {
            if (typeof options.model !== "undefined") {
                this.model = options.model;
            }
            this.isEdited = options.isEdited;
            this.isDirty = options.isDirty;
            return ItemView.prototype.render.call(this);
        },

        onRender: function() {
            this.ui.dropdownContent.on("aui-dropdown2-show", _.bind(function(e) {
                //HACK to use Backbone's event delegation on the dropdown content.
                //Backbone's 'delegateEvents()' works only on 'this.$el', so we need to set it before calling
                //the delegate method, and restore it afterwards to the real value.
                var viewEl = this.$el;
                this.$el = jQuery(e.target);
                this.delegateEvents();
                this.$el = viewEl;
            }, this));

            if (this.dirtyTipsy) {
                this.dirtyTipsy.remove();
                delete this.dirtyTipsy;
            }

            if (this.isDirty) {
                this.markAsInvalid();
            }
            new Tipsy({el: this.ui.filterFavouriteButton});
        },

        markAsInvalid: function() {
            this.ui.saveButton.addClass("disabled");
            this.ui.saveButton.attr("aria-disabled", "true");

            this.dirtyTipsy = new Tipsy({
                el: this.ui.saveButton,
                tipsy: {
                    delayIn: 0,
                    title: function() {
                        return AJS.I18n.getText("search.cant.save.need.to.run.search.first");
                    }
                }
            });
        }
    });
});
