AJS.test.require(["com.atlassian.jira.jira-issue-nav-plugin:filters-component-test"], function () {
    "use strict";

    var CopyDialog = require('jira/issues/components/filters/views/dialogs/copy');
    var DeleteDialog = require('jira/issues/components/filters/views/dialogs/delete');
    var RenameDialog = require('jira/issues/components/filters/views/dialogs/rename');
    var SaveAsDialog = require('jira/issues/components/filters/views/dialogs/save-as');
    var Dialog = require("jira/issues/components/filters/controllers/dialog");
    var Filter = require("jira/issues/components/filters/entities/filter");

    module('jira/issues/components/filters/controllers/dialog', {
        setup: function () {
            this.filterModel = new Filter({
                id: "1234"
            });
        },

        buildDialogController: function () {
            return new Dialog({});
        },

        /**
         * Asserts that a new dialog is shown.
         *
         * @param {function} constructor Constructor of the dialog.
         * @param {string} showMethod Method used to display the dialog. Valid values: "showDeleteDialog",
         *                            "showRenameDialog", "showCopyDialog" and "showSaveAsDialog".
         */
        assertDialogIsRendered: function (constructor, showMethod) {
            var dialogController = this.buildDialogController();
            var renderStub = this.stub(constructor.prototype, "render");

            dialogController[showMethod]();

            sinon.assert.calledOnce(renderStub);
        },

        /**
         * Asserts that a dialog throws an error event when the form is submitted with errors.
         *
         * @param {function} constructor Constructor of the dialog.
         * @param {string} showMethod Method used to display the dialog. Valid values: "showDeleteDialog",
         *                            "showRenameDialog", "showCopyDialog" and "showSaveAsDialog".
         * @param {string} eventToTest Method that the dialog would throw in case of an error.
         */
        assertErrorEvent: function (constructor, showMethod, eventToTest) {
            var dialogController = this.buildDialogController();
            var eventSpy = this.spy();
            this.stub(constructor.prototype, "render");

            dialogController[showMethod]();
            dialogController.on(eventToTest, eventSpy);
            dialogController.activeView.trigger('submit:error');

            sinon.assert.calledOnce(eventSpy);
        },

        /**
         * Asserts that a dialog throws a success event when the form is submitted without errors.
         *
         * @param {function} constructor Constructor of the dialog.
         * @param {string} showMethod Method used to display the dialog. Valid values: "showDeleteDialog",
         *                            "showRenameDialog", "showCopyDialog" and "showSaveAsDialog".
         * @param {string} eventToTest Method that the dialog would throw when the form is submitted.
         * @param {Object} serverResponse Fake response from the server when the form is submitted.
         */
        assertSuccessEvent: function (constructor, showMethod, eventToTest, serverResponse) {
            var dialogController = this.buildDialogController();
            var eventSpy = this.spy();
            this.stub(constructor.prototype, "render");

            dialogController[showMethod](this.filterModel);
            dialogController.on(eventToTest, eventSpy);
            dialogController.activeView.trigger('submit:success', serverResponse);

            sinon.assert.calledOnce(eventSpy);
            return eventSpy.firstCall.args[0];
        }
    });

    test("It shows the delete dialog", function () {
        this.assertDialogIsRendered(DeleteDialog, "showDeleteDialog");
    });

    test("It shows the rename dialog", function () {
        this.assertDialogIsRendered(RenameDialog, "showRenameDialog");
    });

    test("It shows the copy dialog", function () {
        this.assertDialogIsRendered(CopyDialog, "showCopyDialog");
    });

    test("It shows the save-as dialog", function () {
        this.assertDialogIsRendered(SaveAsDialog, "showSaveAsDialog");
    });

    test("When showing the delete dialog, it triggers 'delete:success' then the dialog is sent successfully", function () {
        this.assertSuccessEvent(DeleteDialog, "showDeleteDialog", "delete:success", {name: "filter name"});
    });

    test("When showing the delete dialog, it triggers 'delete:error' then the dialog is sent unsuccessfully", function () {
        this.assertErrorEvent(DeleteDialog, "showDeleteDialog", "delete:error");
    });

    test("When submitting the delete dialog, it returns the model being deleted", function () {
        var model = this.assertSuccessEvent(DeleteDialog, "showDeleteDialog", "delete:success", {name: "filter name"});
        equal(model, this.filterModel);
    });

    test("When showing the rename dialog, it triggers 'rename:success' then the dialog is sent successfully", function () {
        this.assertSuccessEvent(RenameDialog, "showRenameDialog", "rename:success", {name: "filter name"});
    });

    test("When showing the rename dialog, it triggers 'rename:error' then the dialog is sent unsuccessfully", function () {
        this.assertErrorEvent(RenameDialog, "showRenameDialog", "rename:error");
    });

    test("When submitting the rename dialog, it saves the new name in the model", function () {
        var model = this.assertSuccessEvent(RenameDialog, "showRenameDialog", "rename:success", {name: "filter name"});
        equal(model.get("name"), "filter name", "The filter has the new name");
        equal(model, this.filterModel, "The affected model is returned in the event's arguments");
    });

    test("When showing the copy dialog, it triggers 'copy:success' then the dialog is sent successfully", function () {
        this.assertSuccessEvent(CopyDialog, "showCopyDialog", "copy:success", {id: "2345", subscriptions: {items: []}});
    });

    test("When showing the copy dialog, it triggers 'copy:error' then the dialog is sent unsuccessfully", function () {
        this.assertErrorEvent(CopyDialog, "showCopyDialog", "copy:error");
    });

    test("When submitting the save as dialog, it returns the new model in the event arguments", function () {
        var model = this.assertSuccessEvent(CopyDialog, "showCopyDialog", "copy:success", {
            id: "2345",
            subscriptions: {items: []}
        });
        ok(model instanceof Filter, "It includes a FilterModel");
        equal(model.get("id"), "2345", "The new model contains the attributes from the response");
    });

    test("When showing the save-as dialog, it triggers 'saveas:success' then the dialog is sent successfully", function () {
        this.assertSuccessEvent(SaveAsDialog, "showSaveAsDialog", "saveas:success", {
            id: "2345",
            subscriptions: {items: []}
        });
    });

    test("When showing the save-as dialog, it triggers 'saveas:error' then the dialog is sent unsuccessfully", function () {
        this.assertErrorEvent(SaveAsDialog, "showSaveAsDialog", "saveas:error");
    });

    test("When submitting the save-as dialog, it returns the new model in the event arguments", function () {
        var model = this.assertSuccessEvent(SaveAsDialog, "showSaveAsDialog", "saveas:success", {
            id: "2345",
            subscriptions: {items: []}
        });
        ok(model instanceof Filter, "It includes a FilterModel");
        equal(model.get("id"), "2345", "The new model contains the attributes from the response");
    });

    test("When showing a dialog, it closes the previous one", function () {
        var dialogController = this.buildDialogController();
        this.stub(RenameDialog.prototype, "render");
        this.stub(DeleteDialog.prototype, "render");

        dialogController.showRenameDialog();
        var closeStub = this.stub(dialogController.activeView, "close");
        dialogController.showDeleteDialog();

        sinon.assert.calledOnce(closeStub);
    });

    test("When showing a dialog, it is exposed in the activeView property", function () {
        var dialogController = this.buildDialogController();
        this.stub(RenameDialog.prototype, "render");

        dialogController.showRenameDialog();

        ok(dialogController.activeView instanceof RenameDialog);
    });
});
