AJS.test.require(["com.atlassian.jira.jira-issue-nav-plugin:backbone-ext"], function () {
    "use strict";

    require([
        "jira/components/util/functionalutils",
        "jira/issues/basemodel"
    ], function(
        FunctionalUtils,
        BaseModel
    ) {
        module('BaseModel');

        test("Can create BaseModel with no properties defined", function () {
            var ModelConstructor = BaseModel.extend();
            var model = new ModelConstructor();
            ok(model instanceof BaseModel);
        });

        test("BaseModel adds mixin", function () {
            var TestMixin = {
                mixinProperty: "hi"
            };

            var TestModel = BaseModel.extend({
                mixins: [TestMixin]
            });
            var testModelInstance = new TestModel();

            equal(testModelInstance.mixinProperty, "hi");
        });

        test("Mixin name clash fails violently on class declaration", function () {
            var TestMixin1 = {
                mixinProperty: "hi"
            };
            var TestMixin2 = {
                mixinProperty: "hi"
            };

            raises(function () {
                BaseModel.extend({
                    mixins: [TestMixin1, TestMixin2]
                });
            });
        });

        test("Mixin initialize is mixed in when class does not contain initialize method", function () {
            var TestMixin = {
                initialize: function () {
                    this.mixin1Initialized = true;
                }
            };
            var TestModel = BaseModel.extend({
                mixins: [TestMixin]
            });

            var testModelInstance = new TestModel();

            ok(testModelInstance.mixin1Initialized);
        });

        test("Class initialize is called when mixin initialize is specified", function () {
            var TestMixin = {
                initialize: function () {
                    this.myMixinHas = "beenCalled";
                }
            };

            var TestMixin2 = {
                initialize: function () {
                    this.myMixin2Has = "beenCalled";
                }
            };

            var TestModel = BaseModel.extend({
                mixins: [TestMixin, TestMixin2],
                initialize: function () {
                    this.classInitializeHas = "beenCalled";
                }
            });
            var testModelInstance = new TestModel();

            equal(testModelInstance.myMixinHas, "beenCalled");
            equal(testModelInstance.myMixin2Has, "beenCalled");
            equal(testModelInstance.classInitializeHas, "beenCalled");
        });

        test("Mixin defaults compose with class defaults", function () {
            var TestMixin = {
                defaults: {
                    yada: "yada"
                }
            };

            var TestModel = BaseModel.extend({
                mixins: [TestMixin],
                defaults: {
                    blah: "blah"
                }
            });

            deepEqual(TestModel.prototype.defaults, {
                blah: "blah",
                yada: "yada"
            });
        });

        test("Mixin defaults replace class defaults if class has no defaults specified", function () {
            var TestMixin = {
                defaults: {
                    yada: "yada"
                }
            };

            var TestModel = BaseModel.extend({
                mixins: [TestMixin]
            });

            deepEqual(TestModel.prototype.defaults, {
                yada: "yada"
            });
        });

        test("Mixin validate is mixed in when class does not contain validate method", function () {
            var TestMixin = {
                validate: function () {
                    this.mixin1Validated = true;
                }
            };
            var TestModel = BaseModel.extend({
                mixins: [TestMixin]
            });

            var testModelInstance = new TestModel();
            testModelInstance.set({}, {validate: true}); // cause validation

            ok(testModelInstance.mixin1Validated);
        });

        test("Validate is composed in multiple mixins", function () {
            var TestMixin1 = {
                validate: function () {
                    this.mixin1Validated = true;
                }
            };
            var TestMixin2 = {
                validate: function () {
                    this.mixin2Validated = true;
                }
            };
            var TestModel = BaseModel.extend({
                mixins: [TestMixin1, TestMixin2],
                validate: function () {
                    this.modelValidated = true;
                }
            });

            var testModelInstance = new TestModel();
            testModelInstance.set({}, {validate: true}); // cause validation

            ok(testModelInstance.mixin1Validated);
            ok(testModelInstance.mixin2Validated);
            ok(testModelInstance.modelValidated);
        });

        test("When a mixin's validate function fails, it is returned before the original function", function () {
            var TestMixin = {
                validate: function () {
                    return "you suck!";
                }
            };
            var TestModel = BaseModel.extend({
                mixins: [TestMixin]
            });

            var onInvalid = sinon.spy();
            var testModelInstance = new TestModel();
            testModelInstance.on("invalid", onInvalid);
            testModelInstance.set({}, {validate: true});

            ok(onInvalid.called);
            equal(onInvalid.firstCall.args[1], "you suck!");
        });

        test("Attributes Mixin", function () {

            var ContactModel = BaseModel.extend({
                properties: ["name", "number", "address"]
            });

            var myContactModel = new ContactModel({
                name: "scott",
                number: 412947430
            });

            equal("scott", myContactModel.getName());
            equal(412947430, myContactModel.getNumber());

            myContactModel.setName("jonothan");
            myContactModel.setNumber(0);


            equal("jonothan", myContactModel.getName());
            equal(0, myContactModel.getNumber());
        });


        test("Attributes Mixin passes options to underlying set", function () {

            var ContactModel = BaseModel.extend({
                properties: ["name", "number", "address"]
            });

            var myContactModel = new ContactModel({
                name: "scott",
                number: 412947430
            });
            var spy = sinon.spy();

            myContactModel.bind("change:name", spy);

            myContactModel.setName("jonothan");
            myContactModel.setNumber(0);

            ok(spy.calledOnce, "Event trigged after set");

            myContactModel.setName("jared", {silent: true});

            ok(spy.calledOnce, "Event not trigged with silent passed to set");
        });

        test("lambda λ", function () {
            equal(FunctionalUtils.lambda("fred")(), "fred");

            var o = FunctionalUtils.lambda({what: "an object", foo: 123})();
            equal(o.what, "an object");
            equal(o.foo, 123);

            equal(FunctionalUtils.lambda(FunctionalUtils.lambda)(), FunctionalUtils.lambda);
        });

        test("bindObjectTo", function () {
            var obj = {
                method1: function () {
                    return this.value1;
                },
                method2: function () {
                    return this.value2;
                }
            };
            var values = {
                value1: "1",
                value2: "2"
            };

            FunctionalUtils.bindObjectTo(obj, values);

            equal(obj.method1(), "1");
            equal(obj.method2(), "2");
        });
});


});
