define("jira/components/test-utils/fakeserver", ['require'], function (require) {
    "use strict";

    var wrmContextPath = require('wrm/context-path');
    var _ = require('jira/components/libs/underscore');

    /**
     * This method transforms a request body like 'a=1&b=2&b=3' into an object like
     * {
     *    a: 1
     *    b: [2, 3]
     * }
     */
    function parseRequestBody(request) {

        var variables = request.requestBody.split("&");

        var result = {};
        _.each(variables, function(chunk) {
            var chunks = chunk.split("=");
            var key = chunks[0];
            var value = chunks[1] || "";

            if (key in result) {
                result[key] = [].concat(result[key], value);
            }else{
                result[key] = value;
            }
        });
        return result;
    }

    function createIssueSearchResponse(issues) {
        return {
            "issueTable": {
                "displayed": 2,
                "end": 2,
                "issueIds": _.pluck(issues, "id"),
                "issueKeys": _.pluck(issues, "key"),
                "page": 0,
                "pageSize": 2,
                "startIndex": 0,
                "table": [ issues[0], issues[1] ],
                "total": 1100
            }
        };
    }

    function createStableSearchResponse(requestParameteres, issues) {
        // Note: the pagination state for a stable response is broken in the actual server implementation. This
        // mocked has the same problems (e.g. total == pageSize, page always 0...) for the sake of testing.
        requestParameteres.id = [].concat(requestParameteres.id);
        return {
            "issueTable": {
                "displayed": requestParameteres.id.length,
                "end": requestParameteres.id.length,
                "page": 0,
                "pageSize": requestParameteres.id.length,
                "startIndex": 0,
                "table": _.map(requestParameteres.id, function (id) {
                    return _.findWhere(issues, {id: Number(id)});
                }),
                "total": requestParameteres.id.length
            }
        };
    }

    /**
     * Creates a sinon FakeServer that mimics the response of /rest/issueNav/1/issueTable (regular search) and
     * /rest/issueNav/1/issueTable/stable (stable search). It does not perform any search though, it always returns
     * the issues provided in the argument. It simulates a pageSize of 2.
     *
     * It receives a list of issues to work with. Each issue have the format:
     *
     *   {
     *       id: 3,
     *       key: 'DEMO-3',
     *       summary: 'The third issue in the project',
     *       status: "Open",
     *       type: {
     *           name: 'bug',
     *           description: 'A bug in your application',
     *           icon: 'iconUrl.png'
     *       }
     *   }
     *
     * @param {Object} sinon Sinon implementation used to create the fake server
     * @param {Array} issues List of issues.
     * @return {Object} Sinon's Fake Server
     */
    return {
        create: function(sinon, issues) {
            var server = sinon.fakeServer.create();
            server.respondWith("POST", wrmContextPath() + "/rest/issueNav/1/issueTable", function (request) {
                var response = createIssueSearchResponse(issues);

                request.respond(200, { "Content-Type": "application/json" }, JSON.stringify(response));
            });

            server.respondWith("POST", wrmContextPath() + "/rest/issueNav/1/issueTable/stable", function (request) {
                var requestParameteres = parseRequestBody(request);

                var response = createStableSearchResponse(requestParameteres, issues);

                request.respond(200, { "Content-Type": "application/json" }, JSON.stringify(response));
            });

            return server;
        },
        createIssueSearchResponse: createIssueSearchResponse,
        createStableSearchResponse: createStableSearchResponse
    };
});
