AJS.test.require("com.atlassian.jira.jira-issue-nav-components:simpleissuelist-test", function() {
    "use strict";

    require([
        "jira/components/search/results",
        "jira/components/simpleissuelist/views/issuelist",
        "jira/components/simpleissuelist/views/issue",
        "jira/components/simpleissuelist/views/inaccessibleissue"
    ], function(
        Results,
        IssueList,
        Issue,
        InaccessibleIssue
    ) {
        module("jira/components/simpleissuelist/views/issuelist", {
            setup: function() {
                this.collection = new Results([
                    {id: 1, key: 'KEY-1', type: {}},
                    {id: 2, key: 'KEY-2', type: {}},
                    {id: 3, key: 'KEY-3', type: {}}
                ]);

                this.view = new IssueList();
                this.view.render();
                this.view.update(this.collection);
            }
        });

        test("When unselecting an issue, it unhighlights the issue", function() {
            var unhighlightStub = this.stub(this.view.children.findByIndex(0), "unhighlight");

            this.view.unselectIssue(1);

            sinon.assert.calledOnce(unhighlightStub);
        });

        test("When selecting an issue, it highlights the issue", function() {
            var highlightStub = this.stub(this.view.children.findByIndex(0), "highlight");

            this.view.selectIssue(1);

            sinon.assert.calledOnce(highlightStub);
        });

        test("When updating the view with a new collection, it highlights the selected issue", function() {
            this.collection.selected = this.collection.get(1);
            // 'this.view.children' will be overwritten by 'this.view.render()', so we need to use the Issue view prototype
            // to create stub the 'highlight' method
            var highlightStub = this.stub(Issue.prototype, "highlight");

            this.view.update(this.collection);

            sinon.assert.calledOnce(highlightStub);
            equal(highlightStub.getCall(0).thisValue.model, this.collection.selected);
        });

        test("When updating the view with a new collection, it triggers the 'update' event", function() {
            var onUpdate = this.spy();
            this.view.on("update", onUpdate);

            this.view.update(this.collection);

            sinon.assert.calledOnce(onUpdate);
        });

        test("When updating the view with a new collection, it renders the new data", function() {
            var newCollection = new Results([
                {id: 4, key: 'KEY-4', type: {}},
                {id: 5, key: 'KEY-5', type: {}}
            ]);

            this.view.update(newCollection);

            equal(this.view.children.findByIndex(0).$el.attr("data-key"), "KEY-4");
            equal(this.view.children.findByIndex(1).$el.attr("data-key"), "KEY-5");
        });

        test("It can scroll the selected issue into view", function() {
            this.collection.selected = this.collection.get(3);
            var scrollIntoView = this.spy(this.view.children.findByIndex(2), "scrollIntoView");

            this.view.scrollSelectedIssueIntoView();

            sinon.assert.calledOnce(scrollIntoView);
        });

        test("When creating the view for each issue, it uses the Issue view for valid issues", function() {
            var issue = this.collection.get(1);

            var viewConstructor = this.view.getChildView(issue);

            equal(viewConstructor, Issue);
        });

        test("When creating the view for each issue, it uses the InaccessibleIssue view for inaccessible issues", function() {
            var issue = this.collection.get(1);
            issue.set('inaccessible', true);

            var viewConstructor = this.view.getChildView(issue);

            equal(viewConstructor, InaccessibleIssue);
        });

        test("When updating an issue, it triggers the 'update' event", function() {
            var issue = this.collection.get(1);
            var onUpdate = this.spy();
            this.view.on("update", onUpdate);

            this.view.updateIssue(issue);

            sinon.assert.calledOnce(onUpdate);
        });

        test("When updating an issue, it updates the data of that issue", function() {
            var issue = this.collection.get(1);
            issue.set('summary', "This is a test");

            this.view.updateIssue(issue);

            var issueView = this.view.children.findByModel(issue);
            ok(issueView.$el.text().indexOf("This is a test") > -1);
        });

        test("When updating an invalidated issue, it updates the row of that issue with the inaccessible issue message", function() {
            var issue = this.collection.get(1);
            issue.set('inaccessible', true);

            this.view.updateIssue(issue);

            var issueView = this.view.children.findByModel(issue);
            //Lets check the classname to avoid using the I18N string
            ok(issueView.$el.is(".inaccessible-issue"));
        });

        test("Focusing the list delegates the call into the first itemView", function() {
            this.spy(this.view.children.findByIndex(0), "focus");
            this.spy(this.view.children.findByIndex(1), "focus");
            this.spy(this.view.children.findByIndex(2), "focus");

            this.view.focus();

            sinon.assert.calledOnce(this.view.children.findByIndex(0).focus);
            sinon.assert.notCalled(this.view.children.findByIndex(1).focus);
            sinon.assert.notCalled(this.view.children.findByIndex(2).focus);
        });
    });
});
