AJS.test.require("com.atlassian.jira.jira-issue-nav-components:simpleissuelist-test", function() {
    "use strict";

    require([
        "jira/components/test-utils/fakeserver",
        "jira/components/test-utils/marionettemocker",
        "jira/components/test-utils/mockutils",
        "jira/components/search",
        "jira/components/simpleissuelist/controllers/list",
        "jira/components/simpleissuelist/services/api",
        "jira/components/simpleissuelist/services/inline-issue-create",
        "jquery"
    ], function(
        FakeServer,
        MarionetteMocker,
        MockUtils,
        Search,
        ListController,
        API,
        InlineIssueCreate,
        jQuery
    ) {
        var ISSUES = [
            {
                id: 1,
                key: 'DEMO-1',
                summary: 'The first issue in the project',
                status: "Open",
                type: {
                    name: 'bug',
                    description: 'A bug in your application',
                    icon: 'iconUrl.png'
                }
            },
            {
                id: 2,
                key: 'DEMO-2',
                summary: 'The second issue in the project',
                status: "Open",
                type: {
                    name: 'bug',
                    description: 'A bug in your application',
                    icon: 'iconUrl.png'
                }
            },
            {
                id: 3,
                key: 'DEMO-3',
                summary: 'The third issue in the project',
                status: "Open",
                type: {
                    name: 'bug',
                    description: 'A bug in your application',
                    icon: 'iconUrl.png'
                }
            }
        ];

        module("jira/components/simpleissuelist", {
            setup: function() {
                this.listController = MarionetteMocker.createEventedMock(sinon, ListController);
                this.inlineIssueCreate = MarionetteMocker.createEventedMock(sinon, InlineIssueCreate);

                this.SimpleIssueListConstructor = MockUtils.requireWithMocks("jira/components/simpleissuelist", {
                    "jira/components/simpleissuelist/controllers/list": this.listController.constructor,
                    "jira/components/simpleissuelist/services/inline-issue-create": this.inlineIssueCreate.constructor
                });
                this.simpleIssueList = new this.SimpleIssueListConstructor();
                this.simpleIssueList.show(jQuery(JIRA.Components.SimpleIssueList.Templates.layout()));

                this.server = FakeServer.create(sinon, ISSUES);
                this.search = new Search();
            },

            searchAndGetResults: function(issueKey) {
                var searchResults;
                this.search.search().done(function(results){ searchResults = results; });
                this.server.respond();

                this.simpleIssueList.load(searchResults, issueKey);
                this.server.respond();

                return searchResults;
            },

            searchAndGetEmptyResults: function() {
                var searchResults;
                this.search.search().done(function(results){ searchResults = results; });
                this.server.requests[0].respond(200, { "Content-Type": "application/json" }, JSON.stringify({"issueTable": {}}));

                this.simpleIssueList.load(searchResults);

                return searchResults;
            }
        });

        test("When loading a search, it shows the results in the list", function() {
            var searchResults = this.searchAndGetResults();

            sinon.assert.calledOnce(this.listController.update);
            sinon.assert.calledWith(this.listController.update, searchResults);
        });

        test("When loading a search, it there is no results, it does not show the list view", function() {
            this.searchAndGetEmptyResults();

            sinon.assert.notCalled(this.listController.update);
        });

        test("When loading a search, it loads the first page and selects the first issue", function() {
            var searchResults = this.searchAndGetResults("DEMO-1");

            equal(searchResults.state.currentPage, 0);
            equal(searchResults.selected.get("key"), "DEMO-1");
        });

        test("When loading a new search, it stop listening events from the old search", function() {
            var oldSearchResults = this.searchAndGetResults();
            this.searchAndGetResults();
            oldSearchResults.get(2).set('summary', 'New Summary');

            sinon.assert.notCalled(this.listController.updateIssue);
        });

        test("When loading a search for a particular key, it loads the page for that key and selects it", function() {
            var searchResults = this.searchAndGetResults('DEMO-3');

            equal(searchResults.state.currentPage, 1);
            equal(searchResults.selected.get("key"), "DEMO-3");
        });

        test("When an issue is selected in the collection, it is marked as selected in the list", function() {
            var searchResults = this.searchAndGetResults();
            this.listController.selectIssue.reset(); // It has been called by searchAndGetResults()

            searchResults.select("DEMO-2");

            sinon.assert.calledOnce(this.listController.selectIssue);
            sinon.assert.calledWith(this.listController.selectIssue, 2);
        });

        test("When an issue is selected in the collection, it triggers the 'select' event", function() {
            var searchResults = this.searchAndGetResults();
            var onSelect = this.spy();
            this.simpleIssueList.on("select", onSelect);

            searchResults.select("DEMO-2");

            sinon.assert.calledOnce(onSelect);
            sinon.assert.calledWith(onSelect, {id: 2, key: 'DEMO-2'});
        });

        test("When an issue is unselected in the collection, it is marked as unselected in the list", function() {
            var searchResults = this.searchAndGetResults();

            searchResults.unselect();

            sinon.assert.calledOnce(this.listController.unselectIssue);
            sinon.assert.calledWith(this.listController.unselectIssue, 1);
        });

        test("When an issue is changed in the collection, it updates the issue in the list", function() {
            var searchResults = this.searchAndGetResults();

            // Update the summary of DEMO-3
            searchResults.get(2).set('summary', 'New Summary');

            sinon.assert.calledOnce(this.listController.updateIssue);
            sinon.assert.calledWith(this.listController.updateIssue, searchResults.get(2));
        });

        test("When the list wants to go to the previous page, the collection jumps to the previous page", function() {
            var searchResults = this.searchAndGetResults('DEMO-3');
            this.spy(searchResults, "jumpToPage");

            this.listController.trigger("goToPreviousPage");

            sinon.assert.calledOnce(searchResults.jumpToPage);
            sinon.assert.calledWith(searchResults.jumpToPage, "prev");
        });

        test("When the list wants to go to the next page, the collection jumps to the next page", function() {
            var searchResults = this.searchAndGetResults();
            this.spy(searchResults, "jumpToPage");

            this.listController.trigger("goToNextPage");

            sinon.assert.calledOnce(searchResults.jumpToPage);
            sinon.assert.calledWith(searchResults.jumpToPage, "next");
        });

        test("When the list wants to go to an arbitrary page, the collection jumps to that page", function() {
            var searchResults = this.searchAndGetResults();
            this.spy(searchResults, "jumpToPage");

            this.listController.trigger("goToPage", 1);

            sinon.assert.calledOnce(searchResults.jumpToPage);
            sinon.assert.calledWith(searchResults.jumpToPage, 1);
        });

        test("When the list wants to go to refresh the search, the component triggers the 'refresh' event too", function() {
            var onRefresh = this.spy();
            this.simpleIssueList.on('refresh', onRefresh);

            this.listController.trigger("refresh");

            sinon.assert.calledOnce(onRefresh);
        });

        test("When the list wants to select an issue, the collection selects that issue", function() {
            var searchResults = this.searchAndGetResults();
            this.spy(searchResults, "select");

            this.listController.trigger("selectIssue", {id: 2, key: "DEMO-2"});

            sinon.assert.calledOnce(searchResults.select);
            sinon.assert.calledWith(searchResults.select, 2);
        });

        test("When the list wants to select an issue, it triggers the 'list:select' event with the issue information", function() {
            this.searchAndGetResults();
            var onSelect = this.spy();
            this.simpleIssueList.on("list:select", onSelect);

            this.listController.trigger("selectIssue", {id: 2, key: "DEMO-2"});

            sinon.assert.calledOnce(onSelect);
            sinon.assert.calledWith(onSelect, {id: 2, key: "DEMO-2", relativePosition: 1, absolutePosition: 1});
        });

        test("When the list wants to sort the results, the component triggers the 'sort' event with the JQL", function() {
            var onSort = this.spy();
            this.simpleIssueList.on('sort', onSort);

            this.listController.trigger("sort", "project=DEMO order by issuekey DESC");

            sinon.assert.calledOnce(onSort);
            sinon.assert.calledWith(onSort, "project=DEMO order by issuekey DESC");
        });

        test("When the list is updated, the component triggers the update event", function() {
            var onUpdate = this.spy();
            this.simpleIssueList.on("update", onUpdate);

            this.listController.trigger("update");

            sinon.assert.calledOnce(onUpdate);
        });

        test("When the component is created, it initializes the API", function() {
            this.spy(API, "init");

            var SimpleIssueList = require("jira/components/simpleissuelist");
            this.simpleIssueList = new SimpleIssueList();

            sinon.assert.calledOnce(API.init);
        });

        test("When disabling an issue, it sets the propper attribute in the model", function() {
            var searchResults = this.searchAndGetResults();

            this.simpleIssueList.disableIssue(1);

            equal(searchResults.get(1).get('inaccessible'), true);
        });

        test("When the 'before:loadpage' event is triggered, it retriggers it", function() {
            var searchResults = this.searchAndGetResults();
            this.spy(this.simpleIssueList, "trigger");

            searchResults.trigger("before:loadpage");

            sinon.assert.calledOnce(this.simpleIssueList.trigger);
            sinon.assert.calledWith(this.simpleIssueList.trigger, "before:loadpage");
        });

        test("When the 'error:loadpage' event is triggered, it retriggers it with the given payload", function() {
            var payload = {data: 'some text'};
            var searchResults = this.searchAndGetResults();
            this.spy(this.simpleIssueList, "trigger");

            searchResults.trigger("error:loadpage", payload);

            sinon.assert.calledOnce(this.simpleIssueList.trigger);
            sinon.assert.calledWith(this.simpleIssueList.trigger, "error:loadpage", payload);
        });

        test("When the 'displayInlineIssueCreate' option is passed when instantiating the class, it creates an InlineIssueCreate view", function() {
            new this.SimpleIssueListConstructor({
                displayInlineIssueCreate: true
            });

            sinon.assert.calledOnce(this.inlineIssueCreate.constructor);
        });

        test("When the InlineIssueCreate triggers the event 'issueCreated', it is re-triggered", function() {
            var simpleIssueList = new this.SimpleIssueListConstructor({
                displayInlineIssueCreate: true
            });
            var onIssueCreated = this.spy();
            simpleIssueList.on("issueCreated", onIssueCreated);
            var issueInfo = {};

            this.inlineIssueCreate.trigger("issueCreated", issueInfo);

            sinon.assert.calledOnce(onIssueCreated);
        });

        test("When the InlineIssueCreate triggers the event 'activated' event, we adjust the size of the list (deferred)", function() {
            var clock = this.sandbox.useFakeTimers();
            new this.SimpleIssueListConstructor({
                displayInlineIssueCreate: true
            });

            this.inlineIssueCreate.trigger("activated");

            sinon.assert.notCalled(this.listController.adjustSize);
            clock.tick(100);
            sinon.assert.calledOnce(this.listController.adjustSize);
        });

        test("When the InlineIssueCreate triggers the event 'deactivated' event, we adjust the size of the list (deferred)", function() {
            var clock = this.sandbox.useFakeTimers();
            new this.SimpleIssueListConstructor({
                displayInlineIssueCreate: true
            });

            this.inlineIssueCreate.trigger("deactivated");

            sinon.assert.notCalled(this.listController.adjustSize);
            clock.tick(100);
            sinon.assert.calledOnce(this.listController.adjustSize);
        });

        test("When we load a new set of results, it sets the JQL in the InlineIssueCreate component if present", function() {
            var simpleIssueList = new this.SimpleIssueListConstructor({
                displayInlineIssueCreate: true
            });
            var searchResults;
            this.search.search().done(function(results){ searchResults = results; });
            this.server.respond();
            searchResults.jql = "project='TEST'";

            simpleIssueList.load(searchResults);

            sinon.assert.calledOnce(this.inlineIssueCreate.setJQL);
            sinon.assert.calledWith(this.inlineIssueCreate.setJQL, "project='TEST'");
        });

        test("When we show a SimpleIssueList with InlineIssueCreate component, it passes the component to the list", function() {
            this.listController.constructor.reset();
            new this.SimpleIssueListConstructor({
                displayInlineIssueCreate: true
            });

            sinon.assert.calledOnce(this.listController.constructor);
            ok(this.listController.constructor.lastCall.args[0].inlineIssueCreate === this.inlineIssueCreate);
        });

        test("Focusing the list delegates into the List controller", function() {
            this.simpleIssueList.focus();

            sinon.assert.calledOnce(this.listController.focus);
        });
    });
});
