AJS.test.require("com.atlassian.jira.jira-issue-nav-components:search-test", function() {
    "use strict";

    require([
        "jira/components/test-utils/fakeserver",
        "jira/components/search/results",
        "jira/components/libs/underscore",
        "jira/components/libs/backbone.paginator"
    ], function(
        FakeServer,
        Results,
        _,
        PageableCollection
    ) {
        var ISSUES = [
            {
                id: 1,
                key: 'DEMO-1',
                summary: 'The first issue in the project',
                status: "Open",
                type: {
                    name: 'bug',
                    description: 'A bug in your application',
                    icon: 'iconUrl.png'
                }
            },
            {
                id: 2,
                key: 'DEMO-2',
                summary: 'The second issue in the project',
                status: "Open",
                type: {
                    name: 'bug',
                    description: 'A bug in your application',
                    icon: 'iconUrl.png'
                }
            },
            {
                id: 3,
                key: 'DEMO-3',
                summary: 'The third issue in the project',
                status: "Open",
                type: {
                    name: 'bug',
                    description: 'A bug in your application',
                    icon: 'iconUrl.png'
                }
            },
            {
                id: 4,
                key: 'DEMO-4',
                summary: 'The fourth issue in the project',
                status: "Open",
                type: {
                    name: 'bug',
                    description: 'A bug in your application',
                    icon: 'iconUrl.png'
                }
            }
        ];

        module("jira/components/search/results", {
            setup: function() {
                this.searchResults = new Results([], {
                    issues: _.map(ISSUES, function(issue) { return {id: issue.id, key: issue.key}; } ),
                    pageSize: 2,
                    totalRecordsInSearch: 4,
                    totalRecordsInDB: 5,
                    jql: "project=DEMO"
                });
                this.server = FakeServer.create(sinon, ISSUES);
            },

            serverAlwaysReturnsError: function() {
                this.server.restore();
                this.server.respondWith([ 400, {}, "" ]);
            },

            goToLastIssueOfFirstPage: function() {
                this.searchResults.jumpToPageForIssue("DEMO-2");
                this.server.respond();
            },

            goToFirstIssueOfSecondPage: function() {
                this.searchResults.jumpToPageForIssue("DEMO-3");
                this.server.respond();
            }
        });

        test("When requesting a page, it does a request to the correct URL using POST", function() {
            this.searchResults.getPage(0);

            var lastRequest = _.last(this.server.requests);
            equal(lastRequest.url, AJS.contextPath() + "/rest/issueNav/1/issueTable/stable");
            equal(lastRequest.method, "POST");
        });

        test("When requesting a page, it asks the server to skip token validation", function() {
            this.searchResults.getPage(0);

            equal(_.last(this.server.requests).requestHeaders["X-Atlassian-Token"], "no-check");
        });

        test("When requesting a page, it sends the IDs to the server", function() {
            this.searchResults.getPage(0);

            var lastRequest = _.last(this.server.requests);
            ok(lastRequest.requestBody.indexOf("id=1" > -1 ), "Request Body contains the ID of the first issue");
            ok(lastRequest.requestBody.indexOf("id=2" > -1 ), "Request Body contains the ID of the second issue");
        });

        test("When requesting a page, it uses the 'split-view' layout", function() {
            this.searchResults.getPage(0);

            var lastRequest = _.last(this.server.requests);
            ok(lastRequest.requestBody.indexOf("layoutKey=split-view" > -1 ), "Request Body contains the layout key");
        });

        test("When jumping to a page, it requests that page from the server", function() {
            this.spy(this.searchResults, "getPage");

            this.searchResults.jumpToPage(0);

            sinon.assert.calledOnce(this.searchResults.getPage);
            sinon.assert.calledWith(this.searchResults.getPage, 0);
        });

        test("When jumping to a page, it selects the first issue in that page", function() {
            this.searchResults.jumpToPage("last");
            this.server.respond();

            equal(this.searchResults.selected.get("key"), "DEMO-3");
        });

        test("When jumping to a page, it resolves the promise with the length of the requested page", function() {
            var onResolve = this.spy();
            this.searchResults.jumpToPage("last").done(onResolve);
            this.server.respond();

            sinon.assert.calledOnce(onResolve);
            sinon.assert.calledWith(onResolve, 2);
        });

        test("When jumping to the page for an issue, it selects that page and that issue", function() {
            this.searchResults.jumpToPageForIssue("DEMO-3");
            this.server.respond();

            equal(this.searchResults.state.currentPage, 1, "Second page is selected");
            equal(this.searchResults.selected.get("key"), "DEMO-3", "Issue is selected");
        });

        test("When jumping to the page for an issue, it selects that first page and first issue if the issue is not in the search", function() {
            this.searchResults.jumpToPageForIssue("DEMO-42");
            this.server.respond();

            equal(this.searchResults.state.currentPage, 0, "First page is selected");
            equal(this.searchResults.selected.get("key"), "DEMO-1", "First issue is selected");
        });

        test("It can select the next issue in the page", function() {
            this.searchResults.jumpToPage(0);
            this.server.respond();

            this.searchResults.selectNext();

            equal(this.searchResults.selected.get("key"), "DEMO-2", "Second issue is selected");
        });

        test("If the last issue in the page is selected, selecting the next issue again will load another page", function() {
            this.searchResults.jumpToPage(0);
            this.server.respond();
            this.searchResults.selectNext(); // Pagesize = 2, so this selects the last issue in the page

            this.searchResults.selectNext();
            this.server.respond();

            equal(this.searchResults.state.currentPage, 1, "Second page is selected");
            equal(this.searchResults.selected.get("key"), "DEMO-3", "Third issue is selected");
        });

        test("When selecting the page, the current position is returned", function() {
            this.searchResults.jumpToPage(0);
            this.server.respond();
            var data = this.searchResults.selectNext();
            deepEqual(data, {current: 1, total: 2}, "Returned data is correct");
            data = this.searchResults.selectPrev();
            deepEqual(data, {current: 0, total: 2}, "Returned data is correct");
        });

        test("It can select the previous issue in the page", function() {
            this.searchResults.jumpToPage(0);
            this.server.respond();
            this.searchResults.selectNext(); // Pagesize = 2, so this selects the last issue in the page

            this.searchResults.selectPrev();

            equal(this.searchResults.selected.get("key"), "DEMO-1", "First issue is selected");
        });

        test("If the first issue in the page is selected, selecting the prev issue again will load another page", function() {
            this.searchResults.jumpToPageForIssue("DEMO-3");
            this.server.respond();

            this.searchResults.selectPrev();
            this.server.respond();

            equal(this.searchResults.state.currentPage, 0, "First page is selected");
            equal(this.searchResults.selected.get("key"), "DEMO-2", "Second issue is selected");
        });

        test("If the first issue is selected and we try to select the previous issue, nothing happens", function() {
            this.searchResults.jumpToPage(0);
            this.server.respond();

            this.searchResults.selectPrev();

            equal(this.server.requests.length, 1, "No new requests");
            equal(this.searchResults.state.currentPage, 0, "First page is still selected");
            equal(this.searchResults.selected.get("key"), "DEMO-1", "First issue is selected");
        });

        test("If the last issue is selected and we try to select the next issue, nothing happens", function() {
            this.searchResults.jumpToPageForIssue("DEMO-4");
            this.server.respond();

            this.searchResults.selectNext();

            equal(this.server.requests.length, 1, "No new requests");
            equal(this.searchResults.state.currentPage, 1, "Second page is still selected");
            equal(this.searchResults.selected.get("key"), "DEMO-4", "Fourth issue is selected");
        });

        test("It can select the first issue in the page", function() {
            this.searchResults.jumpToPage(0);
            this.server.respond();
            this.searchResults.selectNext();

            this.searchResults.select("first");
            equal(this.searchResults.selected.get("key"), "DEMO-1");
        });

        test("It can select the last issue in the page", function() {
            this.searchResults.jumpToPage(0);
            this.server.respond();

            this.searchResults.select("last");
            equal(this.searchResults.selected.get("key"), "DEMO-2");
        });

        test("It can select any issue based on the model", function() {
            this.searchResults.jumpToPage(0);
            this.server.respond();

            this.searchResults.select(this.searchResults.get(2));

            equal(this.searchResults.selected.get("key"), "DEMO-2");
        });

        test("It can select any issue based on the key", function() {
            this.searchResults.jumpToPage(0);
            this.server.respond();

            this.searchResults.select("DEMO-2");

            equal(this.searchResults.selected.get("key"), "DEMO-2");
        });

        test("It can select any issue based on the ID", function() {
            this.searchResults.jumpToPage(0);
            this.server.respond();

            this.searchResults.select(2);

            equal(this.searchResults.selected.get("key"), "DEMO-2");
        });

        test("When selecting an issue, it will trigger the 'select' event on the selected issue", function() {
            this.searchResults.jumpToPage(0);
            this.server.respond();
            var issueToSelect = this.searchResults.get(2);
            var spy = this.spy();
            issueToSelect.on("select", spy);

            this.searchResults.select(issueToSelect);

            sinon.assert.calledOnce(spy);
            sinon.assert.calledWith(spy, issueToSelect);
        });

        test("When selecting an issue, it will trigger the 'select' event on the collection too", function() {
            this.searchResults.jumpToPage(0);
            this.server.respond();
            var issueToSelect = this.searchResults.get(2);
            var spy = this.spy();
            this.searchResults.on("select", spy);

            this.searchResults.select(issueToSelect);

            sinon.assert.calledOnce(spy);
            sinon.assert.calledWith(spy, issueToSelect);
        });

        test("It can unselect the selected issue", function() {
            this.searchResults.jumpToPage(0);
            this.server.respond();

            this.searchResults.unselect();

            equal(this.searchResults.selected, null);
        });

        test("When unselecting an issue, it will trigger the 'unselect' event on the selected issue", function() {
            this.searchResults.jumpToPage(0);
            this.server.respond();
            var selectedIssue = this.searchResults.selected;
            var spy = this.spy();
            selectedIssue.on("unselect", spy);

            this.searchResults.unselect();

            sinon.assert.calledOnce(spy);
            sinon.assert.calledWith(spy, selectedIssue);
        });

        test("When unselecting an issue, it will trigger the 'unselect' event on the collection", function() {
            this.searchResults.jumpToPage(0);
            this.server.respond();
            var selectedIssue = this.searchResults.selected;
            var spy = this.spy();
            this.searchResults.on("unselect", spy);

            this.searchResults.unselect();

            sinon.assert.calledOnce(spy);
            sinon.assert.calledWith(spy, selectedIssue);
        });

        test("When selecting a new issue, it unselects the previously selected issue an issue", function() {
            this.searchResults.jumpToPage(0);
            this.server.respond();
            var selectedIssue = this.searchResults.selected;
            var spy = this.spy();
            this.searchResults.on("unselect", spy);

            this.searchResults.select(2);

            sinon.assert.calledWith(spy, selectedIssue);
        });

        test("When selecting an already selected issue, it does not unselects it first", function() {
            this.searchResults.jumpToPage(0);
            this.server.respond();
            var spy = this.spy();
            this.searchResults.on("unselect", spy);

            this.searchResults.select(1);

            sinon.assert.notCalled(spy);
        });

        test("When selecting an already selected issue, it still triggers the 'select' event", function() {
            this.searchResults.jumpToPage(0);
            this.server.respond();
            var spy = this.spy();
            this.searchResults.on("select", spy);

            this.searchResults.select(1);

            sinon.assert.calledOnce(spy);
        });

        test("It can update the selected issue", function() {
            this.searchResults.jumpToPage(0);
            this.server.respond();

            this.searchResults.updateSelectedIssue();

            var updateIssueUrl = _.last(this.server.requests).url;
            ok(updateIssueUrl.indexOf(AJS.contextPath() + "/rest/api/2/issue/1") > -1 );
        });

        test("It detects the end of stable search when we are in the last page", function() {
            this.searchResults.jumpToPage("last");
            this.server.respond();

            ok(this.searchResults.isAtTheEndOfStableSearch());
        });

        test("It does not detects the end of stable search if we are not in the last page", function() {
            this.searchResults.jumpToPage("first");
            this.server.respond();

            ok(!this.searchResults.isAtTheEndOfStableSearch());
        });

        test("It does not detects the end of stable search it there are no extra records in the database", function() {
            this.searchResults.totalRecordsInDB = 4; // Same number of records in the collection
            this.searchResults.jumpToPage("last");
            this.server.respond();

            ok(!this.searchResults.isAtTheEndOfStableSearch());
        });

        test("When removing an issue, it is removed from the list of issues", function() {
            this.searchResults.jumpToPage(0);
            this.server.respond();

            var model = this.searchResults.at(0);
            this.searchResults.removeAndUpdateSelectionIfNeeded(model);

            ok( !this.searchResults.contains(model), "Issue in collection" );
            ok( !_.any(this.searchResults.allIssues, function(issue) {return issue.id === model.id;}), "Issue in list of issues");
        });

        test("When removing an issue, it is removed from the pagination totals", function() {
            this.searchResults.jumpToPage(0);
            this.server.respond();

            var model = this.searchResults.at(0);
            this.searchResults.removeAndUpdateSelectionIfNeeded(model);

            equal( this.searchResults.state.totalRecords, 3, "Total records in the search results" ); //Initially it is 4
            equal( this.searchResults.totalRecordsInDB, 4, "Total records in the database" ); //Initially it is 5
        });

        test("When removing the selected issue, it selects the next issue in the list", function() {
            this.searchResults.jumpToPage(0);
            this.server.respond();
            this.searchResults.select("DEMO-1");

            this.searchResults.removeAndUpdateSelectionIfNeeded(this.searchResults.selected);
            this.server.respond();

            equal(this.searchResults.selected.get("key"), "DEMO-2");
        });

        test("When removing the selected issue, it selects the previous issue in the list if it was the last one", function() {
            this.searchResults.jumpToPage("last");
            this.server.respond();
            this.searchResults.select("DEMO-4");

            this.searchResults.removeAndUpdateSelectionIfNeeded(this.searchResults.selected);
            this.server.respond();

            equal(this.searchResults.selected.get("key"), "DEMO-3");
        });

        test("When removing the last issue, nothing gets selected", function() {
            this.searchResults.jumpToPage("first");
            this.server.respond();
            equal(this.searchResults.selected.get("key"), "DEMO-1");

            this.searchResults.removeAndUpdateSelectionIfNeeded(this.searchResults.at(0));
            this.server.respond();
            equal(this.searchResults.selected.get("key"), "DEMO-2");

            this.searchResults.removeAndUpdateSelectionIfNeeded(this.searchResults.at(0));
            this.server.respond();
            equal(this.searchResults.selected.get("key"), "DEMO-3");

            this.searchResults.removeAndUpdateSelectionIfNeeded(this.searchResults.at(0));
            this.server.respond();
            equal(this.searchResults.selected.get("key"), "DEMO-4");

            this.searchResults.removeAndUpdateSelectionIfNeeded(this.searchResults.at(0));
            equal(this.searchResults.selected, null);
        });

        test("When removing an issue, it resolves the promise with the length of the requested page", function() {
            this.searchResults.jumpToPage("first");
            this.server.respond();
            var onResolve;

            // Four issues left, page is size 2
            onResolve = this.spy();
            this.searchResults.removeAndUpdateSelectionIfNeeded(this.searchResults.at(0)).done(onResolve);
            this.server.respond();
            sinon.assert.calledOnce(onResolve);
            sinon.assert.calledWith(onResolve, 2);

            // Three issues left, page is still size 2
            onResolve = this.spy();
            this.searchResults.removeAndUpdateSelectionIfNeeded(this.searchResults.at(0)).done(onResolve);
            this.server.respond();
            sinon.assert.calledOnce(onResolve);
            sinon.assert.calledWith(onResolve, 2);

            //Only two issues left, page size after deletion is 1
            onResolve = this.spy();
            this.searchResults.removeAndUpdateSelectionIfNeeded(this.searchResults.at(0)).done(onResolve);
            this.server.respond();
            sinon.assert.calledOnce(onResolve);
            sinon.assert.calledWith(onResolve, 1);

            // Only one issue left, page size after deletion is 0
            onResolve = this.spy();
            this.searchResults.removeAndUpdateSelectionIfNeeded(this.searchResults.at(0)).done(onResolve);
            sinon.assert.calledOnce(onResolve);
            sinon.assert.calledWith(onResolve, 0);
        });

        test("It can get the issueKey based on the issue index", function() {
            equal(this.searchResults.getIssueKeyForIndex(0), "DEMO-1");
            equal(this.searchResults.getIssueKeyForIndex(3), "DEMO-4");
            equal(this.searchResults.getIssueKeyForIndex(999), undefined);
        });

        test("When jumping to a specified page, a 'before:loadpage' event is triggered", function() {
            this.spy(this.searchResults, "trigger");

            this.searchResults.jumpToPage(0);
            this.server.respond();

            sinon.assert.calledWith(this.searchResults.trigger, "before:loadpage");
        });

        test("When jumping to a specific issue, a 'before:loadpage' event is triggered", function() {
            this.spy(this.searchResults, "trigger");

            this.searchResults.jumpToPageForIssue(1);
            this.server.respond();

            sinon.assert.calledWith(this.searchResults.trigger, "before:loadpage");
        });

        test("When selecting the next issue, it triggers a 'before:loadpage' event if the next issue is on the next page", function() {
            this.goToLastIssueOfFirstPage();
            this.spy(this.searchResults, "trigger");

            this.searchResults.selectNext();
            this.server.respond();

            sinon.assert.calledWith(this.searchResults.trigger, "before:loadpage");
        });

        test("When selecting the previous issue, it triggers a 'before:loadpage' event if the previous isssue is on the previous page", function() {
            this.goToFirstIssueOfSecondPage();
            this.spy(this.searchResults, "trigger");

            this.searchResults.selectPrev();
            this.server.respond();

            sinon.assert.calledWith(this.searchResults.trigger, "before:loadpage");
        });

        test("When jumping to a specified page and there is an error, an 'error:loadpage' event is triggered", function() {
            this.serverAlwaysReturnsError();
            var trigger = this.spy(this.searchResults, "trigger");
            this.searchResults.on("error:loadpage", trigger);

            this.searchResults.jumpToPage(0);
            this.server.respond();

            sinon.assert.called(trigger);
        });

        test("It returns the number of issues in the database", function() {
            equal(this.searchResults.getTotalIssuesInDb(), 5);
        });

        test("It returns the number of issues in the search", function() {
            equal(this.searchResults.getTotalIssuesInSearch(), 4);
        });

        /*eslint-disable no-console */
        test("It shows a deprecation message when using getTotalIssues()", function() {
            this.spy(console, "warn");

            this.searchResults.getTotalIssues();

            ok(console.warn.firstCall.args[0].match(/^DEPRECATED/));
        });
        /*eslint-enable no-console */

        test("When calling getPage and there is preloaded data, it does not make a REST call", function () {
            var spy = this.sandbox.spy(PageableCollection.prototype, "fetch");

            this.searchResults.setPreloadedData({ids: [1, 2], response: FakeServer.createStableSearchResponse({id: [1, 2]}, ISSUES)});

            this.searchResults.getPage(0, {}).done(_.bind(function () {
                equal(this.searchResults.at(0).get('key'), 'DEMO-1');
                equal(this.searchResults.at(1).get('key'), 'DEMO-2');
            }, this));

            sinon.assert.callCount(spy, 0);
        });

        test("When calling getPage and consuming preloaded data, it is deleted and not used again", function () {
            var spy = this.sandbox.spy(PageableCollection.prototype, "fetch");

            this.searchResults.setPreloadedData({ids: [1, 2], response: FakeServer.createStableSearchResponse({id: [1, 2]}, ISSUES)});

            this.searchResults.getPage(0, {}).done(_.bind(function () {
                equal(this.searchResults.at(0).get('key'), 'DEMO-1');
                equal(this.searchResults.at(1).get('key'), 'DEMO-2');
            }, this));

            sinon.assert.callCount(spy, 0);

            this.searchResults.getPage(0, {}).done(_.bind(function () {
                equal(this.searchResults.at(0).get('key'), 'DEMO-1');
                equal(this.searchResults.at(1).get('key'), 'DEMO-2');
            }, this));

            sinon.assert.callCount(spy, 1);
        });

        asyncTest("It gets the next issue when that issue is on the current page", function() {
            expect(1);
            this.searchResults.jumpToPage(0); //Selects DEMO-1, the first issue in the page
            this.server.respond();

            this.searchResults.getNextIssue().done(function(issue) {
                QUnit.start();
                equal(issue.get("key"), "DEMO-2");
            });
        });

        asyncTest("It gets the next issue when that issue is in a different page", function() {
            expect(1);
            this.searchResults.jumpToPage(0);
            this.server.respond();
            this.searchResults.select("DEMO-2");  //Selects DEMO-2, the last issue in the page

            this.searchResults.getNextIssue().done(function(issue) {
                QUnit.start();
                equal(issue.get("key"), "DEMO-3");
            });

            // Responds to the request for loading the second page of results
            this.server.respond();
        });

        asyncTest("It gets the next issue when that issue is on the current page", function() {
            expect(1);
            this.searchResults.jumpToPage(0); //Selects DEMO-1, the first issue in the page
            this.server.respond();

            this.searchResults.getNextIssue().done(function(issue) {
                QUnit.start();
                equal(issue.get("key"), "DEMO-2");
            });
        });

        asyncTest("It rejects the promise then the selected issue doesn't have a next issue", function() {
            expect(1);
            this.searchResults.jumpToPage(1);
            this.server.respond();
            this.searchResults.select("DEMO-4"); //Selects DEMO-4, the last issue in the second page

            // If the promise is resolved, this test fails.
            this.searchResults.getNextIssue()
                .done(function() {
                    QUnit.start();
                    ok(false, "The promise is resolved");
                })
                .fail(function() {
                    QUnit.start();
                    ok(true, "The promise is rejected");
                });
        });

    });

});
