define("jira/components/query/jql/jqlqueryview", ["require"], function(require) {
    "use strict";

    var _ = require("jira/components/libs/underscore");
    var jQuery = require("jquery");
    var Brace = require("jira/components/libs/brace");
    var IssueNavQueryJql = require("jira/components/query/templates/issuenavqueryjql");
    var JQLAutoCompleteView = require("jira/components/query/jql/jqlautocompleteview");
    var Meta = require("jira/util/data/meta");
    var Tipsy = require("jira/issues/tipsy");

    /**
     * Renders the JQL textarea.
     */
    return Brace.View.extend({
        template: IssueNavQueryJql.jqlQueryView,

        namedEvents: ["verticalResize", "searchRequested", "searchChanged"],

        events: {
            "expandedOnInput": "_handleExpandOnInput",
            "input": function() {
                this.trigger("searchChanged", {
                    newJql: this.readRawJql()
                });
            }
        },

        initialize: function(options) {
            this.queryStateModel = options.queryStateModel;
            this.queryStateModel.on("change:jql", this.setQuery, this);
            jQuery(document).bind('issueNavWidthChanged', _.bind(this._resizeHeight, this));

            if (JQLAutoCompleteView) {
                this.JQLAutoCompleteView = new JQLAutoCompleteView({model: this.queryStateModel});
                this.JQLAutoCompleteView.onJqlValid(function(jql) {
                    if (this.queryStateModel.getAdvancedAutoUpdate()) {
                        this.triggerSearchRequested(jql);
                    }
                }, this);
                this.JQLAutoCompleteView.onSearchRequested(this.triggerSearchRequested, this);
            }
        },

        render: function() {

            this.$el.html(this.template({
                helpUrl: Meta.get('advanced-search-help-url'),
                helpTitle: Meta.get('advanced-search-help-title'),
                hasSearchButton: this.queryStateModel.hasSearchButton()
            }));

            this.$el.addClass("loading");

            if (this.JQLAutoCompleteView) {
                this.JQLAutoCompleteView.setElement(this.$el.find(".advanced-search"));
                this.JQLAutoCompleteView.getJqlAutoCompleteData();
            }

            new Tipsy({
                el: this.$el.find(".search-button"),
                tipsy: {
                    trigger: "hover",
                    delayIn: 300
                }
            });

            this.setQuery();
            return this;
        },

        /**
         * Gets the JQL from the input field.
         *
         * The difference with readJQL() is that this method has no
         * side effects, it just returns the JQL as-is.
         *
         * @returns {string} The JQL, or empyt string if the JQL is empty
         */
        readRawJql: function() {
            var $inputField = this._getInputField();
            return $inputField.val() || "";
        },

        /**
         * Sanizites the JQL from the input field and returns it.
         *
         * Note: this function have a few side effects:
         *      * It trims the JQL
         *      * It changes the field to the trimmed JQL
         *      * It changes the height of the field
         *
         * @returns {string} the JQL
         */
        readJql: function() {
            var jql = this.readRawJql();
            var $inputField = this._getInputField();

            // Prettify input field with trimmed JQL
            var trimmedJql = jQuery.trim(jql);
            if (jql !== trimmedJql) {
                $inputField.val(trimmedJql);
            }
            this._resizeHeight();
            return trimmedJql;
        },

        _handleExpandOnInput: function() {
            this.triggerVerticalResize();
        },

        _resizeHeight: function() {
            var $input = this._getInputField();
            // Need to set the height of the input to 0 for expandOnInput to reliably expand.
            // However, expandOnInput doesn't change the height of empty inputs, so need to handle those a little differently.
            if ($input.val()) {
                $input.height(0).expandOnInput();
            } else {
                $input.expandOnInput().height(0).trigger('refreshInputHeight');
            }

        },

        setQuery: function() {
            this.$el.removeClass("loading"); // Just in case we have come here from basic, loading class might be persisted. So remove.
            this._getInputField().val(this.queryStateModel.getJql());
            this._getInputField().trigger("updateParseIndicator");
            this._resizeHeight();
        },

        focus: function() {
            this._getInputField().focus();
        },

        _getInputField: function() {
            return this.$el ? this.$el.find("textarea") : jQuery();
        }
    });
});
