define("jira/components/query/basic/searchergrouplistdialogview", ["require"], function(require) {
    "use strict";

    var _ = require("jira/components/libs/underscore");
    var jQuery = require("jquery");
    var Brace = require("jira/components/libs/brace");
    var IssueNavQueryBasic = require("jira/components/query/templates/issuenavquerybasic");
    var SearcherDialog = require("jira/components/query/basic/searcherdialog");
    var CheckboxMultiSelectSuggestHandler = require("jira/ajs/select/suggestions/checkbox-multi-select-suggest-handler");
    var CheckboxMultiSelect = require("jira/ajs/select/checkbox-multi-select");
    var SuggestHelper = require("jira/ajs/select/suggestions/suggest-helper");
    var GroupDescriptor = require("jira/ajs/list/group-descriptor");
    var Tipsy = require("jira/issues/tipsy");
    var AJSHelper = require("jira/components/utils/ajshelper");

    var $moreCriteriaFooter;

    /**
     * List of searchers that can be added to a search
     */
    var module = Brace.View.extend({

        template: IssueNavQueryBasic.searcherDropdownContent,

        /**
         * searcherSelected(id): a searcher has been selected. id is the id of the searcher
         * hideRequested: dialog close has been requested
         */
        namedEvents: ["searcherSelected", "hideRequested"],

        events: {
            "keydown": "_keyPressed"
        },

        initialize: function(options) {
            this.searcherCollection = options.searcherCollection;
            this.$el.scrollLock('.aui-list-scroll');
        },

        render: function() {

            var descriptors = this.searcherCollection.getAddMenuGroupDescriptors();
            var tooManySearchers = descriptors.length && descriptors[0].properties.items.length > module.CRITERIA_DISPLAY_LIMIT;

            var select = jQuery(AJSHelper.Templates.queryableSelect({
                descriptors: descriptors,
                id: "criteria"
            }));

            // Even though it works, weird stuff happens if you call .html(select) since select is a jQuery object.
            this.$el.empty().append(select);

            var options = {
                element: select,
                suggestionsHandler: module.SuggestHandler,
                hideFooterButtons: true
            };

            var searchersHiddenMessage = AJS.I18n.getText("issues.components.query.searcher.hidden.global");
            var searchersHiddenDetails = AJS.I18n.getText("issues.components.query.searcher.hidden.global.desc");

            // Performance optimisation. When appending more than 100 custom fields performance suffers dramatically.
            if (tooManySearchers) {
                options.maxInlineResultsDisplayed = module.CRITERIA_DISPLAY_LIMIT;
                searchersHiddenMessage = AJS.I18n.getText("issues.components.query.searcher.hidden.global.too.many.searchers");
                searchersHiddenDetails = AJS.I18n.getText("issues.components.query.searcher.hidden.global.desc.too.many.searchers");
            }

            new CheckboxMultiSelect(options);

            $moreCriteriaFooter = jQuery("<div class='more-criteria-footer' />").html(searchersHiddenMessage);
            this.$el.append($moreCriteriaFooter);
            new Tipsy({
                el: $moreCriteriaFooter,
                tipsy: {
                    title: function() {
                        return searchersHiddenDetails;
                    },
                    className: "tipsy-front"
                }
            });

            // this.$el is an element owned by AJS.InlineLayer and is detached from the dom each time we switch
            // from basic to advanced. Thus, we need to rebind when we render rather than use the backbone way of
            // binding.
            this.$el.unbind("selected").bind("selected", _.bind(this._searcherSelected, this));
            this.$el.unbind("unselect").bind("unselect", _.bind(this._searcherUnselected, this));

            return this.$el;
        },

        _searcherUnselected: function(e, descriptor) {
            this.searcherCollection.clearClause(descriptor.properties.value);
        },

        _searcherSelected: function(e, descriptor) {
            var searcher = this.searcherCollection.getSearcher(descriptor.properties.value);
            searcher.select();
            SearcherDialog.instance.hide();
            SearcherDialog.instance.show(searcher);
        },

        _keyPressed: function(event) {
            if (event.keyCode === jQuery.ui.keyCode.TAB) {
                var tabbableElements = jQuery(":tabbable", this.$el);

                var noTabbableElements = (tabbableElements.length === 0);
                var shiftTabbingOnFirst = (event.shiftKey && (document.activeElement === tabbableElements.first()[0]));
                var tabbingOnLast = (!event.shiftKey && (document.activeElement === tabbableElements.last()[0]));

                if (noTabbableElements || shiftTabbingOnFirst || tabbingOnLast) {
                    this.triggerHideRequested(AJSHelper.HIDE_REASON.tabbedOut);
                    event.preventDefault();
                }
            }
        }
    }, {
        CRITERIA_DISPLAY_LIMIT: 100
    });

    module.SuggestHandler = CheckboxMultiSelectSuggestHandler.extend({
        formatSuggestions: function(groups, query) {
            var numberHidden = 0;
            var selectedItems = SuggestHelper.removeDuplicates(this.model.getDisplayableSelectedDescriptors());

            // Prepend a group containing all selected items.
            groups.splice(0, 0, new GroupDescriptor({
                actionBarHtml: selectedItems.length > 1 ? this.createClearAll() : null,
                items: selectedItems,
                styleClass: "selected-group"
            }));

            _.each(groups, function(group) {
                if (query.length === 0) {
                    var items = _.filter(group.items(), function(item, index) {
                        var meta = item.meta();
                        return meta && meta.isShown && index < module.CRITERIA_DISPLAY_LIMIT;
                    });

                    numberHidden += group.items().length - items.length;
                    group.items(items);
                } else {
                    _.each(group.items(), function(item) {
                        var meta = item.meta();
                        item.disabled(!(meta && meta.isShown));
                    });
                }
            });

            _.defer(function() {
                if ($moreCriteriaFooter) {
                    if (numberHidden) {
                        $moreCriteriaFooter.show().find(".hidden-no").text(numberHidden);
                    } else {
                        $moreCriteriaFooter.hide();
                    }
                }
            });

            return groups;
        }
    });

    return module;
});
