define("jira/components/query/basic/searcherdialog", ["require"], function(require) {
    "use strict";

    var skate = require('jira/skate');
    var SearcherEditDialogView = require("jira/components/query/basic/searchereditdialogview");
    var _ = require("jira/components/libs/underscore");
    var InlineLayer = require('jira/ajs/layer/inline-layer');
    var AJSHelper = require("jira/components/utils/ajshelper");
    var jQuery = require("jquery");
    var trace = require('jira/components/utils/trace');

    var HideReasons = AJSHelper.HIDE_REASON;

    /**
     * A singleton that reuses the same InlineLayer for many different criteria selectors.
     */
    var module = function() {

        return {
            initialize: function(options) {
                if (!this.instance) {
                    this.instance = initSearcherDialog(options);
                }
            }
        };

        function initSearcherDialog(options) {
            HideReasons.switchLozenge = "switchLozenge";

            var reasonsToFocusCriteria = [
                HideReasons.escPressed,
                HideReasons.toggle,
                HideReasons.cancelClicked,
                HideReasons.submit,
                HideReasons.tabbedOut
            ];

            function findLozenge(searcher) {
                return searcher ? jQuery(".criteria-selector[data-id='" + searcher.getId() + "']") : jQuery();
            }

            var promise;
            var _currentSearcher; // JIRA.Issues.SearcherModel
            var _currentView;

            var _dialog = new InlineLayer({
                width: "auto",
                alignment: AJSHelper.LEFT,
                offsetTarget: function() {
                    // Don't cache the lozenge under the current implementation since it can get reblatted
                    return findLozenge(_currentSearcher);
                },
                // Uses AJS.DeferredContentRetriever.
                content: function() {
                    _currentView = new SearcherEditDialogView({
                        model: _currentSearcher,
                        queryStateModel: options.queryStateModel
                    });
                    _currentView.onHideRequested(function(reason) {
                        _dialog.hide(reason);
                    });
                    return _currentView.renderDeferred();
                }
            });

            _dialog.bind(InlineLayer.EVENTS.show, function(e, $layer) {
                skate.init($layer[0]);
                var input = $layer.find(":input:not(submit):visible:first");
                input.focus();

                // List.js also resets the scrollTop but because the dialog is still hidden at that point, the browser won't actually do any scrolling.
                // @see JRADEV-15097
                $layer.find(".aui-list-scroll").scrollTop(0);
            });

            _dialog.bind(InlineLayer.EVENTS.beforeHide, function(e, layer, reason, id, originalTarget) {
                if (reason === HideReasons.clickOutside && originalTarget && jQuery(originalTarget).closest(".calendar").length) {
                    e.preventDefault();
                }
            });

            _dialog.bind(InlineLayer.EVENTS.hide, function(e, layer, reason) {
                var _searcher = _currentSearcher;

                if (_.contains(reasonsToFocusCriteria, reason)) {
                    findLozenge(_searcher).focus();
                }

                function doSearch() {
                    // A searcher has been submitted. Create a clause and add it to the clause collection
                    promise = _currentSearcher.createOrUpdateClauseWithQueryString(reason === HideReasons.submit);
                    promise.done(function() {
                        // Check for an "error" class in the searcher's editHtml.
                        // If so, leave the dialog open and rerender to the the updated editHtml.
                        // Otherwise close the dialog.
                        if (_searcher.hasErrorInEditHtml()) {
                            // Prevent displaying the error dialog if the mode has switched to advanced as a
                            // result of the click outside (i.e. on the Switch to Advanced link)
                            if (options.queryStateModel.getSearchMode() === "basic") {
                                module.instance.show(_searcher);
                            }
                        }

                        if (InlineLayer.current) {
                            // page layout might have changed as a result of updating a lozenge content (need to make sure dialog is in correct position)
                            InlineLayer.current.setPosition();
                        }
                    });

                    promise.always(function() {
                        promise = null;
                    });

                    _searcher.clearEditHtml();
                }

                if (reason === HideReasons.submit) {
                    // If this is a submit, always do de search
                    _currentView.applyFilter();
                    doSearch();
                } else if (reason !== HideReasons.cancelClicked && reason !== HideReasons.escPressed &&
                    reason !== HideReasons.tabbedOut) {
                    // If the dialog has not been cancelled, do the search only if the selector has not changed.
                    if (_currentView.applyFilter()) {
                        doSearch();
                    }
                } else {
                    trace("jira.search.searchers.hiddenWithoutUpdate");
                }

                _currentView.$el.remove();
                _currentView = null;
                _currentSearcher = null;
            });

            /**
             * JRADEV-15697 - Ensure all the dialog is closed when we switch search modes.
             */
            options.queryStateModel.on("change:searchMode", function() {
                _dialog.hide();
            });

            return {
                /**
                 * @return {SearcherModel} the searcher for which the dialog was last shown.
                 */
                getCurrentSearcher: function() {
                    return _currentSearcher;
                },

                /**
                 * Hide and show dialog
                 */
                toggle: function(searcher) {
                    if (_currentSearcher != null) {
                        if (_currentSearcher !== searcher) {
                            _dialog.hide(HideReasons.switchLozenge);
                            this.show(searcher);
                        } else {
                            _dialog.hide(HideReasons.toggle);
                        }
                    } else {
                        this.show(searcher);
                    }
                },

                _show: function(searcher) {
                    _.defer(function() {
                        _currentSearcher = searcher;
                        _dialog.show();
                        _dialog.setPosition();
                    });
                },

                /**
                 * shows dialog with correct searcher
                 * @param {JIRA.Issues.SearcherModel} searcher
                 */
                show: function(searcher) {
                    var waitingToShow;
                    if (module.waitingToShow) {
                        waitingToShow = true;
                    }

                    var instance = this;
                    module.waitingToShow = function() {
                        instance._show(searcher);
                        module.waitingToShow = null;
                    };

                    if (!waitingToShow) {
                        // Defer showing until all searchers are ready.
                        searcher.searchersReady().done(function() {
                            if (promise) {
                                promise.done(function() {
                                    module.waitingToShow();
                                });
                            } else {
                                module.waitingToShow();
                            }
                        });
                    }
                },

                /**
                 * Bind a handler to be called when the dialog is shown.
                 *
                 * @param handler The function to call when the dialog is shown.
                 */
                onShow: function(handler) {
                    _dialog.bind(InlineLayer.EVENTS.show, handler);
                },

                /**
                 * Hide the current dialog. Drrrrr!
                 */
                hide: function() {
                    _dialog.hide();
                },

                /**
                 * Bind a handler to be called when the dialog is hidden.
                 *
                 * @param handler The function to call when the dialog is hidden.
                 */
                onHide: function(handler) {
                    _dialog.bind(InlineLayer.EVENTS.hide, handler);
                }
            };
        }
    }();

    return module;
});
