define("jira/components/query/queryview", ["require"], function(require) {
    "use strict";

    var Brace = require("jira/components/libs/brace");
    var IssueNavQuery = require("jira/components/query/templates/issuenavquery");
    var SwitcherCollection = require("jira/components/query/switchercollection");
    var SwitcherView = require("jira/components/query/switcherview");
    var QuerySwitcherViewModel = require("jira/components/query/queryswitcherviewmodel");
    var ComponentUtilTemplate = require("jira/components/query/templates/issues/componentutil");

    /**
     * This renders the container for either the basic query view or the advanced query view.
     *
     * @see JIRA.Issues.JqlQueryView
     * @see JIRA.Issues.BasicQueryView
     */
    return Brace.View.extend({

        namedEvents: ["verticalResize"],

        template: IssueNavQuery.queryView,

        events: {
            "click .search-button": "search",
            "submit": "preventDefault"
        },

        initialize: function(options) {
            this.queryStateModel = options.queryStateModel;

            var switcherCollection = new SwitcherCollection([
                {
                    id: this.queryStateModel.BASIC_SEARCH,
                    name: "Basic",
                    view: options.basicQueryModule.createView()
                }, {
                    id: this.queryStateModel.ADVANCED_SEARCH,
                    name: "Advanced",
                    view: options.jqlQueryModule.createView()
                }
            ]);

            // TODO: hack for search() below, to fix
            this.jqlQueryModule = options.jqlQueryModule;

            this.switcherViewModel = new QuerySwitcherViewModel({
                collection: switcherCollection
            }, {
                queryStateModel: this.queryStateModel
            });

            this.switcherView = new SwitcherView({
                template: IssueNavQuery.searchSwitcher,
                model: this.switcherViewModel,
                containerClass: ".search-container"
            }).onVerticalResize(this.triggerVerticalResize, this);

        },

        render: function() {
            this.$el.html(this.template(this.queryStateModel.toJSON()));
            this.switcherView.setElement(this.$el).render();
            return this;
        },

        preventDefault: function(e) {
            e.preventDefault();
        },

        /**
         * Clear the notifications area (errors and warnings).
         */
        clearNotifications: function() {
            this.$(".notifications").empty();
        },

        /**
         * Performs a search with a query defined by the value of the textarea.
         *
         * This calls search on the model, only if we don't have an active saved search,
         * or if the query was changed from the current active saved search.
         *
         */
        search: function() {
            // TODO: temp hack for jql query view rework
            if (this.queryStateModel.ADVANCED_SEARCH === this.switcherViewModel.getSelected().getId()) {
                this.jqlQueryModule.search();
            }
            else {
                this.getView().search();
            }
        },

        getView: function() {
            return this.switcherViewModel.getSelected().getView();
        },

        /**
         * Display one or more errors in the notification area.
         *
         * @param {Array} errors The error(s) to be displayed.
         */
        showErrors: function(errors) {
            this.$(".notifications").append(ComponentUtilTemplate.errorMessage({messages: errors}));
        },

        /**
         * Display one or more warnings in the notification area.
         *
         * @param {Array} warnings The warning(s) to be displayed.
         */
        showWarnings: function(warnings) {
            this.$(".notifications").append(ComponentUtilTemplate.warningMessage({messages: warnings}));
        },

        /**
         * Hide or show the view (excluding error messages).
         *
         * @param isVisible Whether the view should be visible.
         */
        setVisible: function(isVisible) {
            this.$(".search-container").toggleClass('hidden', !isVisible);
            this.switcherView.setVisible(isVisible);
        }
    });
});
